#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations

import pytest
import time
import yaml

from si_tests.utils import update_child_clusterrelease_actions, utils, waiters
from si_tests.utils import update_release_names
from si_tests import settings
from si_tests import logger
from dateutil import parser
from future.backports.datetime import timedelta

LOG = logger.logger

update_release_names = list(update_release_names.generate_update_release_names())


def check_new_estimated_value_is_bigger(update_actions, target_cr_version, step_names, prev_estimated_list):
    check_list = []
    for step_name in step_names:
        LOG.info(f"Check that new estimated value for step: {step_name} is bigger than before...")
        current_steps_estimated_list = update_actions.create_estimated_time_list_from_update_plan(target_cr_version)
        current_estimated = [x for x in current_steps_estimated_list if x["step_name"] == step_name][0]
        assert current_estimated, (f"No step with name: {step_name} in current_estimated"
                                   f" dict: {current_steps_estimated_list}")
        prev_estimated = [x for x in prev_estimated_list if x["step_name"] == step_name][0]
        assert prev_estimated, (f"No step with name: {step_name} in current_estimated"
                                f" dict: {prev_estimated_list}")
        check_list.append({'step_name': step_name, 'cur_est': current_estimated.get("estimated_time", ""),
                           'prev_est': prev_estimated.get("estimated_time", ""), 'checked': 'PASSED'})
    for step in check_list:
        current_time = parser.parse(step['cur_est']).time()
        prev_estimated_time = parser.parse(step['prev_est']).time()
        prev_est = timedelta(hours=prev_estimated_time.hour,
                             minutes=prev_estimated_time.minute,
                             seconds=prev_estimated_time.second)

        current = timedelta(hours=current_time.hour,
                            minutes=current_time.minute,
                            seconds=current_time.second)
        if prev_est >= current:
            step["checked"] = "FAILED"

    LOG.info(f"Check list result: \n{yaml.dump(check_list)}")

    failed_steps = [step for step in check_list if step["checked"] == 'FAILED']
    if failed_steps:
        LOG.info(f"Current estimated time is less than prev calculated "
                 f"estimated time for steps: \n{yaml.dump(failed_steps)}")
        return False
    return True


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.TARGET_CLUSTER)])
def test_refresh_values_because_of_node_scale(kaas_manager, check_sg, openstack_client, show_step, _):
    """Check that scale operations affects estimated values

    Scenario:
        1. Store estimated time before operations for all steps
        2. Add worker machine
        3. Add compute node to openstack
        4. Add node to Ceph
        5. Add disk to new ceph node
        6. Compare new estimates with old ones

    """

    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    machine_image_name = settings.KAAS_CHILD_CLUSTER_MACHINE_IMAGE_NAME.split(',')[0]
    machine_flavor_name = settings.KAAS_CHILD_CLUSTER_MACHINE_FLAVOR_NAME
    ceph_machine_flavor = "compact.nal"
    machine_az_name = settings.KAAS_CHILD_CLUSTER_AZ_NAME
    region = kaas_manager.get_mgmt_cluster().region_name

    # step 001 - Get namespace
    LOG.info("Namespace name - %s", namespace_name)
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    show_step(1)
    update_release_name = update_release_names[-1]
    LOG.info(f"Update release name: {update_release_name}")
    update_actions = update_child_clusterrelease_actions.UpdateChildClusterreleaseActions(cluster)
    # ClusterRelease spec.version is used to find the correct update plan object
    target_cr_version = kaas_manager.get_clusterrelease(
        update_release_name).data.get("spec", {}).get("version")
    steps_estimated_list_before_scale = update_actions.create_estimated_time_list_from_update_plan(
        target_cr_version)

    if settings.KAAS_OFFLINE_DEPLOYMENT:
        offline_security_group = [check_sg['name']]
    else:
        offline_security_group = None

    show_step(2)
    slave_node = cluster.create_os_machine(
        node_type="node",
        node_flavor=machine_flavor_name,
        node_image=machine_image_name,
        node_az=machine_az_name,
        region=region,
        node_sg=offline_security_group)

    show_step(3)
    openstack_label = [{"key": "openstack-compute-node", "value": "enabled"},
                       {"key": "openvswitch", "value": "enabled"}]
    slave_node_os = cluster.create_os_machine(
        node_type="node",
        node_flavor=machine_flavor_name,
        node_image=machine_image_name,
        node_az=machine_az_name,
        region=region,
        node_sg=offline_security_group,
        node_labels=openstack_label)

    show_step(4)
    ceph_node = cluster.create_os_machine(
        node_type="node",
        node_flavor=ceph_machine_flavor,
        node_image=machine_image_name,
        node_az=machine_az_name,
        region=region,
        node_sg=offline_security_group)

    new_machines = []
    for m_node in [ceph_node, slave_node, slave_node_os]:

        # Waiting for master node become Ready
        cluster.check.wait_machine_status_by_name(machine_name=m_node.name,
                                                  timeout=3600,
                                                  expected_status='Ready')
        new_machines.append(cluster.get_machine(name=m_node.name))

    # Waiting for machines are Ready
    cluster.check.check_machines_status(timeout=1800)
    cluster.check.check_cluster_nodes()
    cluster.check.check_k8s_nodes()

    show_step(5)
    node = cluster.get_machine(ceph_node.name)
    os_node_name = node.get_k8s_node_name()
    os_node_id = openstack_client.get_server_by_name(
        os_node_name).id
    # check disks before
    existed_disks = node.run_cmd(
        'lsblk -o NAME --json').stdout_yaml
    disks_names_before = [
        d['name'] for d in existed_disks['blockdevices']]
    # add volume
    volume = openstack_client.cinder.volumes.create(
        size=50, name="ceph-volume-{}".format(os_node_name))
    volume_id = volume.id
    waiters.wait(
        lambda: openstack_client.cinder.volumes.get(
            volume_id)._info.get('status') == 'available')
    LOG.info("Attaching volume {} to node {}".format(
        volume_id, os_node_id))
    openstack_client.nova.volumes.create_server_volume(
        server_id=os_node_id, volume_id=volume_id)
    # Need to wait some time for disk appears in devices
    time.sleep(20)
    LOG.info("Checking for new disk")
    disks_after_attach = node.run_cmd(
        'lsblk -o NAME --json').stdout_yaml
    disks_names_after_attach = [
        d['name'] for d in disks_after_attach['blockdevices']]
    new_disk = [d for d in disks_names_after_attach if
                d not in disks_names_before]
    assert len(new_disk) > 0, ("No new disks were added")
    # Add node to Ceph
    ceph_tools_pod = cluster.get_ceph_tool_pod()
    cmd_status = ['/bin/sh', '-c', 'ceph -s -f json']
    ceph_health_timeout = 2400
    ceph_crd = cluster.get_miracephcluster(name="cephcluster")
    nodes = ceph_crd.read().spec['nodes']
    ceph_config = nodes[0].copy()
    ceph_config['name'] = node.get_k8s_node_name()
    ceph_config['roles'] = ['mgr']
    nodes.append(ceph_config)
    add_ceph_node = {"spec": {"nodes": nodes}}
    LOG.info("MiraCeph CRD will be patched with next data:{}{}".format(
        "\n", yaml.dump(add_ceph_node)))
    # Get osds num before patch and calculate num after
    disk_count = len(ceph_config['devices'])
    num_osds_before_patch = int(yaml.safe_load(ceph_tools_pod.exec(
        cmd_status)).get('osdmap').get('num_osds'))
    num_osds_after_patch = num_osds_before_patch + disk_count
    LOG.info("num_osds_before_patch: {}".format(num_osds_before_patch))
    LOG.info("num_osds_after_patch: {}".format(num_osds_after_patch))
    LOG.info("Patching miraceph CRD")
    cluster.patch_ceph_data(data=add_ceph_node, crd=ceph_crd)
    time.sleep(5)
    # Wait for num osds is inceased depending on disks count
    LOG.info("Waiting for OSDs number is increased in ceph cluster")
    waiters.wait(
        lambda: yaml.safe_load(ceph_tools_pod.exec(cmd_status)).get(
            'osdmap').get('num_osds') == num_osds_after_patch,
        timeout=ceph_health_timeout, interval=30,
        timeout_msg="OSDs number doesn't match expected number after {} sec. "
                    "Current osd number is {}. But shoud be {}".format(
            ceph_health_timeout,
            yaml.safe_load(ceph_tools_pod.exec(cmd_status)).get(
                'osdmap').get(
                'num_osds'), num_osds_after_patch))

    show_step(6)
    waiters.wait(lambda: check_new_estimated_value_is_bigger(update_actions=update_actions,
                                                             target_cr_version=target_cr_version,
                                                             step_names=["openstack",
                                                                         "k8s-workers-cluster-mos-default",
                                                                         "ceph"],
                                                             prev_estimated_list=steps_estimated_list_before_scale),
                 interval=15,
                 timeout=600)

    # Collecting artifacts
    cluster.store_k8s_artifacts()
    cluster.check.check_cluster_readiness()
    cluster.provider_resources.save_artifact()


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.TARGET_CLUSTER)])
def test_refresh_values_because_of_ceph_size(kaas_manager, show_step, _):
    """Check that Ceph size affects estimated values

    Scenario:
        1. Store estimated time before operations for all steps
        2. Create new PVC in Ceph
        3. Write 13 Gb file into new Ceph PV
        4. Compare new estimate with previous one

    """

    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE

    # step 001 - Get namespace
    LOG.info("Namespace name - %s", namespace_name)
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)
    rnd = utils.gen_random_string(6)
    namespace = f'test-ns-{rnd}'
    cluster.k8sclient.namespaces.create(name=namespace, body={'metadata': {'name': namespace}})

    show_step(1)
    update_release_name = update_release_names[-1]
    LOG.info(f"Update release name: {update_release_name}")
    update_actions = update_child_clusterrelease_actions.UpdateChildClusterreleaseActions(cluster)
    # ClusterRelease spec.version is used to find the correct update plan object
    target_cr_version = kaas_manager.get_clusterrelease(
        update_release_name).data.get("spec", {}).get("version")
    steps_estimated_list_before_scale = update_actions.create_estimated_time_list_from_update_plan(
        target_cr_version)

    show_step(2)
    pvc_size = 128
    pvc_name = "test-pv-claim-updateplan-functional"
    storage_classes = cluster.k8sclient.api_storage.list_storage_class().to_dict()['items']
    storage_class_name = [storageclass for storageclass in storage_classes
                          if storageclass['provisioner'] == 'rook-ceph.rbd.csi.ceph.com'
                          and storageclass['metadata']['annotations']
                          ['storageclass.kubernetes.io/is-default-class'] == 'true'][0]['metadata']['name']

    test_pvc = cluster.check.create_pvc(ns=namespace, pvc_name=pvc_name,
                                        pvc_size=pvc_size, storage_class=storage_class_name)
    show_step(3)
    pod = cluster.check.create_pod(pvc_name=test_pvc.name, name="test-pod-test", ns=namespace)
    LOG.info("Pod was created successfully")
    cmd = ['/bin/sh', '-c', "dd if=/dev/urandom of=/data/test/test.img bs=100M count=400 &"]
    pod.exec(cmd)
    LOG.info("File was created successfully")
    cmd_length = ['/bin/sh', '-c', "wc -c < /data/test/test.img"]
    # Expected file size 13G in bytes format
    expected_file_size = 13421772398

    def _file_filled(cmd_length):
        file_length = pod.exec(cmd_length)
        LOG.info(f"File length: {file_length}")
        assert int(file_length) > expected_file_size

    waiters.wait_pass(lambda: _file_filled(cmd_length), interval=15, timeout=900)
    show_step(4)
    waiters.wait(lambda: check_new_estimated_value_is_bigger(update_actions=update_actions,
                                                             target_cr_version=target_cr_version,
                                                             step_names=["ceph"],
                                                             prev_estimated_list=steps_estimated_list_before_scale),
                 interval=15,
                 timeout=300)
