import copy

from si_tests import logger
from si_tests import settings
from si_tests.managers.kaas_manager import Machine, Cluster, Manager
from si_tests.utils import waiters
from si_tests.utils.utils import Provider
from si_tests.managers.machine_deletion_policy_manager import check_machine_graceful_delete


LOG = logger.logger


def create_master_node(kaas_manager: Manager, cluster: Cluster):
    """
    Get data from previous deleted master machine and create a new one
    Args:
        cluster: current cluster object

    Returns: None

    """

    provider = Provider.equinixmetalv2
    facility = cluster.spec['providerSpec']['value']['facility']
    region = kaas_manager.get_mgmt_cluster().region_name

    new_machine = cluster.create_equinix_machine(
        node_type="master",
        os=settings.KAAS_EQUINIX_OS,
        billing_cycle=settings.KAAS_EQUINIX_BILLING_CYCLE,
        machine_type=settings.KAAS_EQUINIX_MACHINE_TYPE,
        region=region,
        provider_name=provider.provider_name,
        facility=facility)

    LOG.info(f'Check newly created machine {new_machine.name}')
    # Waiting for master node become Ready
    cluster.check.wait_machine_status_by_name(machine_name=new_machine.name,
                                              expected_status='Ready',
                                              timeout=5800, interval=60)

    # Waiting for machines are Ready
    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()
    cluster.check.check_k8s_nodes()
    if cluster.is_child:
        LOG.info("Wait Ceph HEALTH_OK status")
        try:
            health_info = cluster.check.get_ceph_health_detail()
            assert health_info['status'] == "HEALTH_OK", f'Health is not OK. Will not proceed. ' \
                                                         f'Current ceph health status: {health_info}'
        except AssertionError:
            cluster.check.wait_ceph_health_status(timeout=3600, interval=30)

    # Check/wait for correct docker service replicas in cluster
    ucp_worker_agent_name = cluster.check.get_ucp_worker_agent_name()
    cluster.check.check_actual_expected_docker_services(
        changed_after_upd={'ucp-worker-agent-x': ucp_worker_agent_name})
    cluster.check.check_k8s_pods()
    cluster.check.check_actual_expected_pods(timeout=3200)
    cluster.check.check_cluster_readiness()
    cluster.check.check_deploy_stage_success(skipped_stages_names='IAM objects created')


def move_ceph_monitor(kaas_manager: Manager, cluster: Cluster, machine_from: Machine, machine_to: Machine):
    """
    Move Ceph monitor role from master to worker machine
    Args:
        kaas_manager: KaaS manager object
        cluster: current cluster object
        machine_from: machine object from which to transfer
        machine_to: machine object to pass to

    Returns:

    """
    node_from_name = machine_from.get_k8s_node_name()
    node_to_name = machine_to.get_k8s_node_name()
    LOG.info(f" Move Ceph monitor \n"
             f" from machine {machine_from.name} k8s node {node_from_name}\n"
             f" to machine {machine_to.name} k8s node {node_to_name}")

    is_ceph_manual_configured = cluster.data['spec']['providerSpec']['value']['ceph'].get('manualConfiguration', False)
    if not is_ceph_manual_configured:
        LOG.info("Allow manual Ceph configuration")
        keys = {
            "spec": {
                "providerSpec": {
                    "value": {
                        "ceph": {
                            "manualConfiguration": True
                        }
                    }
                }
            }
        }
        cluster.patch(keys)
    else:
        LOG.info("Ceph already allow manual configuration")

    ceph_cluster = kaas_manager.api.kaas_cephclusters.list(
        namespace=settings.TARGET_NAMESPACE, name_prefix=settings.TARGET_CLUSTER)
    roles_migrated_before = \
        not ceph_cluster[0].data['spec']['cephClusterSpec']['nodes'].get(machine_from.name).get('roles', [])

    def _move_roles():
        ceph_cluster = kaas_manager.api.kaas_cephclusters.list(
            namespace=settings.TARGET_NAMESPACE, name_prefix=settings.TARGET_CLUSTER)
        cluster_ceph_data = copy.deepcopy(ceph_cluster[0].data)
        machine_from_roles = \
            cluster_ceph_data['spec']['cephClusterSpec']['nodes'].get(machine_from.name).get('roles', [])
        if not machine_from_roles:
            LOG.info("Ceph mon roles already migrated")
            return True
        else:
            LOG.info("Update Ceph cluster for migrate mon roles")
            cluster_ceph_data['spec']['cephClusterSpec']['nodes'].get(machine_from.name)['roles'] = None
            cluster_ceph_data['spec']['cephClusterSpec']['nodes'].get(machine_to.name)['roles'] = ['mon', 'mgr']
            patch_data = {'spec': cluster_ceph_data['spec']}
            ceph_cluster[0].patch(body=patch_data)
            return False

    LOG.info("Move Ceph mon node roles")
    waiters.wait(_move_roles, interval=30, timeout=300)

    def _get_miraceph_node_roles():
        nonlocal machine_from, machine_to
        miraceph = cluster.k8sclient.miracephs.get(name='rook-ceph', namespace='ceph-lcm-mirantis').data['spec']
        m_from = next(o for o in miraceph['nodes'] if o['name'] == node_from_name)
        m_to = next(o for o in miraceph['nodes'] if o['name'] == node_to_name)
        if not m_from.get('roles', []) and set(m_to.get('roles', [])) == set(['mon', 'mgr']):
            return True
        else:
            LOG.info(f"Expected roles for node:\n"
                     f" {m_from['name']} should be empty. Actual is: {m_from.get('roles', [])}\n"
                     f" {m_to['name']} should be [mon, mgr]. Actual is: {m_to.get('roles', [])}")
            return False

    LOG.info("Wait until miraceph nodes changed")
    waiters.wait(_get_miraceph_node_roles, interval=10, timeout=600)

    actual_mon_pods = [
        pod for pod in cluster.k8sclient.pods.list_starts_with(
            pattern='rook-ceph-mon-', namespace='rook-ceph')]
    mon_pods_to_delete = [p for p in actual_mon_pods if p.data['spec']['node_name'] == node_from_name]

    for pod in mon_pods_to_delete:
        LOG.info(f"Delete pod {pod.name} from {node_from_name} node")
        pod.delete()

    if roles_migrated_before:
        LOG.info("Ceph cluster should not be changed")
    else:
        LOG.info("Wait approximately 20 minutes until rook-ceph-operator performs a failover of the Pending mon pod.")
        # Note - usually takes 2 iterations of 10 minutes each.
        LOG.info("Wait Ceph HEALTH_WARN status")
        try:
            health_info = cluster.check.get_ceph_health_detail()
            assert health_info['status'] == "HEALTH_WARN", f'Health is not OK. Will not proceed. ' \
                                                           f'Current ceph health status: {health_info}'
        except AssertionError:
            cluster.check.wait_ceph_health_status(expected_status='HEALTH_WARN', timeout=600, interval=30)
    LOG.info("Wait Ceph HEALTH_OK status")
    try:
        health_info = cluster.check.get_ceph_health_detail()
        assert health_info['status'] == "HEALTH_OK", f'Health is not OK. Will not proceed. ' \
                                                     f'Current ceph health status: {health_info}'
    except AssertionError:
        cluster.check.wait_ceph_health_status(timeout=3600, interval=30)
    cluster.check.check_cluster_readiness()


def test_replace_master_node(kaas_manager: Manager, show_step):
    """Replace EquinixMetal v2 control plane node.

    Scenario:
        1. Find master node where VIP hosted
        2. Move Ceph monitor from master to worker node if it is child
        3. Delete master node
        4. Create new master node
    """

    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE

    ns = kaas_manager.get_namespace(namespace_name)
    LOG.info("Namespace name - %s", namespace_name)

    cluster = ns.get_cluster(cluster_name)
    LOG.info("Cluster name - %s", cluster_name)

    workers = cluster.get_machines(machine_type='worker')
    show_step(1)
    master = cluster.get_keepalive_master_machine()

    show_step(2)
    if cluster.is_child:
        move_ceph_monitor(kaas_manager, cluster, master, workers[0])
    else:
        LOG.info("No need to move Ceph monitor")
    show_step(3)
    if cluster.machine_deletion_policy_enabled():
        check_machine_graceful_delete(cluster, master, wait_deletion_timeout=3600)
    else:
        master.delete()
        cluster.check.check_deleted_node(master.name)
    show_step(4)
    create_master_node(kaas_manager, cluster)
