import pytest
import time

from si_tests import logger
from si_tests.deployments.utils import kubectl_utils
from si_tests.utils import waiters

LOG = logger.logger

# Get Prometheus Service IP
kubectl = kubectl_utils.Kubectl()

# Ironic test server name
ironic_test_server = 'testserver-fake-ironic'


def _get_counters_values(counters, counter_name):
    values = {}
    for counter, value in counters.items():
        if counter_name in counter:
            values[counter[len(counter_name):-1]] = value

    return values


def counter_is_true(openstack_client_manager, service_ip, hostname, counter_type, status, cvalue):

    counters = openstack_client_manager.get_prometheus_counters(service_ip)
    counter_group = _get_counters_values(counters, counter_type)

    if status in ('disabled', 'enabled', 'up', 'down'):
        for key, value in counter_group.items():
            if status in key and hostname in key and value == cvalue:
                LOG.info("Counter %s is changed to value %s as expected" % (key, value))
                return True

        return False


def get_osdpl_prometheus_counters(openstack_client_manager, sub_counters):
    svc_name = f"{openstack_client_manager.oc_name}-exporter"
    svc_ip = kubectl.get('svc', f'{svc_name} -o yaml', 'osh-system').result_yaml['spec']['clusterIP']
    counters = openstack_client_manager.get_prometheus_counters(svc_ip)
    return _get_counters_values(counters, sub_counters)


def check_ironic_nodes_counters(openstack_client_manager, exp_ctr):
    osdpl_ironic_nodes_available = get_osdpl_prometheus_counters(openstack_client_manager,
                                                                 'osdpl_ironic_nodes_available')
    for key, value in osdpl_ironic_nodes_available.items():
        if ironic_test_server in key and value == exp_ctr:
            return True

    return False


def test_nova_service_status_metric(openstack_client_manager):
    """Verify osdpl_nova_service_status metric
    Parameters required for test execution:
        - KUBECONFIG

    Scenario:
        1. Verify initial Nova service status counter for compute node
        2. Disable compute service for one of the Nova compute nodes
        3. Check that osdpl_nova_service_status counter is changed for this node
        4. Enable compute service for the Nova compute node
        5. Check that osdpl_nova_service_status is changed for this node
        6. Cleanup

    """

    _test_nova_service_metric(openstack_client_manager, metric_type='osdpl_nova_service_status')


def test_nova_service_state_metric(openstack_client_manager):
    """Verify osdpl_nova_service_state metric
    Parameters required for test execution:
        - KUBECONFIG

     Scenario:
        1. Verify initial Nova service state counter for compute node
        2. Set compute service to Down state for one of the Nova compute nodes
        3. Check that osdpl_nova_service_state counter is changed for this node
        4. Set compute service to Up state for the Nova compute node
        5. Check that osdpl_nova_service_state is changed for this node
        6. Cleanup

    """

    _test_nova_service_metric(openstack_client_manager, metric_type='osdpl_nova_service_state')


def test_osdpl_ironic_nodes_metric(openstack_client_manager):
    """Verify osdpl_ironic_nodes_available and osdpl_ironic_nodes_total metrics
    Parameters required for test execution:
        - KUBECONFIG
        - environment with Ironic

     Scenario:
        1. Verify initial osdpl_ironic_nodes counters
        2. Create baremetal fake node
        3. Check osdpl_ironic_nodes counters
        4. Set test Ironic node to available state
        5. Check that osdpl_nova_service_state is changed for this node
        6. Cleanup

    """

    ironic_svc = False

    try:
        for svc in openstack_client_manager.hypervisor.list([]):
            if 'ironic' in svc.get('Hypervisor Type'):
                ironic_svc = True

        if not ironic_svc:
            msg = "Skipping test_osdpl_ironic_nodes_available_metric because Ironic service is not enabled"
            LOG.info(msg)
            pytest.skip(msg)

        LOG.info("Step 1: Verify initial osdpl_ironic_nodes_available counters")
        # wait until prometheus exporter responds with not empty counters
        waiters.wait(lambda: len(list(get_osdpl_prometheus_counters(openstack_client_manager,
                                 'osdpl_ironic_nodes_total').keys())) > 0, timeout=300, interval=10)

        osdpl_ironic_nodes_total_step1 = get_osdpl_prometheus_counters(openstack_client_manager,
                                                                       'osdpl_ironic_nodes_total')['{osdpl="osh-dev"']

        LOG.info("Step 2: Create baremetal fake node")
        openstack_client_manager.baremetal.node_create(["--name", ironic_test_server, "--driver", "fake-hardware"])

        LOG.info("Step 3: Check osdpl_ironic_nodes counters")
        waiters.wait(lambda: float(get_osdpl_prometheus_counters(openstack_client_manager,
                     'osdpl_ironic_nodes_total')['{osdpl="osh-dev"']) == float(osdpl_ironic_nodes_total_step1) + 1,
                     interval=10, timeout=300,
                     timeout_msg="osdpl_ironic_nodes_total counter is not incremented as expected")

        waiters.wait(lambda: check_ironic_nodes_counters(openstack_client_manager, exp_ctr='0.0'),
                     interval=10, timeout=300)

        LOG.info("Step 4: Set test Ironic node to available state")
        openstack_client_manager.baremetal.node_manage([ironic_test_server])
        openstack_client_manager.baremetal.node_provide([ironic_test_server])

        LOG.info("Step 5: Check osdpl_ironic_nodes_available is changed for this node")
        waiters.wait(lambda: check_ironic_nodes_counters(openstack_client_manager, exp_ctr='1.0'),
                     interval=10, timeout=300)

    finally:
        LOG.info("Step 6: Cleanup")
        openstack_client_manager.baremetal.node_delete([ironic_test_server])


def test_osdpl_certificate_expiry_metrics(openstack_client_manager):
    """Verify osdpl_certificate_expiry metrics:
        - keystone_public
        - libvirt_vnc_client
        - libvirt_vnc_server
        - octavia_amphora_ca

    Parameters required for test execution:
        - KUBECONFIG

     Scenario:
        1. Get osdpl_certificate_expiry counters
        2. Verify keystone_public counter
        3. Verify libvirt_vnc_client counter
        4. Verify libvirt_vnc_server counter
        5. Verify octavia_amphora_ca counter

    """

    try:

        current_ts = time.time()
        LOG.info("Step 1: Get osdpl_certificate_expiry counters")
        # wait until prometheus exporter responds with not empty counters
        waiters.wait(lambda: len(list(get_osdpl_prometheus_counters(openstack_client_manager,
                                      'osdpl_certificate_expiry').keys())) > 0, timeout=300, interval=10)

        osdpl_certificate_expiry = get_osdpl_prometheus_counters(openstack_client_manager, 'osdpl_certificate_expiry')

        LOG.info("Step 2: Verify keystone_public counter")
        keystone_public = osdpl_certificate_expiry['{identifier="keystone_public",osdpl="osh-dev"']
        assert float(keystone_public) > current_ts, \
            "keystone_public counter is incorrect and has a value of %s" % keystone_public

        LOG.info("Step 3: Verify libvirt_vnc_client counter")
        libvirt_vnc_client = osdpl_certificate_expiry['{identifier="libvirt_vnc_client",osdpl="osh-dev"']
        assert float(libvirt_vnc_client) > current_ts,\
            "libvirt_vnc_client counter is incorrect and has a value of %s" % libvirt_vnc_client

        LOG.info("Step 4: Verify libvirt_vnc_server counter")
        libvirt_vnc_server = osdpl_certificate_expiry['{identifier="libvirt_vnc_server",osdpl="osh-dev"']
        assert float(libvirt_vnc_server) > current_ts, \
            " libvirt_vnc_server counter is incorrect and has a value of %s" % libvirt_vnc_server

        LOG.info("Step 5: Verify octavia_amphora_ca counter")
        octavia_amphora_ca = osdpl_certificate_expiry['{identifier="octavia_amphora_ca",osdpl="osh-dev"']
        assert float(octavia_amphora_ca) > current_ts, \
            "octavia_amphora_ca is incorrect and has a value of %s" % octavia_amphora_ca

    finally:
        LOG.info("Test finished")


def _test_nova_service_metric(openstack_client_manager, metric_type):

    state_word_enable = "up"
    state_word_disable = "down"

    if metric_type == 'osdpl_nova_service_status':
        state_word_enable = "enabled"
        state_word_disable = "disabled"

    oc_name = openstack_client_manager.oc_name
    svc_name = f"{oc_name}-exporter"
    svc_ip = kubectl.get('svc', f'{svc_name} -o yaml', 'osh-system').result_yaml['spec']['clusterIP']

    try:
        # wait until prometheus exporter responds with not empty counters
        waiters.wait(lambda: len(list(get_osdpl_prometheus_counters(openstack_client_manager, metric_type).keys())) > 0,
                     timeout=300, interval=10)

        osdpl_nova_service_counters = get_osdpl_prometheus_counters(openstack_client_manager, metric_type)

        if metric_type == 'osdpl_nova_service_status':
            hostname = list(osdpl_nova_service_counters.keys())[0].split(',')[1].split('"')[1]
        else:
            hostname = list(osdpl_nova_service_counters.keys())[2].split(',')[1].split('"')[1]

        LOG.info(f"Step 1: Verify initial Nova service {metric_type} counter for compute node")
        waiters.wait(
            lambda: counter_is_true(openstack_client_manager, svc_ip, hostname,
                                    metric_type, state_word_disable, cvalue='0.0'), timeout=300, interval=10,
            timeout_msg=f"Counter {metric_type} is not changed as expected for {hostname}")

        waiters.wait(
            lambda: counter_is_true(openstack_client_manager, svc_ip, hostname,
                                    metric_type, state_word_enable, cvalue='1.0'), timeout=300, interval=10,
            timeout_msg=f"Counter {metric_type} is not changed as expected for {hostname}")

        if metric_type == 'osdpl_nova_service_status':
            LOG.info("Step 2: Disable compute service for one of the Nova compute nodes")
            openstack_client_manager.compute_service.set(["--disable", hostname, "nova-compute"])
        elif metric_type == 'osdpl_nova_service_state':
            LOG.info("Step 2: Set compute service to Down state for one of the Nova compute nodes")
            openstack_client_manager.compute_service.set(["--down", hostname, "nova-compute"])

        LOG.info(f"Step 3: Check that {metric_type} counter is changed for this node")
        waiters.wait(
            lambda: counter_is_true(openstack_client_manager, svc_ip, hostname, metric_type, state_word_disable,
                                    cvalue='1.0'), timeout=300, interval=10,
            timeout_msg=f"Counter {metric_type} %s is not changed as expected for {hostname}")

        waiters.wait(
            lambda: counter_is_true(openstack_client_manager, svc_ip, hostname, metric_type, state_word_enable,
                                    cvalue='0.0'), timeout=300, interval=10,
            timeout_msg=f"Counter {metric_type} is not changed as expected for {hostname}")

        if metric_type == 'osdpl_nova_service_status':
            LOG.info("Step 4: Enable compute service for one of the Nova compute nodes")
            openstack_client_manager.compute_service.set(["--enable", hostname, "nova-compute"])
        elif metric_type == 'osdpl_nova_service_state':
            LOG.info("Step 4: Set compute service to Up state for one of the Nova compute nodes")
            openstack_client_manager.compute_service.set(["--up", hostname, "nova-compute"])

        LOG.info(f"Step 5: Check {metric_type} is changed for this node")
        waiters.wait(
            lambda: counter_is_true(openstack_client_manager, svc_ip, hostname,
                                    metric_type, state_word_disable, cvalue='0.0'), timeout=300, interval=10,
            timeout_msg=f"Counter {metric_type} is not changed as expected for {hostname}")

        waiters.wait(
            lambda: counter_is_true(openstack_client_manager, svc_ip, hostname,
                                    metric_type, state_word_enable, cvalue='1.0'), timeout=300, interval=10,
            timeout_msg=f"Counter {metric_type} is not changed as expected for {hostname}")

    finally:
        LOG.info("Step 6: Cleanup")
        openstack_client_manager.compute_service.set(["--enable", hostname, "nova-compute"])
        openstack_client_manager.compute_service.set(["--up", hostname, "nova-compute"])
