import json
import pytest
import yaml

from si_tests import settings
from si_tests import logger
from si_tests.utils import templates, waiters

LOG = logger.logger


def get_mariadb_data_for_test(os_manager):
    LOG.info("Get mariadb-server statefulset")
    mariadb = os_manager.get_os_statefulset('mariadb-server')

    LOG.info("Get mariadb-server deployment replicas")

    deployment_values = os_manager.os_helm_manager.get_release_values(
        "openstack-mariadb"
    )

    deployment_replicas = deployment_values['pod']['replicas']['server']

    LOG.info("Check that currently we have replicas as described "
             "in deployment")
    mariadb_status = mariadb.read().status
    assert deployment_replicas == os_manager.get_k8s_object_field(
        mariadb_status, "ready_replicas")
    return mariadb, deployment_replicas


def check_mariadb_health(os_manager):
    LOG.info("Check mariadb-server status in osdpl")
    osdpl = os_manager.get_osdpl_deployment_status()

    def _wait_ready():
        osdpl_status = osdpl.read().status
        mariadb_status = osdpl_status['health']['mariadb']['server']['status']
        assert mariadb_status == 'Ready'

    waiters.wait_pass(
        _wait_ready, interval=30, expected=AssertionError,
        timeout=180)


@pytest.mark.usefixtures("mos_workload_downtime_report")
@pytest.mark.usefixtures("mos_per_node_workload_check_after_test")
def test_change_mariadb_replicas_count_and_wait_for_restoration(os_manager):
    """
    Test mariadb-server statefulset replicas changed
    and then restored to number that was in deployment.
    Parameters required for test execution:
        - KUBECONFIG
        - OPENSTACK_K8S_OBJECT_REPLICAS
    """
    replicas = settings.OPENSTACK_K8S_OBJECT_REPLICAS

    mariadb, deployment_replicas = get_mariadb_data_for_test(os_manager)

    os_manager.change_and_restore_replicas_count(
        mariadb, replicas, deployment_replicas)

    check_mariadb_health(os_manager)


@pytest.mark.usefixtures("mos_workload_downtime_report")
@pytest.mark.usefixtures("mos_per_node_workload_check_after_test")
def test_delete_mariadb_pod_and_wait_for_restoration(os_manager):
    """
       Test mariadb-server statefulset pods deleted
       and then it's quantity restored to number that was in deployment.
       Parameters required for test execution:
           - KUBECONFIG
           - OPENSTACK_K8S_OBJECT_NUM_DELETE_PODS
    """
    delete_num = settings.OPENSTACK_K8S_OBJECT_NUM_DELETE_PODS

    mariadb, deployment_replicas = get_mariadb_data_for_test(os_manager)

    assert delete_num <= deployment_replicas

    os_manager.delete_pod_and_wait_for_rescale(mariadb, delete_num,
                                               deployment_replicas)

    check_mariadb_health(os_manager)


@pytest.mark.skip(reason="PRODX-43713, PRODX-31186")
@pytest.mark.usefixtures("mos_workload_downtime_report")
@pytest.mark.usefixtures("mos_per_node_workload_check_after_test")
def test_force_kill_mariadb_pod_and_wait_for_restoration(os_manager):
    """
       Test mariadb-server statefulset pods force deleted
       and then it's quantity restored to number that was in deployment.
       Parameters required for test execution:
           - KUBECONFIG
           - OPENSTACK_K8S_OBJECT_NUM_DELETE_PODS
    """
    delete_num = settings.OPENSTACK_K8S_OBJECT_NUM_DELETE_PODS

    mariadb, deployment_replicas = get_mariadb_data_for_test(os_manager)

    assert delete_num <= deployment_replicas

    os_manager.delete_force_pod_and_wait_for_rescale(
        mariadb, delete_num, deployment_replicas,
        ["bash", "-c", f"kill -9 $(pidof {os_manager.mysql_binary})"], container='mariadb')

    check_mariadb_health(os_manager)


@pytest.mark.usefixtures("mos_workload_downtime_report")
@pytest.mark.usefixtures("mos_per_node_workload_check_after_test")
def test_force_sst_mariadb_and_wait_for_restoration(os_manager):
    """
       Test mariadb-server statefulset pods force sst
       by wiping mariadb data directory.
       Parameters required for test execution:
           - KUBECONFIG
           - OPENSTACK_K8S_OBJECT_NUM_DELETE_PODS
    """
    mariadb, deployment_replicas = get_mariadb_data_for_test(os_manager)

    os_manager.delete_force_pod_and_wait_for_rescale(
        mariadb, 1, deployment_replicas,
        ["bash", "-c", f"rm -rf /var/lib/mysql/*; kill -9 $(pidof {os_manager.mysql_binary})"],
        container='mariadb')

    check_mariadb_health(os_manager)


def test_mariadb_fill_data(os_manager):
    """
       Test to fill mariadb with test data
    """
    check_mariadb_health(os_manager)
    options = {
        "MARIADB_FILL_DATA_SIZE_MB": settings.MARIADB_FILL_DATA_SIZE_MB,
        "MARIADB_FILL_DATA_IMAGE": settings.MARIADB_FILL_DATA_IMAGE
    }
    pod_yaml = templates.render_template(settings.MARIADB_FILL_DATA_POD_YAML,
                                         options=options)

    json_body = json.dumps(yaml.load(pod_yaml, Loader=yaml.SafeLoader))
    job = os_manager.api.jobs.create(
        name="mariadb-fill-data", namespace=os_manager.openstack_namespace,
        body=json.loads(json_body))
    job.wait_succeded(timeout=settings.MARIADB_FILL_DATA_TIMEOUT,
                      interval=60)

    check_mariadb_health(os_manager)
