#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import pytest

from si_tests import logger, settings
from si_tests.utils import waiters
from datetime import datetime, timezone

LOG = logger.logger

ROTATION_TS_FORMAT = "%Y-%m-%d %H:%M:%S.%f"


def _get_octavia_expire_date(os_manager):
    octavia_pods = os_manager.api.pods.list_starts_with(
        "octavia-api", namespace="openstack")
    assert len(octavia_pods) > 0, "PODs <octavia-*> not found"
    octavia_pod = octavia_pods[0]
    octavia_cert_date = octavia_pod.exec(["/bin/bash", "-c",
                                          "openssl x509 -in /etc/octavia/certs/ca_01.pem -enddate -noout"])[9:-1]

    dt = datetime.strptime(octavia_cert_date, "%b %d %H:%M:%S %Y %Z")
    # Set timezone to UTC (since 'GMT' = UTC)
    dt = dt.replace(tzinfo=timezone.utc)
    return dt


@pytest.mark.usefixtures('mos_loadtest_os_refapp')  # Should be used if ALLOW_WORKLOAD == True
def test_octavia_certificate_rotation(os_manager, openstack_client_manager, show_step):
    """Verifies octavia certificate rotation procedure
    Parameters required for the test execution:
        - KUBECONFIG
    Scenario:
        1. Get octavia certificate expiration date
        2. Rotate octavia certificate
        3. Check certificate expiration date is updated
        4. Update Loadbalancer (disable members)
        5. Check Loadbalancer is active
        6. Update Loadbalancer (enable members)
        7. Check Loadbalancer is active

    """

    # Get Loadbalancer to update
    lb = openstack_client_manager.loadbalancer.list([])[0]

    # Get current expiration cert date
    show_step(1)
    previous_date = _get_octavia_expire_date(os_manager)

    # Rotate Octavia certs
    show_step(2)
    cmd = "osctl certificates rotate --osdpl osh-dev --type octavia:amphora --wait"
    LOG.info(f"Executing cmd '{cmd}' on Openstack controller")
    os_manager.osctlm.exec(cmd, timeout=settings.OPENSTACK_PASSWORD_ROTATION_TIMEOUT)

    # Check statuses
    # TODO(vsaienko): decrease timeout here when we sure that exec command does not break
    os_manager.wait_os_deployment_status(timeout=settings.OPENSTACK_PASSWORD_ROTATION_TIMEOUT)
    # Also we need to check osdplst of all services, because they can still be in IN_PROGRESS state
    LOG.info("Wait osdpl health status=Ready")
    os_manager.wait_openstackdeployment_health_status(timeout=1800)
    LOG.info("Octavia certificate rotation has been completed")

    # Check expiration cert date after rotation
    show_step(3)
    new_date = _get_octavia_expire_date(os_manager)
    assert new_date > previous_date, "Expire date in the updated certificate is not correct"

    # Update Loadbalancer
    show_step(4)
    # Skip unless PRODX-53730 is fixed
    # openstack_client_manager.loadbalancer.set([lb['id'], "--disable"])

    listener = openstack_client_manager.loadbalancer.listener_list(['--loadbalancer', lb['id']])[0]
    openstack_client_manager.loadbalancer.listener_set([listener['id'], "--disable"])

    # openstack loadbalancer pool list
    pool = openstack_client_manager.loadbalancer.pool_list(["--loadbalancer", lb['id']])[0]
    openstack_client_manager.loadbalancer.pool_set([pool['id'], "--disable"])

    # openstack loadbalancer member list <pool>
    member = openstack_client_manager.loadbalancer.member_list([pool['id']])[0]
    openstack_client_manager.loadbalancer.member_set([pool['id'], member['id'], "--disable"])
    openstack_client_manager.loadbalancer.member_set([pool['id'], member['id'], "--weight", "2"])

    # Check LB is Active
    show_step(5)
    waiters.wait(lambda: openstack_client_manager.loadbalancer.show([lb['id']])
                 .get('operating_status') == "ONLINE",
                 timeout=60, interval=10)

    show_step(6)
    # Skip unless PRODX-53730 is fixed
    # openstack_client_manager.loadbalancer.set([lb['id'], '--enable'])

    openstack_client_manager.loadbalancer.member_set([pool["id"], member['id'], "--enable"])
    openstack_client_manager.loadbalancer.pool_set([pool['id'], "--enable"])
    openstack_client_manager.loadbalancer.listener_set([listener['id'], "--enable"])

    # Check LB is Active
    show_step(7)
    waiters.wait(lambda: openstack_client_manager.loadbalancer.show([lb['id']])
                 .get('operating_status') == "ONLINE",
                 timeout=60, interval=10)
