import json
import os

import exec_helpers
import pytest
import yaml

from si_tests import settings
from si_tests import logger
import si_tests.utils.templates as utils
from si_tests.utils import helpers

LOG = logger.logger


@pytest.mark.parametrize('_', ['CLUSTER_NAME={0}'
                               .format(settings.TARGET_CLUSTER)])
@pytest.mark.usefixtures('store_cluster_description')
def test_sl(kaas_manager, stacklight_prerequisites, keycloak_ip, _):

    cluster_name = settings.TARGET_CLUSTER
    target_namespace = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    target_cluster = target_namespace.get_cluster(cluster_name)
    secrets = get_sl_uis_secrets(target_cluster)

    keycloak_address = settings.KEYCLOAK_URL or f'https://{keycloak_ip}'
    writer_password = kaas_manager.si_config.get_keycloak_user_password(
        'writer')

    target_cluster_hb = target_cluster.get_helmbundle()
    sl_release = get_sl_release(target_cluster_hb)
    sl_test_image = settings.STACKLIGHT_TEST_IMAGE or \
        get_sl_test_image(sl_release)
    sl_chart_url = sl_release['chartURL'] if sl_release else ""

    with open(os.path.join(settings.ARTIFACTS_DIR,
                           "stacklight_test_image.txt"), "w") as f:
        f.write(sl_test_image)
    with open(os.path.join(settings.ARTIFACTS_DIR,
                           "stacklight_chart_url.txt"), "w") as f:
        f.write(sl_chart_url)

    test_pod_ns = settings.STACKLIGHT_TEST_NAMESPACE
    test_pod_name = settings.STACKLIGHT_TEST_POD_NAME

    payload = {
        'KEYCLOAK_URL': keycloak_address,
        'KEYCLOAK_USER': 'writer',
        'KEYCLOAK_PASSWORD': writer_password,
        'KUBECONFIG_SECRET_NAME': stacklight_prerequisites.secret_name,
        'STACKLIGHT_TEST_SA': test_pod_name,
        'STACKLIGHT_TEST_POD_NAME': test_pod_name,
        'STACKLIGHT_TEST_IMAGE': sl_test_image
    }
    if settings.STACKLIGHT_TEST_POD_RESOURCES:
        payload.update({
          'STACKLIGHT_TEST_POD_RESOURCES': settings.STACKLIGHT_TEST_POD_RESOURCES
        })

    payload.update(secrets)
    run_sl_test(stacklight_prerequisites.parent_api,
                test_pod_ns, test_pod_name,
                payload, f'artifacts/lma_{cluster_name}_report.xml')

    # Checking all pods are in consistent state after test run
    # exclude conformance ns from ns_list for check pods
    pods_sanity_check(target_cluster, exclude_ns=['conformance'])


def run_sl_test(k8s_cluster, test_pod_ns, test_pod_name, payload, result_file):

    templates = utils.render_template(
        settings.STACKLIGHT_TEST_POD_YAML, payload)
    LOG.debug(templates)

    # yaml -> json
    json_body = json.loads(json.dumps(
        yaml.load(templates, Loader=yaml.SafeLoader)))
    run_sl_test_pod(k8s_cluster, test_pod_name, test_pod_ns,
                    json_body, result_file)


def get_sl_uis_secrets(target_cluster):
    sl_uis_svc_list = ['prometheus', 'alerta', 'alertmanager', 'grafana']

    if target_cluster.logging_enabled():
        LOG.info('Logging is enabled on this cluster')
        sl_uis_svc_list.append('kibana')

    kubectl_client = target_cluster.k8sclient

    secrets = {}
    for svc in sl_uis_svc_list:
        field_selector = f'metadata.name=iam-proxy-{svc}-config'
        LOG.info(f'Checking configmap with selector {field_selector}')
        cm = kubectl_client.api_core.list_namespaced_config_map(
            namespace='stacklight',
            field_selector=field_selector)

        if not cm.items:
            raise Exception(f'Cannot find configmap for {svc} proxy')

        cm = cm.items[0]
        if cm.data.get('config.yaml'):
            LOG.info('louketo-proxy detected, '
                     'reading encryption-key')
            secret_key = f'IAM_PROXY_{svc.upper()}_SECRET'
            cm_content = yaml.load(cm.data['config.yaml'],
                                   Loader=yaml.SafeLoader)
            secrets[secret_key] = cm_content['encryption-key']
        elif cm.data.get('oauth2.cfg'):
            LOG.info('oauth2-proxy detected, '
                     'skipping encryption-key reading')
    return secrets


def get_sl_test_image(sl_helm_release):
    sl_version = helpers.get_chart_version(sl_helm_release)

    sl_test_version = settings.STACKLIGHT_TEST_IMAGE_DICT.get(
        sl_version, sl_version)

    cdn_region = settings.ARTIFACTS_REGION_MAPPING[settings.CDN_REGION]
    sl_image_base = cdn_region['images_base_url']

    sl_test_image = (
        f'{sl_image_base}/stacklight/stacklight-pytest:{sl_test_version}')
    return sl_test_image


def get_sl_release(helmbundle):
    for release in helmbundle.data['spec']['releases']:
        if release['name'] == 'stacklight':
            return release
    return None


def run_sl_test_pod(k8s_cluster, pod_name, pod_ns, payload, result_file):
    pod = k8s_cluster.pods.create(
        name=pod_name, namespace=pod_ns, body=payload)
    pod.wait_phase('Running')
    pod.wait_test(filepath='./test_*', timeout=18600)

    pod.cp_from_pod(
        source_dir=settings.STACKLIGHT_TEST_REPORTS_DIR,
        compression='-z')

    exec_helpers.Subprocess().check_call(
        'test -f artifacts/report.xml && '
        f'mv artifacts/report.xml {result_file}')

    pod.delete()


def pods_sanity_check(target_cluster, exclude_ns=None):
    exclude_ns = exclude_ns or []
    kubectl_client = target_cluster.k8sclient
    ns_list = [namespace.name
               for namespace in kubectl_client.namespaces.list()
               if not any(ns in namespace.name for ns in exclude_ns)]
    LOG.info(f'ns for check: {ns_list}')
    target_cluster.check.check_k8s_pods(target_namespaces=ns_list)
