import os.path
import re
# import pytest

from si_tests import settings
from si_tests import logger
from si_tests.utils import utils

from si_tests.managers import bootstrap_manager


LOG = logger.logger


def test_prepare_vsphere_vm_template(openstack_client, show_step):
    """Bootstrap seed node and build VM template with selected OS.

    Scenario:
        1. Erase previous KaaS environment and seed node, if exists
        2. Create a seed VM using heat template
        3. Prepare seed node
        4. Check templates
        5. Download iso to seed node
        6. Build VM template
        7. Save artifacts
        8. Remove Heat stack
    """

    # step 001
    if settings.KEEP_ENV_BEFORE:
        LOG.info("Skip erase due KEEP_ENV_BEFORE flag is setted")
    else:
        if openstack_client._current_stack:
            # Erase the stack with the seed node
            show_step(1)
            seed_ip = openstack_client.get_seed_ip()
            old_bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)
            old_bootstrap.step_001_erase_env_before()
            LOG.info(f"Remove heat stack {settings.ENV_NAME}")
            openstack_client.delete_stack()
        else:
            LOG.warning("Can't erase environment due "
                        "seed node stack is absent")

    # Create a stack with a seed node
    show_step(2)
    openstack_client.create_seed_node()
    seed_ip = openstack_client.get_seed_ip()
    bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)
    bootstrap.wait_for_seed_node_ssh()

    # step 003
    show_step(3)
    bootstrap.step_003_prepare_seed_node()

    # step 004
    if settings.SKIP_BOOTSTRAP_TEMPLATES_CHECK:
        LOG.warning("Skip bootstrap templates check due "
                    "SKIP_BOOTSTRAP_TEMPLATES_CHECK flag is set")
    else:
        show_step(4)
        bootstrap.check_templates()

    remote = bootstrap.remote_seed()
    show_step(5)
    iso_file_name = os.path.basename(settings.KAAS_VSPHERE_TEMPLATE_ISO_URL)
    remote.execute(
        f"curl -L {settings.KAAS_VSPHERE_TEMPLATE_ISO_URL} -o /tmp/{iso_file_name}",
        verbose=True, timeout=7200)

    show_step(6)
    vsphere_cluster_name = 'vCluster-BMTeam'
    vsphere_os_name = 'centos'
    if settings.KAAS_CI_RHEL_LICENSES_IS_ACTIVATION_KEY and settings.KAAS_CI_RHEL_LICENSES_RPM_URL:
        vsphere_cluster_name = 'vCluster-BMTeam-RHEL'
    if 'rhel' in settings.KAAS_VSPHERE_TEMPLATE_OS_NAME:
        vsphere_os_name = 'rhel'
    if 'ubuntu' in settings.KAAS_VSPHERE_TEMPLATE_OS_NAME:
        vsphere_os_name = 'ubuntu'
    run_envs = {
        "VSPHERE_PACKER_ISO_FILE": f"/tmp/{iso_file_name}",
        "VSPHERE_CLUSTER_NAME": vsphere_cluster_name,
        "VSPHERE_PACKER_IMAGE_OS_NAME": vsphere_os_name,
        "VSPHERE_PACKER_IMAGE_OS_VERSION":
            settings.KAAS_VSPHERE_TEMPLATE_OS_VERSION,
        "CLUSTER_NAME": settings.CLUSTER_NAME,
        "TEMPLATES_DIR":
            f"{settings.KAAS_BOOTSTRAP_TARGET_DIR}/templates/vsphere"
    }
    if settings.KAAS_EXTERNAL_PROXY_ACCESS_STR:
        if settings.KAAS_SSL_PROXY_CERTIFICATE_FILE:
            # workaround for https://mirantis.jira.com/browse/PRODX-30668
            run_envs['PROXY_CA_CERTIFICATE_PATH'] = \
                f"/home/{settings.SEED_SSH_LOGIN}/{settings.KAAS_SSL_PROXY_CERTIFICATE_REMOTE_FILE}"
        run_envs['HTTP_PROXY'] = settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
        run_envs['HTTPS_PROXY'] = settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
        LOG.info(f"HTTP_PROXY and HTTPS_PROXY is set to {settings.KAAS_EXTERNAL_PROXY_ACCESS_STR}")
        if settings.KAAS_PROXYOBJECT_NO_PROXY:
            run_envs['NO_PROXY'] = settings.KAAS_PROXYOBJECT_NO_PROXY
            LOG.info(f"NO_PROXY is set to {settings.KAAS_PROXYOBJECT_NO_PROXY}")

    envs_string = utils.make_export_env_strting(run_envs)
    envs_string = envs_string + \
        f"; export VSPHERE_RO_USER={settings.KAAS_VSPHERE_RO_USER}" + \
        f"; export VSPHERE_RO_PASSWORD={settings.KAAS_VSPHERE_RO_PASSWORD}"
    res = remote.execute(f"{envs_string}; bash -x ./{settings.KAAS_BOOTSTRAP_TARGET_DIR}/bootstrap.sh "
                         "vsphere_template 2>&1 | tee -a image_build_log", verbose=True)

    assert res.exit_code == 0, \
        "Building a new vSphere template has been failed"
    show_step(7)
    template_name = None
    pattern = re.compile(r".*Full template path: (.*?)\. Please update.*")

    for line in res.stdout_str.splitlines():
        if "Full template path:" in line:
            LOG.info("Found target log line - %s", line)
            match = pattern.match(line)
            assert match is not None, \
                "Can't find template name in the bootstrap.sh output"
            template_name = match.group(1)
            break

    assert template_name is not None, \
        "Can't find template name in the bootstrap.sh output"
    artifacts_dir = settings.ARTIFACTS_DIR
    template_name_filename = os.path.join(artifacts_dir, "new_template_name")
    with open(template_name_filename, "w") as f:
        f.write(template_name)

    show_step(8)
    LOG.info(f"Remove heat stack {settings.ENV_NAME}")
    openstack_client.delete_stack()
