import os.path
import yaml

from si_tests import logger
from si_tests import settings
from si_tests.managers import bootstrap_manager
from si_tests.managers.kaas_manager import Manager
from si_tests.utils import waiters
from si_tests.clients.openstack import OpenstackClient
from si_tests.utils import templates

LOG = logger.logger


def test_mcc_bootstrap_cleanup(show_step):
    """Clean up MCC environment

    Scenario:
        1. Verify seed node accessibility
        2. Collect all deployed clusters (if kubeconfig exist)
        3. Remove all child clusters
        4. Try to cleanup MCC management/region clusters and remove seed node
    """

    show_step(1)
    openstack_client = None
    if settings.SEED_STANDALONE_EXTERNAL_IP:
        assert waiters.icmp_ping(settings.SEED_STANDALONE_EXTERNAL_IP)
        seed_ip = settings.SEED_STANDALONE_EXTERNAL_IP
    else:
        clouds_yaml = yaml.load(
            templates.render_template(settings.CLOUDS_YAML_PATH),
            Loader=yaml.Loader)
        auth = clouds_yaml["clouds"][settings.CLOUD_NAME]

        openstack_client = OpenstackClient(auth)
        seed_ip = openstack_client.get_seed_ip()

    show_step(2)
    if settings.KUBECONFIG_PATH and os.path.isfile(settings.KUBECONFIG_PATH):
        kaas_manager = Manager(kubeconfig=settings.KUBECONFIG_PATH)

        all_ns = kaas_manager.get_namespaces()
        LOG.info('Detected %s namespace(s) - %s',
                 len(all_ns), [n.name for n in all_ns])

        clusters = []
        for ns in all_ns:
            for child in ns.get_clusters():
                clusters.append(child)
        LOG.info("Collected clusters - {}".format([c.name for c in clusters]))

        show_step(3)
        clusters_meta = []
        for one in list(clusters):
            LOG.info("Deleting child cluster - %s", one.name)
            # collecting metadata as artifact
            clusters_meta.append(
                {
                    "name": one.name,
                    "uid": one.uid,
                    "namespace": one.namespace
                }
            )
            one.delete()
            LOG.info("Waiting for cluster deletion")
            one.check.check_cluster_deleted(timeout=settings.KAAS_CHILD_CLUSTER_DELETE_TIMEOUT)

        fname = os.path.join(settings.ARTIFACTS_DIR, 'deleted_clusters.yaml')
        with open(fname, 'w') as f:
            f.write(yaml.dump(clusters_meta))

    show_step(4)
    bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)
    bootstrap.step_001_erase_env_before()
    if openstack_client:
        LOG.info(f"Remove heat stack {settings.ENV_NAME}")
        openstack_client.delete_stack()
