import pytest
import yaml
import json
from kubernetes.client.rest import ApiException
from si_tests import logger
from si_tests import settings
from si_tests.deployments.utils.file_utils import save_to_yaml
from si_tests.utils import exceptions
from si_tests.utils import waiters
from si_tests.utils.utils import Provider

LOG = logger.logger


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.TARGET_CLUSTER)])
@pytest.mark.usefixtures("store_cluster_description")
def test_delete_child_cluster(kaas_manager, _):
    """Delete child cluster"""

    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    child_cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    child_cluster_provider = child_cluster.provider
    child_cluster_name = child_cluster.name
    mgmt_version = \
        kaas_manager.get_mgmt_cluster().spec['providerSpec']['value']['kaas'][
            'release']
    LOG.info(f"KaaS mgmt version is:{mgmt_version}")

    volumes_cleanup_check = False
    if settings.CHECK_RESOURCES_DELETION:
        child_cluster_resources = child_cluster.provider_resources.resources
        if child_cluster_provider is Provider.openstack:  # TODO PRODX-9728 Check that only OS provider
            child_cluster.set_volumes_cleanup_enabled()
            volumes_cleanup_check = True

    LOG.info(f"Deleting child cluster - {child_cluster_name}")
    child_cluster.delete()

    if child_cluster_provider == Provider.vsphere:

        def check_delete_lcmmachines():
            try:

                LOG.info("Check if the cluster %s exists",
                         child_cluster_name)
                if child_cluster.is_existed():
                    LOG.info("Check amount of lcmmachines in cluster %s",
                             child_cluster_name)
                    lcmmachines = len(child_cluster.get_lcmmachines())
                    LOG.info("Cluster %s has %s lcmmachine(s)",
                             child_cluster_name, lcmmachines)
                    if lcmmachines == 0:
                        return True
                else:
                    return True

            # HTTP response headers: HTTPHeaderDict(
            #   {'Cache-Control': 'no-cache, private',
            #    'Content-Length': '252',
            #    'Content-Type': 'application/json',
            #    'Date': 'Fri, 04 Jun 2021 18:53:18 GMT'})
            # HTTP response body: {
            #   "kind": "Status",
            #   "apiVersion": "v1",
            #   "metadata": {},
            #   "status": "Failure",
            #   "message": "clusters.cluster.k8s.io \"core-lcm-child-231\" not found",  # noqa
            #   "reason": "NotFound",
            #   "details":
            #       {"name": "core-lcm-child-231",
            #        "group": "cluster.k8s.io",
            #        "kind": "clusters"},
            #   "code": 404}
            except ApiException as ex:
                try:
                    body = json.loads(ex.body)
                except TypeError:
                    body = {}
                if str(ex.status) == "404" and \
                        ex.reason == "Not Found" and \
                        body.get('reason') == "NotFound" and \
                        str(body.get('code')) == "404" and \
                        body.get(
                            'details', {}).get(
                                "kind") == "clusters" and \
                        body.get(
                            'details', {}).get(
                                "name") == child_cluster_name:
                    return True
            return False

        LOG.info("Wait when lcmmachines will be deleted")
        waiters.wait(
            check_delete_lcmmachines,
            timeout=1200,
            interval=60)

    LOG.info("Waiting for cluster deletion")
    child_cluster.check.check_cluster_deleted(timeout=settings.KAAS_CHILD_CLUSTER_DELETE_TIMEOUT)

    if child_cluster_provider == Provider.baremetal:
        LOG.info('Check that BMHs are in ready state')
        ns.wait_baremetalhosts_statuses(
            wait_status='ready', retries=80, interval=30)

    if settings.CHILD_NAMESPACE_FORCE_CLEANUP:
        LOG.info("Cleanup child namespace")
        if child_cluster_provider == Provider.baremetal and \
                settings.KAAS_BM_WA_PRODX_6630:
            LOG.warning("Removing bmh before ns")
            bmhs_for_delete = []
            for bmh in ns.get_baremetalhosts():
                bmhs_for_delete.append(bmh)
                ns.delete_baremetalhost(name=bmh.name)
            ns.wait_all_bmhs_deletion(bmhs_for_delete=bmhs_for_delete)
        ns.delete()
        LOG.info("Waiting for child namespace deletion")
        ns.wait_for_deletion()
    if settings.CHECK_RESOURCES_DELETION:
        LOG.info("Will check resources deletion for {} provider".format(
            child_cluster_provider))
        try:
            waiters.wait(
                lambda:
                not child_cluster.provider_resources.check_resources_deletion(
                    child_cluster_resources, volumes_cleanup_check), timeout=1200, interval=120)
            save_to_yaml(child_cluster.provider_resources.not_deleted_resources,
                         settings.ARTIFACTS_DIR + f"/not_deleted_resources_{child_cluster_name}.yaml")
        except exceptions.TimeoutError:
            save_to_yaml(
                    child_cluster.provider_resources.not_deleted_resources,
                    settings.ARTIFACTS_DIR +
                    "/not_deleted_resources_{}.yaml".format(
                        child_cluster_name))
            raise Exception(
                    "Some resources were not deleted: {}".format(
                        yaml.dump(
                            child_cluster.provider_resources.not_deleted_resources))) # noqa
