import pytest

from si_tests import settings
from si_tests import logger
from si_tests.utils import utils
from si_tests.utils.utils import Provider

LOG = logger.logger


@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('log_method_time')
@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.KAAS_CHILD_CLUSTER_NAME)])
@pytest.mark.usefixtures("store_cluster_description")
@pytest.mark.usefixtures('introspect_child_deploy_objects')
def test_kaas_create_equinixmetalv2_child(kaas_manager, show_step, _):
    """Deploy child cluster on top of management cluster.

    Scenario:
        1. Collecting env data
        2. Creating namespace
        3. Creating public key
        4. Creating equinixmetal credential
        5. Create child cluster and machines with SL label
        6. Check cluster readiness
        7. Pod checks (optional)

    """

    # Collecting env data
    show_step(1)
    cluster_name = settings.KAAS_CHILD_CLUSTER_NAME
    namespace_name = settings.KAAS_NAMESPACE
    release_name = settings.KAAS_CHILD_CLUSTER_RELEASE_NAME
    LOG.info("Available releases: {}".format(
        kaas_manager.get_clusterrelease_names()))
    assert release_name in kaas_manager.get_clusterrelease_names()
    slave_nodes_count = int(settings.KAAS_CHILD_CLUSTER_SLAVE_NODES_COUNT)
    master_nodes_count = int(settings.KAAS_CHILD_CLUSTER_MASTER_NODES_COUNT)
    provider = Provider.equinixmetalv2
    rnd_string = utils.gen_random_string(4)
    custom_indexes = settings.KAAS_CHILD_CLUSTER_SET_CUSTOM_INDEXES
    indexes = utils.generate_upgrade_indexes()
    region = kaas_manager.get_mgmt_cluster().region_name

    # Creating namespace
    show_step(2)
    LOG.info("Namespace name - %s", namespace_name)
    ns = kaas_manager.get_or_create_namespace(namespace_name)

    # Creating public key
    show_step(3)
    public_key_name = "{cluster_name}-{rnd_string}-key".format(
        cluster_name=cluster_name, rnd_string=rnd_string)
    LOG.info("Public key name - {public_key_name}".format(
        public_key_name=public_key_name))
    with open(settings.KAAS_CHILD_CLUSTER_PUBLIC_KEY_FILE) as pub_key:
        pub_key_content = pub_key.read()
    ns.create_publickey(public_key_name, pub_key_content)

    # Creating Equinix Metal credentials
    show_step(4)
    credential_name = "{cluster_name}-{rnd_string}-cred".format(
        cluster_name=cluster_name, rnd_string=rnd_string)
    LOG.info("Credential name - {credential_name}".format(
        credential_name=credential_name))

    api_token = settings.KAAS_EQUINIX_API_TOKEN
    project_id = settings.KAAS_EQUINIX_PROJECT_ID
    ns.create_equinix_credential(
        credential_name,
        api_token,
        project_id,
        region=region)

    # Add proxy if fips enabled
    proxy_child_name = None
    if settings.KAAS_FIPS_ENABLED:
        mgmt_cluster = kaas_manager.get_mgmt_cluster()

        # Check mgmt cluster proxy
        pr_spec = mgmt_cluster.spec.get('providerSpec').get('value').get('proxy')
        proxy_object = kaas_manager.get_proxy_by_name(pr_spec)
        http_proxy = proxy_object[0].data['spec']['httpProxy']

        if kaas_manager.api.kaas_proxies.present("proxy-fips", ns.name):
            proxy_child_object = kaas_manager.api.kaas_proxies.get("proxy-fips", ns.name)
        else:
            proxy_child_object = ns.create_proxyobject(
                name="proxy-fips",
                region=region,
                proxy_str=http_proxy)

        proxy_child_name = proxy_child_object.data['metadata']['name']
        LOG.info(f"Proxy object {proxy_child_name} created")

    show_step(5)
    cluster = ns.create_cluster(
        name=cluster_name,
        release_name=release_name,
        credentials_name=credential_name,
        provider=provider.provider_name,
        region=region,
        lma_enabled=settings.KAAS_CHILD_CLUSTER_DEPLOY_LMA,
        public_key_name=public_key_name,
        services_cidr='172.21.0.0/18',
        pods_cidr='172.21.128.0/17',
        secure_overlay=settings.KAAS_CHILD_CLUSTER_SECURE_OVERLAY_ENABLED,
        proxy_name=proxy_child_name)

    facility = cluster.spec['providerSpec']['value']['facility']

    if custom_indexes:
        LOG.info("Custom indexes is enabled. Random index will be set during machine creation")
    for node in range(master_nodes_count):
        machine_index = indexes.__next__() if custom_indexes else None
        cluster.create_equinix_machine(
            node_type="master",
            billing_cycle=settings.KAAS_EQUINIX_BILLING_CYCLE,
            machine_type=settings.KAAS_EQUINIX_CHILD_MACHINE_TYPE_MASTER,
            region=region,
            provider_name=provider.provider_name,
            facility=facility,
            upgrade_index=machine_index)

    labels = \
        [{'key': k, 'value': v} for k, v in
         cluster.get_allowed_node_labels().items()]

    stacklight_nodes_count = \
        3 if slave_nodes_count >= 3 else slave_nodes_count
    for node in range(stacklight_nodes_count):
        machine_index = indexes.__next__() if custom_indexes else None
        cluster.create_equinix_machine(
            node_type="node",
            billing_cycle=settings.KAAS_EQUINIX_BILLING_CYCLE,
            machine_type=settings.KAAS_EQUINIX_CHILD_MACHINE_TYPE_STACKLIGHT,
            region=region,
            node_labels=labels,
            provider_name=provider.provider_name,
            facility=facility,
            upgrade_index=machine_index)

    slave_nodes_count -= stacklight_nodes_count

    for node in range(slave_nodes_count):
        machine_index = indexes.__next__() if custom_indexes else None
        cluster.create_equinix_machine(
            node_type="node",
            billing_cycle=settings.KAAS_EQUINIX_BILLING_CYCLE,
            machine_type=settings.KAAS_EQUINIX_CHILD_MACHINE_TYPE_WORKER,
            region=region,
            provider_name=provider.provider_name,
            facility=facility,
            upgrade_index=machine_index)

    # Waiting for machines are Ready and helmbundles are deployed
    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()

    machines = [
        x for x in cluster.get_machines(machine_type="worker")
        if 'nodeLabels' in x.spec['providerSpec']['value'].keys() and
        labels == x.nodeLabels
    ]
    assert len(machines) == stacklight_nodes_count,\
        "Not all machines have expected nodeLabels"

    for machine in machines:
        machine.check_k8s_nodes_has_labels()

    show_step(6)
    cluster.check.check_cluster_readiness()
    cluster.check.check_helmbundles()

    cluster.check.check_deploy_stage_success(skipped_stages_names='IAM objects created')
    # Collecting artifacts
    cluster.store_k8s_artifacts()

    cluster.check.check_k8s_nodes()

    show_step(7)
    if settings.RUN_POD_CHECKS:
        # Check/wait for correct docker service replicas in cluster
        cluster.check.check_actual_expected_docker_services()
        cluster.check.check_k8s_pods()
        cluster.check.check_actual_expected_pods(timeout=3200)

        # Due deployment of ceph takes long time we need wait when all pods
        # are created and running
        try:
            health_info = cluster.check.get_ceph_health_detail()
            assert health_info['status'] == "HEALTH_OK", f'Health is not OK. Will not proceed. ' \
                                                         f'Current ceph health status: {health_info}'
        except AssertionError:
            cluster.check.wait_ceph_health_status('HEALTH_OK')
        cluster.check.check_ceph_pvc(base_image_repo='127.0.0.1:44301')
    cluster.check.check_overlay_encryption_functionality()
    cluster.provider_resources.save_artifact()
