import functools
import time
import multiprocessing
from queue import Empty

from si_tests.managers import openstack_manager
from si_tests import logger

LOG = logger.logger


class Result:
    error = "ERROR"
    ok = "OK"


def process_helper(func):
    @functools.wraps(func)
    def wrapped(queue, timeout):
        os_manager = openstack_manager.OpenStackManager()
        func_name = func.__name__
        try:
            func(os_manager, timeout)
            queue.put((Result.ok, func_name))
        except Exception:
            LOG.exception("%s step failed", func_name)
            queue.put((Result.error, func_name))
    return wrapped


def finalize_processes(processes):
    [process.terminate() for process in processes]
    [process.join() for process in processes]


def run_step(actions, timeout):
    queue = multiprocessing.Queue()

    processes = []
    for action in actions:
        process = multiprocessing.Process(
            target=process_helper(action),
            args=(queue, timeout))
        process.start()
        processes.append(process)

    actions = set(action.__name__ for action in actions)
    not_completed = set(action for action in actions)

    for _ in range(len(processes)):
        start_time = time.time()
        completed = actions - not_completed
        try:
            result, func_name = queue.get(
                block=True,
                timeout=timeout)
        except Empty:
            finalize_processes(processes)
            msg = "failed due to timeout. Actions that " \
                  "yet not completed: {}. Completed {}".format(
                    not_completed, completed)
            return False, msg

        timeout = timeout - (time.time() - start_time)

        if result == Result.error:
            finalize_processes(processes)
            msg = "{} failed. Actions that yet not " \
                  "completed: {}. Completed {}".format(
                    func_name, not_completed, completed)
            return False, msg
        else:
            not_completed.remove(func_name)
    return True, None


def assert_step(actions, step, timeout):
    LOG.info("Run %s step", step)
    result, failed_msg = run_step(actions, timeout)
    if not result:
        raise Exception("Step {} failed: {}".format(step, failed_msg))
    LOG.info("Step %s successfully executed", step)
