import base64
import configparser
import os
import re
import io
import json
import random
import time
import yaml

from distutils.version import LooseVersion
from functools import cached_property
from kubernetes.client.rest import ApiException
from retry import retry

from si_tests import settings
from si_tests.clients.k8s import K8sCluster
from si_tests import logger
from si_tests.utils import waiters
from si_tests.managers import helm3_manager
from si_tests.managers.osctl_client_manager import OsCtlManager
from si_tests.managers.tungstenfafric_manager import TFManager
from si_tests.utils import packaging_version as version
import si_tests.utils.templates as templates_utils
from si_tests.utils import utils

LOG = logger.logger


class OpenStackManager(object):
    kubeconfig = None
    _os_controller_version = None
    _tfoperator_version = None

    def __init__(self, kubeconfig=settings.KUBECONFIG_PATH):
        self._api = None
        self.kubeconfig = kubeconfig
        self.openstack_namespace = "openstack"
        self.os_helm_system_namespace = "osh-system"
        self.tf_namespace = "tf"
        self.redis_namespace = "openstack-redis"
        self.os_helm_manager = helm3_manager.Helm3Manager(self.openstack_namespace)
        self.osctlm = OsCtlManager(kubeconfig=kubeconfig)
        self.tf_manager = TFManager(kubeconfig=kubeconfig)

    @property
    def api(self) -> K8sCluster:
        """
        :rtype: cluster.K8sCluster
        """
        if self._api is None:
            if not os.path.isfile(self.kubeconfig):
                raise FileNotFoundError(
                    "kubeconfig file={} not found!".format(self.kubeconfig)
                )
            self._api = K8sCluster(kubeconfig=self.kubeconfig)
        return self._api

    def _skip_resource(self, name, skip_list):
        skip_list = skip_list or []
        for skip_item in skip_list:
            if re.match(skip_item, name):
                return True

    def is_rockoon_used(self):
        return utils.is_rockoon_used(self.api)

    @property
    def oc_name(self):
        if self.is_rockoon_used():
            return "rockoon"
        return "openstack-controller"

    def os_controller_version(self, cache=False):
        if self._os_controller_version is None or cache is False:
            self._os_controller_version = self.osctlm.exec(
                f"pip freeze |awk -F '='  '/{self.oc_name}/" + "{print $3}' | tr -d '\n'")["stdout"]
            LOG.info(f"Discovered openstack controller version is: {self._os_controller_version}")
        return self._os_controller_version

    def is_db_backup_sync_remote_enabled(self):
        osdpl = self.get_osdpl_deployment()
        return (
            osdpl.data["spec"]["features"]
            .get("database", {})
            .get("backup", {})
            .get("sync_remote", {})
            .get("enabled", False)
        )

    @property
    def is_tf_enabled(self):
        osdpl = self.get_osdpl_deployment(read=True)
        try:
            backend = osdpl.spec['features']['neutron'].get('backend')
        except KeyError:
            return False
        return backend == 'tungstenfabric'

    def is_ironic_enabled(self):
        osdpl = self.get_osdpl_deployment(read=True)
        return True if 'baremetal' in osdpl.spec.get('features', {}).get('services', []) else False

    def is_encrypted_volume_type_described(self):
        osdpl = self.get_osdpl_deployment(read=True)
        try:
            osdpl.spec['services']['block-storage']['cinder']['values']['bootstrap']['volume_types'][
                       settings.OPENSTACK_ENCRYPTED_VOLUME_NAME]['arguments']['encryption-provider']
        except KeyError:
            return False
        return True

    def is_pass_hwveb_ports_to_os_vif_plugin(self):
        osdpl = self.get_osdpl_deployment()
        return (
            osdpl.data["spec"]["services"]
            .get("compute", {})
            .get("nova", {})
            .get("values", {})
            .get("conf", {})
            .get("nova", {})
            .get("workarounds", {})
            .get("pass_hwveb_ports_to_os_vif_plugin", False)
        )

    @property
    def is_ovn_enabled(self):
        osdpl = self.get_osdpl_deployment(read=True)
        try:
            backend = osdpl.spec['features']['neutron'].get('backend')
        except KeyError:
            return False
        return backend == 'ml2/ovn'

    @property
    def is_value_from_used(self):
        version = self.os_controller_version()
        if version is None:
            return False
        # Starting from 0.11.8 openstack controller image version
        # we store sensitive data in kubernetes secrets
        if LooseVersion(version) >= LooseVersion("0.11.8"):
            return True

    @property
    def is_dns_test_record_present(self):
        version = self.os_controller_version()
        if version is None:
            return False
        if LooseVersion(version) > LooseVersion("0.17.0.dev1"):
            return True

    def get_credentials_rotation_timestamp(self, creds_type):
        osdplst = self.get_osdpl_deployment_status(read=True)
        return osdplst.status.get("credentials", {}).get("rotation", {}).get(creds_type, {}).get("timestamp")

    def get_resources(self, resources_type, namespace, read=False, skip_list=None):
        resources_api = getattr(self.api, resources_type)
        resources = resources_api.list(namespace)
        res = []
        for resource in resources:
            if not self._skip_resource(resource.name, skip_list):
                if read:
                    resource = resource.read()
                res.append(resource)
        return res

    def get_os_resources(self, resources_type, read=False, skip_list=None):
        return self.get_resources(resources_type, self.openstack_namespace, read, skip_list)

    def get_os_resource(self, resources_type, resource_name, namespace, read=False):
        resources_api = getattr(self.api, resources_type)
        resource = resources_api.get(name=resource_name, namespace=namespace)
        if read:
            return resource.read()
        return resource

    def get_secret_decoded(self, resource_name, namespace):
        secret_data = self.get_os_resource('secrets', resource_name, namespace, read=True)
        data = {
            key: base64.b64decode(value).decode("utf-8")
            for key, value in secret_data.to_dict().get('data', {}).items()
        }
        return data

    def get_os_deployments(self, read=False):
        return self.get_os_resources("deployments", read)

    def get_os_statefulsets(self, read=False):
        return self.get_os_resources("statefulsets", read)

    def get_os_secrets(self, read=False, skip_list=None):
        # NOTE(vsaienko): internal helm data should be skipped
        # for all operations.
        if skip_list is None:
            skip_list = ["sh.helm.release.v1.*"]
        return self.get_os_resources("secrets", read, skip_list)

    def get_os_daemonsets(self, read=False):
        return self.get_os_resources("daemonsets", read)

    def get_os_configmaps(self, read=False):
        return self.get_os_resources("configmaps", read)

    def get_os_pods(self, read=False):
        return self.get_os_resources("pods", read)

    def get_os_pod(self, name, read=False):
        return self.get_os_resource("pods", name, self.openstack_namespace, read)

    def get_powerdns_svc_ip(self):
        svc = self.get_os_service("designate-powerdns-external")
        svc = svc.read()
        return svc.status.load_balancer.ingress[0].ip

    def get_ingress_svc_external_ip(self):
        svc = self.get_os_service("ingress")
        svc = svc.read()
        lb_ingress = svc.status.load_balancer.ingress
        return lb_ingress[0].ip if lb_ingress else ''

    @cached_property
    def keystone_client_pod(self):
        keystone_client_pods = [pod for pod in self.get_os_pods() if 'keystone-client' in pod.name]
        keystone_client_pod = None
        for pod in keystone_client_pods:
            pod_obj = pod.read()
            if pod_obj.status.phase == "Running":
                keystone_client_pod = pod
                break
        assert keystone_client_pod, "No pods found in Running phase with keystone-client name prefix"
        return keystone_client_pod

    def get_os_operator_helmbundle(self, name, read=False):
        return self.get_os_resource(
            "kaas_helmbundles", name, self.os_helm_system_namespace, read
        )

    def get_os_operator_deployment(self):
        return self.get_os_resource(
            "deployments",
            self.oc_name,
            self.os_helm_system_namespace,
            read=False,
        )

    @property
    def operator_config(self):
        cm = self.get_os_resource(
            "configmaps",
            f"{self.oc_name}-config",
            self.os_helm_system_namespace,
            read=False
        )
        data = cm.read().to_dict()["data"].get("extra_conf.ini")
        config = configparser.ConfigParser()
        config.read_string(data)
        return config

    @operator_config.setter
    def operator_config(self, config):
        cm = self.get_os_resource(
            "configmaps",
            f"{self.oc_name}-config",
            self.os_helm_system_namespace,
        )
        body = cm.read().to_dict()
        with io.StringIO() as ss:
            config.write(ss)
            ss.seek(0)
            data = ss.read()
        body["data"]["extra_conf.ini"] = data
        cm.patch(body=body)

    @property
    def lma_config(self):
        os_controller_version = self.os_controller_version()
        # Lma config is added starting 0.13.2
        if version.parse(os_controller_version) >= version.parse("0.13.2"):
            secret = self.get_os_resource(
                "secrets",
                "openstack-lma-config",
                "openstack-lma-shared",
                read=False
            )

            encoded_data = secret.read().to_dict()["data"]["conf.json"]
            return json.loads(base64.b64decode(encoded_data).decode("utf-8"))
        else:
            return {}

    def get_os_coredns_helmbundle(self, read=True):
        return self.get_os_resource(
            "kaas_helmbundles", "coredns", self.os_helm_system_namespace, read
        )

    def get_openstackdeployment(self, name, read=False):
        return self.get_os_resource(
            "openstackdeployment", name, self.openstack_namespace, read
        )

    def get_openstackdeployment_status(self, name, read=False):
        return self.get_os_resource(
            "openstackdeploymentstatus", name, self.openstack_namespace, read
        )

    def get_tf_helmbundle(self, name, read=False):
        return self.get_os_resource("kaas_helmbundles", name, self.tf_namespace, read)

    def get_os_deployment(self, name, read=False):
        return self.get_os_resource(
            "deployments", name, self.openstack_namespace, read
        )

    def get_os_statefulset(self, name, read=False):
        return self.get_os_resource(
            "statefulsets", name, self.openstack_namespace, read
        )

    def get_os_service(self, name, read=False):
        return self.get_os_resource("services", name, self.openstack_namespace, read)

    def get_os_helmbundle_versions(self):
        os_helmbundles_version_map = {}
        release_values = self.os_helm_manager.get_releases_values()
        for release_name, values in release_values.items():
            hb_name = (
                values.get("lcm.mirantis.com/v1alpha1", {})
                .get(self.oc_name, {})
                .get("helmbundle", {})
                .get("name")
            )
            oc_fp = values.get("lcm.mirantis.com/v1alpha1", {}).get(
                self.oc_name, {}
            )
            os_helmbundles_version_map[hb_name] = {
                "fingerprint": oc_fp.get("fingerprint"),
                "version": oc_fp.get("version"),
                "osdpl_generation": oc_fp.get("osdpl_generation"),
            }
        return os_helmbundles_version_map

    def get_os_helmbundle_images(self):
        os_helmbundles_images_map = {}
        release_values = self.os_helm_manager.get_releases_values()
        for release_name, values in release_values.items():
            hb_name = (
                values.get("lcm.mirantis.com/v1alpha1", {})
                .get(self.oc_name, {})
                .get("helmbundle", {})
                .get("name")
            )
            if hb_name in os_helmbundles_images_map:
                os_helmbundles_images_map[hb_name].update(
                    {release_name: values["images"]["tags"]}
                )
            else:
                os_helmbundles_images_map[hb_name] = {
                    release_name: values["images"]["tags"]
                }
        return os_helmbundles_images_map

    def check_resource_version_updated(
        self,
        resource_list_before,
        resource_list_after,
        updated=True,
        skip_list=None,
    ):
        """Check metadata resourceVersion parameter update
        Args:
            :param resource_list_before: (list) list of objects before
            :param resource_list_after: (list) list of objects after
            :param skip_list: (list) list of resource names to skip check
            :param updated: (bool) should be version updated or not
        """
        before = {resource.metadata.name: resource for resource in resource_list_before}
        for resource_after in resource_list_after:
            if self._skip_resource(resource_after.metadata.name, skip_list):
                continue
            resource_before = before[resource_after.metadata.name]
            resource_version_before = resource_before.metadata.resource_version
            resource_version_after = resource_after.metadata.resource_version

            err_msg = (
                "resourceVersion was updated for"
                " {name} {uid} from {before} to {after}".format(
                    name=resource_before.metadata.name,
                    uid=resource_before.metadata.uid,
                    before=resource_version_before,
                    after=resource_version_after,
                )
            )
            actual = not (resource_version_before == resource_version_after)
            assert updated == actual, err_msg

    def check_resource_generation_bumped(
        self, resource_list_before, resource_list_after, bumped=True
    ):
        """Check metadata generation bumped
        Args:
            :param resource_list_before: (list) list of objects before
            :param resource_list_after: (list) list of objects after
            :param bumped: (bool) should be version bumped or not
        """

        before = {resource.metadata.name: resource for resource in resource_list_before}
        for resource_after in resource_list_after:
            resource_before = before[resource_after.metadata.name]
            gen_before = resource_before.metadata.generation
            gen_after = resource_after.metadata.generation

            err_msg = (
                "Generation bump check failed for {name}. "
                "Should updated: {bumped}, before: {before}, "
                "after: {after}.".format(
                    name=resource_before.metadata.name,
                    bumped=bumped,
                    before=gen_before,
                    after=gen_after,
                )
            )
            actual = not (gen_before == gen_after)
            assert bumped == actual, err_msg

    def get_osdpl_deployment(self, read=False):
        """
        Get OSDPL deployment
        :return: osdpl deployment
        :rtype: OpenStackDeployment
        """
        return self.get_openstackdeployment(settings.OSH_DEPLOYMENT_NAME, read)

    def get_osdpl_deployment_status(self, read=False):
        """
        Get OSDPL deployment Status
        :return: osdpl deployment status
        :rtype: OpenStackDeploymentStatus
        """
        return self.get_openstackdeployment_status(settings.OSH_DEPLOYMENT_NAME, read)

    def check_osdpl_status_value(self, section, value, equal=True):
        """Wait osdpl status section value
        Args:
            :param section: (string) section to check value
            :param value: value to check
            :param equal: (bool) should be equal or not
        """

        osdpl = self.get_osdpl_deployment(read=True)
        actual = osdpl.status[section] == value
        return actual == equal

    def wait_openstack_helmbundles_fingerprint(self, name, timeout=600, interval=60):
        """Wait till all helmbundles gets fingerprint of current
           OpenStack deployment
        Args:
            :param name: name of OpenStack deployment
            :param timeout: timeout to wait
            :param interval: time between checks
        """

        def _check_helmbundles_fingerprint():
            release_values = self.os_helm_manager.get_releases_values()
            if all(
                release.get("lcm.mirantis.com/v1alpha1", {})
                .get(self.oc_name, {})
                .get("fingerprint")
                == expected_fingerprint
                for release in release_values.values()
            ):
                return True
            return False

        osdpl = self.get_openstackdeployment(name, read=True)
        expected_fingerprint = osdpl.status["fingerprint"]

        err = (
            "Openstack helmbundles fingerprint wasn't"
            " updated to actual version of"
            " openstackdeployment: {expected_fingerprint}".format(
                expected_fingerprint=expected_fingerprint
            )
        )
        waiters.wait(
            _check_helmbundles_fingerprint,
            timeout=timeout,
            interval=interval,
            timeout_msg=err,
        )
        LOG.info("OpenStack Helmbundles fingerprints are in actual state")

    def wait_openstack_helmbundles_version_update(self, name, timeout=600, interval=60):
        """Wait till all helmbundles gets version of current
           OpenStack deployment
        Args:
            :param name: name of OpenStack deployment
            :param timeout: timeout to wait
            :param interval: time between checks
        """

        def _check_helmbundles_version():
            helmbundles_version_map = self.get_os_helmbundle_versions()
            if all(
                version["version"] == expected_version
                for version in helmbundles_version_map.values()
            ):
                return True
            return False

        osdpl = self.get_openstackdeployment(name)
        expected_version = osdpl.read().status["version"]

        err = (
            "Openstack helmbundles version wasn't"
            " updated to actual version of"
            " openstackdeployment: {expected_version}".format(
                expected_version=expected_version
            )
        )
        waiters.wait(
            _check_helmbundles_version,
            timeout=timeout,
            interval=interval,
            timeout_msg=err,
        )
        LOG.info("OpenStack Helmbundles versions are in actual state")

    def wait_openstackdeployment_health_status(
        self, expected_status="Ready", timeout=2700, interval=90
    ):
        mariadb_restarts = {}  # for W/A PRODX-31186

        def _check_osdpl_health():
            failed_services = []
            health_map = (
                self.get_openstackdeployment_status(settings.OSH_DEPLOYMENT_NAME)
                .read()
                .status["health"]
            )

            for service_name, subservice_map in health_map.items():
                for (
                    subservice_name,
                    subservice_statuses,
                ) in subservice_map.items():
                    if subservice_statuses["status"] != expected_status:
                        failed_services.append(
                            "{sn}/{sbn}: {status}".format(
                                sn=service_name,
                                sbn=subservice_name,
                                status=subservice_statuses["status"],
                            )
                        )

            # for W/A PRODX-31186
            if 'mariadb/server: Unhealthy' in failed_services:
                self.wa_prodx_31186(mariadb_restarts)

            if failed_services:
                msg = (
                    "Timeout waiting for services statuses."
                    "After {timeout} sec next services are not "
                    "in {expected_status} state: {failed_services}.".format(
                        timeout=timeout,
                        expected_status=expected_status,
                        failed_services=failed_services,
                    )
                )
                LOG.warning(msg)
                return False
            return True

        waiters.wait(_check_osdpl_health, timeout=timeout, interval=interval)
        LOG.info(
            "All Services in OpenStack deployment are in "
            "{status} status.".format(status=expected_status)
        )

    def wait_os_deployment_status(self, timeout, status="APPLIED"):
        """
        Wait OpenStack deployment status; Use OpenStackDeploymentStatus k8s object
        Args:
            timeout: timeout to wait
            status: expected sdpl state

        Returns: None or TimeoutError

        """

        # kubernetes api return 404 if kind openstackdeploymentstatus object has not yet been created
        # we need to retry to prevent false negative scenario
        @retry((ApiException,), delay=60, tries=10, logger=LOG)
        def _wait(expected):
            osdpl = self.get_osdpl_deployment_status().read()
            return (
                osdpl.status.get("osdpl") is not None
                and expected == osdpl.status["osdpl"]["state"]
            )

        LOG.info(f"Wait osdpl state is {status}")
        waiters.wait(
            _wait,
            predicate_args=(status,),
            timeout=timeout,
            timeout_msg=f"OpenStack deployment doesn't reached status `{status}'",
        )

    def wait_osdpl_services(self, status="APPLIED", timeout=600, interval=60):
        """
        Wait all osdpl services state
        :param status: expected service state
        :param timeout: timeout to wait
        :param interval: time between checks
        """

        def _check_service_state():
            processing_services = []
            osdpl_service = self.get_osdpl_deployment_status(read=True).status[
                "services"
            ]
            for name, params in osdpl_service.items():
                if params["state"] != status:
                    processing_services.append(f"{name}: {params['state']}")
            LOG.info(f"Next services still processing: {processing_services}")
            return processing_services == []

        LOG.info(f"Wait osdpl services state is {status}")
        waiters.wait(
            _check_service_state,
            timeout=timeout,
            interval=interval,
            timeout_msg=f"Not all services received {status} status",
        )

    # TODO should be replaced with wait_os_deployment_status() and use OpenStackDeploymentStatus
    def wait_osdpl_status(self, timeout):
        LOG.info("Wait osdpl.status.deployed is True")
        waiters.wait(
            (
                lambda expected: expected
                == self.get_openstackdeployment(settings.OSH_DEPLOYMENT_NAME)
                .read()
                .status["deployed"]
            ),
            predicate_kwargs={"expected": True},
            timeout=timeout,
            timeout_msg="OpenStack deployment doesn't reached"
            " status `deployed=True`",
        )

        LOG.info("Wait until all osdpl children statuses success=True")
        self.wait_all_osdpl_children_status()

    def get_osdpl_fingerprint(self):
        osdpl = self.get_osdpl_deployment_status().read()
        return osdpl.status['osdpl']['fingerprint']

    def wait_osdpl_fingerprint_changed(self, fingerprint, interval=30, timeout=600):
        waiters.wait(lambda: fingerprint != self.get_osdpl_fingerprint(),
                     interval=interval, timeout=timeout)

    def wait_all_osdpl_children_status(self, success=True, timeout=600, interval=60):
        """
        Wait all osdpl children status
        :param success: expected children status
        :param timeout: timeout to wait
        :param interval: time between checks
        """

        return self.wait_osdpl_services(timeout=timeout, interval=interval)

    @staticmethod
    def _is_owner_of_kind(resource, kind):
        references = resource.metadata.owner_references
        if references:
            return references[0].kind == kind
        return False

    def wait_resources(
        self, namespace, timeout=300, interval=60, os_transitional_replicas=False, exclude_node_shutdown=False
    ):
        start = time.time()

        def _wait_objects():
            not_ready_resources = []
            for resource in ['daemonsets', 'statefulsets', 'deployments']:
                for obj in getattr(self.api, resource).list(namespace=namespace):
                    if os_transitional_replicas:
                        # Check only ready replicas count match desired replicas count
                        # For checks between ClusterUpdatePlan steps
                        if obj.desired_replicas != obj.ready_replicas:
                            not_ready_resources.append((resource, obj.name))
                        elif not obj.ready:
                            # Just log the warning about replicas generation
                            LOG.warning(f"'{resource}/{obj.name}' replicas '{obj.ready_replicas}' match the desired "
                                        f"replicas count, but some replicas still run old object generation")
                    else:
                        # Check that ready replicas match the replicas from the latest generation
                        # For normal cluster readiness checks
                        if not obj.ready:
                            not_ready_resources.append((resource, obj.name))
            msg = f"Some objects {not_ready_resources} are not ready"
            LOG.info(msg)
            assert not not_ready_resources, msg

        if os_transitional_replicas:
            LOG.warning("os_transitional_replicas=True, checking only objects replicas count")
        waiters.wait_pass(_wait_objects, AssertionError, interval, timeout)
        self.wait_all_os_pods(timeout, interval, exclude_node_shutdown)
        timeout = max(1, int(timeout - (time.time() - start)))
        LOG.info("Wait for Openstack jobs")
        self.wait_all_os_jobs(timeout, interval)

    def wait_os_resources(
        self, timeout=300, interval=60, os_transitional_replicas=False, exclude_node_shutdown=False
    ):
        self.wait_resources(
            self.openstack_namespace, timeout, interval, os_transitional_replicas, exclude_node_shutdown
        )

    def wr_prodx_54579(self):
        """Workaround for PRODX-54579: Find and delete rfs-openstack-redis pods in Completed state
        after reboot/shutdown procedures, especially HA. These pods are appeared on env because of
        PRODX-49106. And redis can't restart on env if we have rfs-openstack-redis in Completed state,
        so we need to find these pods and delete them manually.
        """
        LOG.banner("WA for PRODX-54579, Find and delete rfs-openstack-redis pods in Completed state if exist")
        redis_pods_comp_status = [
            p for p in self.api.pods.list_pods_except_jobs(target_namespaces=self.redis_namespace)
            if (
                p['status']['phase'] == "Succeeded"
                and 'rfs-openstack-redis' in p['metadata']['name']
                and any(
                    cond.get('message') and 'Pod was terminated in response '
                                            'to imminent node shutdown.' in cond['message']
                    for cond in p['status'].get('conditions', [])
                )
            )
        ]
        if redis_pods_comp_status:
            LOG.info(f"Current list of rfs-openstack-redis pods in Completed status is:"
                     f" {[p['metadata']['name'] for p in redis_pods_comp_status]}")
            for pod in redis_pods_comp_status:
                try:
                    LOG.info(f"Deleting pod {pod['metadata']['name']}")
                    self.api.api_core.delete_namespaced_pod(
                        name=pod['metadata']['name'],
                        namespace=self.redis_namespace,
                        grace_period_seconds=0)
                except Exception as e:
                    LOG.info(e)
                    if e.status == 404:
                        LOG.info("Skip deleting due to one of pods not found")
                    else:
                        LOG.info(e)

            redis_pods_comp_status_after_deletion = [
                p for p in self.api.pods.list_pods_except_jobs(target_namespaces=self.redis_namespace)
                if (
                    p['status']['phase'] == "Succeeded"
                    and 'rfs-openstack-redis' in p['metadata']['name']
                    and any(
                        cond.get('message') and 'Pod was terminated in response '
                                                'to imminent node shutdown.' in cond['message']
                        for cond in p['status'].get('conditions', [])
                    )
                )
            ]
            assert len(redis_pods_comp_status_after_deletion) == 0, f'After pod deletion ' \
                                                                    f'still some pods still are in ' \
                                                                    f'Completed status: ' \
                                                                    f'{redis_pods_comp_status_after_deletion}'
        else:
            LOG.info("Current list of rfs-openstack-redis pods in Completed status is EMPTY, nothing to do here.")

    def wr_field_6500(self):
        # TODO(tleontovich) Delete after https://mirantis.jira.com/browse/FIELD-6500 fixed
        libvirt_pods_term_status = [
            p for p in self.api.pods.list_pods_except_jobs(target_namespaces=self.openstack_namespace)
            if p['metadata']['deletion_timestamp'] and 'libvirt' in p['metadata']['name']]
        if libvirt_pods_term_status:
            LOG.info(f"Current  list of libvirt service pods in terminating status is:"
                     f" {[p['metadata']['name'] for p in libvirt_pods_term_status]}")
            for pod in libvirt_pods_term_status:
                try:
                    LOG.info(f"Deleting pod {pod['metadata']['name']}")
                    self.api.api_core.delete_namespaced_pod(
                        name=pod['metadata']['name'],
                        namespace=self.openstack_namespace,
                        grace_period_seconds=0)
                except Exception as e:
                    LOG.info(e)
                    if e.status == 404:
                        LOG.info("Skip deleting due to one of pods not found")
                    else:
                        LOG.info(e)
        self.wait_all_os_pods()
        pod_list = self.api.pods.list_pods_except_jobs(target_namespaces=self.openstack_namespace)
        libvirt_pods_term_status_after_deletion = []
        for p in pod_list:
            if 'libvirt' in p['metadata']['name']:
                if p['metadata']['deletion_timestamp']:
                    libvirt_pods_term_status_after_deletion.append(p)
        assert len(libvirt_pods_term_status_after_deletion) == 0, f'After pod deletion ' \
                                                                  f'still some pods still are in ' \
                                                                  f'Terminating status: ' \
                                                                  f'{libvirt_pods_term_status_after_deletion}'

    def wa_prodx_31186(self, mariadb_restarts, max_restarts=3):
        """Workaround for PRODX-31186: try to restore mariadb-server in case of corrupted buffer error

        1. Ensure that only one mariadb-server replica is not ready
        2. Ensure that mariadb-server restarts count is growing
        3. Ensure that a faultly mariadb-server pod contains "Corrupt buffer" in log
        4. Apply workaround: remove galera.cache, restart the mariadb-server pod, and clear restarts count
        """

        mariadb_pods = [pod for pod in self.get_os_pods() if 'mariadb-server' in pod.name]
        if not mariadb_pods:
            # Nothing to check
            LOG.debug("mariadb-server pods not found, looks like some other error, skipping WA")
            return

        # 1. Ensure that only one replica is not ready, before trying to check the condition of a bad replica
        ready_pods = sum([mariadb_pod.are_containers_ready()
                          for mariadb_pod in mariadb_pods
                          if mariadb_pod.read().status.phase == 'Running'])
        if ready_pods < (len(mariadb_pods) - 1):
            LOG.debug(f"Expected at least N-1 ready replicas for 'mariadb-server' pods, "
                      f"but got only {ready_pods} ready replicas. WA for PRODX-31186 is unsafe, skipping")
            return

        # Check replicas and ensure that this is the condition where WA should be applied
        for mariadb_pod in mariadb_pods:

            # 2. Ensure that mariadb-server restarts count is growing
            restarts = mariadb_pod.get_restarts_number()
            if not restarts:
                # no restarts, skip checking this container
                continue
            if mariadb_pod.name not in mariadb_restarts.keys():
                mariadb_restarts[mariadb_pod.name] = restarts
            if restarts < (mariadb_restarts[mariadb_pod.name] + max_restarts):
                # restarts still less than expected, skip checking this container
                continue

            # 3. Ensure that a faultly mariadb-server pod contains expected message in the log
            try:
                logs = mariadb_pod.get_logs(container='mariadb')
            except Exception:
                logs = ''
            if '[ERROR] WSREP: Corrupt buffer header' not in logs:
                # looks like restarts caused not because of PRODX-31186, skip workaround
                continue

            # 4. Apply workaround
            self._apply_wa_prodx_31186(mariadb_pod)
            # re-start checking this pod
            mariadb_restarts.pop(mariadb_pod.name)

    def _apply_wa_prodx_31186(self, mariadb_pod):
        """Remove galera.cache from the volume attached to specified mariadb pod, and delete the pod to restart"""

        LOG.banner(f"Apply workaround for PRODX-31186 for pod {mariadb_pod.namespace}/{mariadb_pod.name}")

        node_name = mariadb_pod.data['spec'].get('node_name')
        if not node_name:
            # Pod is not scheduled yet to any node
            return

        pvc_names = mariadb_pod.get_pvc_names()
        mariadb_pvc_name = pvc_names.get('mysql-data')
        if not mariadb_pvc_name:
            LOG.error(f"PVC claim name for volume 'mysql-data' not found "
                      f"in POD {mariadb_pod.namespace}/{mariadb_pod.name}")
            return

        # Get local path of the mariadb data on the node
        pvc = self.api.pvolumeclaims.get(name=mariadb_pvc_name, namespace=self.openstack_namespace)
        pvc_volume_name = pvc.data['spec']['volume_name']
        pv = self.api.pvolumes.get(name=pvc_volume_name)
        path = pv.data['spec']['local']['path']

        registry = self.api.pods.extract_registry_url(pod_name=mariadb_pod.name, pod_ns=mariadb_pod.namespace)
        cmd = f"rm -f {path}/galera.cache"
        LOG.info(f"[{node_name}] Execute command '{cmd}'")
        res = self.api.pods.exec_pod_cmd(cmd=cmd, registry=registry, node_name=node_name, verbose=False)
        if res['exit_code'] != 0:
            LOG.error("Error deleting file 'galera.cache', skip restarting the pod")
            return
        LOG.info(f"Delete pod {mariadb_pod.namespace}/{mariadb_pod.name}")
        mariadb_pod.delete()

    def wait_all_os_pods(self, timeout=300, interval=60, exclude_node_shutdown=False):
        def _wait_pods():
            _pods = self.api.pods.list(namespace=self.openstack_namespace)
            pods = []
            for p in _pods:
                try:
                    p_data = p.read()
                    pods.append(p_data)
                except ApiException:
                    continue
            inactive = [
                p
                for p in pods
                if p.status.phase.lower() not in ("running", "succeeded")
            ]
            # filter failed pods owned by jobs, as
            # some jobs can have restartPolicy Never
            inactive_job_pods = [
                pod for pod in inactive if self._is_owner_of_kind(pod, "Job")
            ]

            for pod in inactive_job_pods:
                inactive.remove(pod)

            if exclude_node_shutdown:
                # filter pods in Error state after node shutdown
                node_shutdown_pods = []
                for pod in inactive:
                    phase = pod.status.phase
                    reason = pod.status.reason
                    message = pod.status.message

                    is_failed_by_node_shutdown = (
                            phase == 'Failed' and
                            reason == 'Terminated'
                            and 'Pod was terminated in response to imminent node shutdown.' in message
                    )

                    if is_failed_by_node_shutdown:
                        node_shutdown_pods.append(pod)

                for pod in node_shutdown_pods:
                    inactive.remove(pod)

            def _pod_mapper(pod):
                return pod.metadata.name, pod.status.phase

            LOG.info(
                "Namespace has inactive service %s pods and job %s pods",
                list(map(_pod_mapper, inactive)),
                list(map(_pod_mapper, inactive_job_pods)),
            )

            assert (
                not inactive
            ), "Some pods {} wasn't active/finished in " "proper time {}".format(
                pods, timeout
            )

        LOG.info("Wait for Openstack pods readiness")
        waiters.wait_pass(_wait_pods, AssertionError, interval, timeout)
        LOG.info("Openstack pods are ready")

    def wait_all_os_jobs(self, timeout=300, interval=60):
        def _wait_jobs():
            jobs = self.api.jobs.list(namespace=self.openstack_namespace)
            not_cron = []
            for job in jobs:
                try:
                    if not self._is_owner_of_kind(job.read(), "CronJob"):
                        not_cron.append(job)
                except ApiException:
                    continue

            not_succeeded = []
            for job in not_cron:
                try:
                    if not job.read().status.succeeded == 1:
                        not_succeeded.append(job.name)
                except ApiException:
                    continue
            assert not not_succeeded, "Not all jobs succeeded {}".format(not_succeeded)

        LOG.info("Wait for Openstack jobs conditions")
        waiters.wait_pass(_wait_jobs, AssertionError, interval, timeout)
        LOG.info("Openstack jobs are in expected conditions")

    @staticmethod
    def get_k8s_object_field(status, field):
        return getattr(status, field) or 0

    def wait_k8s_object_replicas_restored(self, k8s_object, k8s_object_type, replicas):
        """
        Wait for k8s object replicas restored to needed quantity.

        :param k8s_object: k8s object that has read and
            patch_scale methods

        :param k8s_object_type: tuple of k8s_object.name
            and k8s_object.resource_type

        :param replicas: number of desired replicas
        """
        get_field = self.get_k8s_object_field

        def check():
            status = k8s_object.read().status
            target_field = "current_replicas"
            if k8s_object_type[1] == "deployment":
                target_field = "available_replicas"
            return (
                get_field(status, "replicas")
                == get_field(status, target_field)
                == get_field(status, "ready_replicas")
                == replicas
            )

        LOG.info(
            "Wait for %s of type %s replicas successfully reached" " required value",
            *k8s_object_type,
        )
        waiters.wait(
            check,
            interval=30,
            timeout=settings.OPENSTACK_LCM_OPERATIONS_TIMEOUT,
        )

    def change_k8s_object_replicas_count(self, k8s_object, k8s_object_type, replicas):
        """
        Change replicas count for k8s_object.
        Verify that replicas are successfully changed.

        :param k8s_object: k8s object that has read and
            patch_scale methods

        :param k8s_object_type: tuple of k8s_object.name
            and k8s_object.resource_type

        :param replicas: number of desired replicas
        """
        if (
            self.get_k8s_object_field(k8s_object.read().status, "ready_replicas")
            == replicas
        ):
            LOG.info(
                "%s of type %s already has " "required number of replicas",
                *k8s_object_type,
            )
            return

        LOG.info("Set %s of type %s replicas to %s", *k8s_object_type, replicas)

        k8s_object.patch_scale(
            [{"op": "replace", "path": "/spec/replicas", "value": replicas}]
        )
        self.wait_k8s_object_replicas_restored(k8s_object, k8s_object_type, replicas)

    def change_and_restore_replicas_count(
        self, k8s_object, replicas, deployment_replicas
    ):
        """
        Change replicas of k8s object to some number
        and then restore to original count.

        :param k8s_object: k8s_object: k8s object that has read and
            patch_scale methods

        :param replicas: number of desired replicas

        :param deployment_replicas: original number of replicas that
            should be restored
        """
        k8s_object_type = (k8s_object.name, k8s_object.resource_type)
        LOG.info("Change %s of type %s replicas to %s", *k8s_object_type, replicas)
        self.change_k8s_object_replicas_count(k8s_object, k8s_object_type, replicas)

        LOG.info(
            "Restore %s of type %s replicas to %s",
            *k8s_object_type,
            deployment_replicas,
        )
        self.change_k8s_object_replicas_count(
            k8s_object, k8s_object_type, deployment_replicas
        )

    def get_pods_to_delete(self, k8s_object, num_pods_to_delete):
        """
        Get all pods with k8s object name as prefix, and then select
        n of them.

        :param k8s_object: k8s_object: k8s object that has read and
            patch_scale methods

        :param num_pods_to_delete: number of pods that should be deleted

        :return: list of randomly selected pods
        """
        LOG.info("Get all %s pods", k8s_object.name)
        pods = self.api.pods.list_starts_with(k8s_object.name, self.openstack_namespace)

        LOG.info("Select pods to delete from %s", pods)
        pods_to_delete = random.sample(pods, num_pods_to_delete)
        LOG.info("Selected pods are %s", pods_to_delete)
        return pods_to_delete

    def wait_pods_are_deleted_and_then_restored(
        self, k8s_object, k8s_object_type, deployment_replicas
    ):
        """
        Wait till at least one of the  pod is deleted and then wait for
        replicas count restores to deployment quantity.

        :param k8s_object: k8s_object: k8s object that has read and
            patch_scale methods

        :param k8s_object_type: tuple of k8s_object.name
            and k8s_object.resource_type

        :param deployment_replicas: original number of replicas that
            should be restored
        """
        get_field = self.get_k8s_object_field

        def check():
            replicas = get_field(k8s_object.read().status, "ready_replicas")
            return replicas < deployment_replicas

        LOG.info("Wait for at least one pod is deleted")
        waiters.wait(check, interval=5, timeout=180)

        self.wait_k8s_object_replicas_restored(
            k8s_object, k8s_object_type, deployment_replicas
        )

    def delete_pod_and_wait_for_rescale(
        self, k8s_object, num_pods_to_delete, deployment_replicas
    ):
        """
        Select pods from list of pods, delete them and then wait till
        k8s restore original quantity of replicas.

        :param k8s_object: k8s_object: k8s object that has read and
            patch_scale methods

        :param num_pods_to_delete: number of pods that should be deleted

        :param deployment_replicas: original number of replicas that
            should be restored
        """
        k8s_object_type = (k8s_object.name, k8s_object.resource_type)

        pods_to_delete = self.get_pods_to_delete(k8s_object, num_pods_to_delete)

        for pod in pods_to_delete:
            LOG.info("Delete pod %s", pod)
            pod.delete(async_del=True)

        self.wait_pods_are_deleted_and_then_restored(
            k8s_object, k8s_object_type, deployment_replicas
        )

    def delete_force_pod_and_wait_for_rescale(
        self,
        k8s_object,
        num_pods_to_delete,
        deployment_replicas,
        bash_command,
        container=None,
    ):
        """
        Select pods from list of pods, delete them and then wait till
        k8s restore original quantity of replicas.

        :param k8s_object: k8s_object: k8s object that has read and
            patch_scale methods

        :param num_pods_to_delete: number of pods that should be deleted

        :param deployment_replicas: original number of replicas that
            should be restored

        :param bash_command: bash command that should be executed to
        delete pod

        :param container: container where bash command should be executed.
         optional
        """
        kwargs = {}
        if container:
            kwargs["container"] = container

        k8s_object_type = (k8s_object.name, k8s_object.resource_type)

        pods_to_delete = self.get_pods_to_delete(k8s_object, num_pods_to_delete)

        for pod in pods_to_delete:
            LOG.info("Delete pod %s with command %s", pod, bash_command)
            resp = pod.exec(bash_command, tty=True, **kwargs)
            LOG.info("Response: %s", resp)

        self.wait_pods_are_deleted_and_then_restored(
            k8s_object, k8s_object_type, deployment_replicas
        )

    def get_db_password(self):
        """
        Returns the password for Maria DB.

        :returns: password
        :rtype: str
        """
        LOG.info("Getting DB password")

        secret = self.get_os_resource(
            "secrets", "mariadb-dbadmin-password", self.openstack_namespace, read=True
        )

        encoded_data = secret.data.get("MYSQL_DBADMIN_PASSWORD")

        if not encoded_data:
            raise ValueError('Invalid value for "db_password", must not be "None"')

        db_password = base64.b64decode(encoded_data).decode("utf-8")

        LOG.info("DB password was taken")

        return db_password

    @cached_property
    def db_password(self):
        return self.get_db_password()

    def get_messaging_password(self):
        """
        Returns admin's password for Rabbitmq for specific service

        :returns: password
        :rtype: str
        """
        messaging_password = None
        LOG.info("Getting messaging password")
        secret = self.get_os_resource(
            "secrets",
            "openstack-rabbitmq-admin-user",
            self.openstack_namespace,
            read=True,
        )
        encoded_data = secret.data.get("RABBITMQ_ADMIN_PASSWORD")
        if not encoded_data:
            raise ValueError('Invalid value for messaging password, must not be "None"')
        messaging_password = base64.b64decode(encoded_data).decode("utf-8")
        LOG.info("Messaging password was taken")
        return messaging_password

    @cached_property
    def messaging_password(self):
        return self.get_messaging_password()

    def get_openstack_endpoints(self, interface=None):
        endpoints_cmd = ['/bin/sh', '-c', 'PYTHONWARNINGS=ignore::UserWarning openstack endpoint list -f yaml']
        data = yaml.safe_load(self.keystone_client_pod.exec(endpoints_cmd))
        if not interface:
            return data
        return [endpoint for endpoint in data if endpoint.get('Interface') == interface]

    def get_service_endpoint(self, service_name, interface='public'):
        endpoints = self.get_openstack_endpoints(interface=interface)
        service_endpoint = [e for e in endpoints if e.get('Service Name') == service_name]
        if not service_endpoint:
            msg = f"No {interface} endpoint found for service {service_name}"
            raise ValueError(msg)
        return service_endpoint[0]

    def create_openstack_admin_user(self, username=None, password=None, recreate_if_exists=True,
                                    project='admin', domain='default'):
        username = username or settings.SI_LOADTOOLS_OS_ADMIN_USERNAME
        password = password or settings.SI_LOADTOOLS_OS_ADMIN_PASSWORD
        user_create_cmd = ['/bin/sh', '-c', f'PYTHONWARNINGS=ignore::UserWarning '
                                            f'openstack user create --domain {domain}'
                                            f' --password {password} {username}']
        add_roles_cmd = [
            '/bin/sh', '-c', f'PYTHONWARNINGS=ignore::UserWarning '
                             f'openstack role add --project {project} --user {username} admin']
        try:
            LOG.info(f"Creating user {username}")
            res = self.keystone_client_pod.exec(user_create_cmd)
        except Exception as e:
            LOG.error(e)
            raise ApiException(f"Failed to create user {username}")
        if 'Duplicate entry found' in res:
            LOG.warning(f"User {username} already exists")
            LOG.info(f"Check password is correct for user {username}")
            check_cmd = ['/bin/sh', '-c', f'export OS_USERNAME={username};export OS_PASSWORD={password};'
                                          f'PYTHONWARNINGS=ignore::UserWarning openstack project list']
            res = self.keystone_client_pod.exec(check_cmd)
            if '(HTTP 401)' in res:
                LOG.info(res)
                LOG.info(f"Password incorrect for user {username}")
                if recreate_if_exists:
                    LOG.info(f"Recreating user {username}")
                    user_delete_cmd = [
                        '/bin/sh', '-c', f'PYTHONWARNINGS=ignore::UserWarning openstack '
                                         f'user delete {username}']
                    self.keystone_client_pod.exec(user_delete_cmd)
                    self.keystone_client_pod.exec(user_create_cmd)
                else:
                    msg = f"User {username} already exists but has another password. Will not be recreated"
                    raise ValueError(msg)
            else:
                LOG.info(f"Password is OK for user {username}")
        try:
            LOG.info(f"Add admin role for user {username}")
            self.keystone_client_pod.exec(add_roles_cmd)
        except Exception as e:
            LOG.error(e)
            raise ApiException(f"Failed to add role admin to user {username} in project {project}")

    def run_mysql_command(self, cmd, **kwargs):
        """
        Executes mysql command on Mariadb-server pod

        :returns: result of mysql command execution
        :rtype: str
        """
        command = ["/bin/sh", "-c", cmd]
        mariadb_pod = self.get_os_pod("mariadb-server-0", read=False)
        response = mariadb_pod.exec(command=command, container="mariadb", **kwargs)

        return response

    def run_bash_command(self, pod_name, cmd, container=None):
        """
        Executes bash command on the specified pod

        :returns: result of bash command execution
        :rtype: str
        """
        command = ["/bin/bash", "-c", cmd]
        pod = self.get_os_pod(pod_name, read=False)
        response = pod.exec(command, container=container)

        return response

    def restart_cloudprober(self):
        cloudprober = self.get_os_deployment("openstack-cloudprober")
        # Remove all found pods and wait for number of desired replicas
        pods = self.api.pods.list_starts_with(
            cloudprober.name, self.openstack_namespace
        )
        for pod in pods:
            LOG.info("Deleting cloudprober pod %s", pod)
            pod.delete(async_del=True)
        self.wait_pods_are_deleted_and_then_restored(
            cloudprober, (cloudprober.name, cloudprober.resource_type), cloudprober.desired_replicas
        )

    def get_cloudprober_start_ts(self):
        """
        Returns start timestamp of the first started pod of
        cloudprober deployment

        :returns: timestamp in iso format
        :rtype: str
        """
        cloudprober = self.get_os_deployment("openstack-cloudprober")
        pods = self.api.pods.list_starts_with(
            cloudprober.name, self.openstack_namespace
        )
        start_ts = []
        for pod in pods:
            start_ts.append(pod.read().status.start_time)
        return sorted(start_ts)[0].isoformat()

    def collect_sos_report_logs(self):

        LOG.info("Get OS controller image name")
        os_controller = self.get_os_operator_deployment()
        os_controller_image = os_controller.read().spec.template.spec.containers[0].image

        LOG.info("Try to deploy openstack sos report container")
        pod = None
        try:
            options = {
                'OPENSTACK_CONTROLLER_SOS_REPORT_IMAGE': os_controller_image,
                'OPENSTACK_CONTROLLER_SOS_REPORT_NAMESPACE': self.os_helm_system_namespace,
                'OPENSTACK_CONTROLLER_SOS_REPORT_TIMEOUT': settings.OPENSTACK_CONTROLLER_SOS_REPORT_TIMEOUT,
                'oc_name': self.oc_name,
            }
            templates = templates_utils.render_template(settings.OPENSTACK_CONTROLLER_SOS_REPORT_POD_TEMPLATE, options)
            LOG.debug(templates)
            json_body = json.dumps(yaml.load(templates, Loader=yaml.SafeLoader))
            pod = self.api.pods.create(
                name="openstack-controller-sos-report",
                namespace=self.os_helm_system_namespace,
                body=json.loads(json_body))
            LOG.info("Wait for pod to be Running...")
            pod.wait_phase('Running')
            LOG.info("Try to collect logs")
            pod.wait_test(filepath='./collect_*',
                          timeout=settings.OPENSTACK_CONTROLLER_SOS_REPORT_TIMEOUT)
            LOG.info("Copy artifacts")
            pod.cp_from_pod(source_dir="/sosreport")
        finally:
            if pod is not None:
                pod.delete()

    @property
    def mysql_binary(self):
        return self.run_mysql_command('ps ax | grep -v grep | grep -owE "mariadbd|mysqld"',).strip()
