import json
import yaml

import si_tests.utils.templates as template_utils
from si_tests import logger
from si_tests import settings
from si_tests.utils import waiters, exceptions, bootstrapv2
from kubernetes.client.rest import ApiException

from typing import TYPE_CHECKING
if TYPE_CHECKING:
    from si_tests.managers.kaas_manager import Manager

LOG = logger.logger


class BoostrapCheckMgr(object):
    """Shortened and adopted check class from ClusterCheck suitable for bv2 cluster w/out cluster object as is"""

    def __init__(self, kind_manager: "Manager"):
        self.__manager = kind_manager
        self.__k8sclient = kind_manager.api

    @property
    def manager(self):
        return self.__manager

    @property
    def k8sclient(self):
        return self.__k8sclient

    @property
    def raw_bootstrap_version(self):
        """
        Get bootstrapversion from kaasrelease. Assuming that there is only one kaasrelease
        in clean and freshly deployed bootstrapv2 cluster
        :return:
        """
        kaasrelease = self.manager.get_active_kaasrelease()
        return kaasrelease.data.get('spec', {}).get('bootstrap', {}).get('version')

    @property
    def clean_kaasrelease_version(self):
        """
        Get clean kaasrelease version (e.g assuming 2.23.0-rc as 2.23.0) provided with bootstrap cluster
        :return:
        """
        kaasrelease = self.manager.get_active_kaasrelease()
        kr_name = kaasrelease.name
        return bootstrapv2.extract_clean_kaasrelease_version(kr_name)

    def any_bootstrapregion_present(self):
        brs = self.manager.list_all_bootstrapregions()
        return len(brs) != 0

    def br_provider_type(self):
        brs = self.manager.list_all_bootstrapregions()
        if len(brs) != 0:
            # Assuming that only one br can be present at bootstrap cluster at one time
            br = brs[0]
            LOG.info(f"Found Bootstrapregion {br.namespace}/{br.name} with provider {br.provider}")
            return br.provider
        else:
            return None

    def __get_desired_replica_number(self, obj, namespace):
        """Return desired number of replicas/scheduled pods according to
           the pod owner specification (e.g. statefulset).
        """
        kind, name = obj.split('/')
        if kind == 'ReplicaSet':
            rs = self.k8sclient.replicasets.get(name=name, namespace=namespace)
            desired_num = rs.desired_replicas
        elif kind == 'StatefulSet':
            ss = self.k8sclient.statefulsets.get(name=name, namespace=namespace)
            desired_num = ss.desired_replicas
        elif kind == 'DaemonSet':
            ds = self.k8sclient.daemonsets.get(name=name, namespace=namespace)
            desired_num = ds.desired_replicas
        elif kind == 'Deployment':
            dp = self.k8sclient.deployments.get(name=name, namespace=namespace)
            desired_num = dp.desired_replicas
        else:
            LOG.warning(f"Unknown kind {kind}")
            desired_num = None
        return desired_num

    def _render_bootstrapv2_expected_pods_template(self, bv2_template='bootstrapv2'):
        br_provider_type = self.br_provider_type()

        any_bootstrapregion_present = self.any_bootstrapregion_present()
        # valid provider types are: openstack, azure, aws, vsphere, equinixmetalv2, baremetal, None
        options = {
            'br_provider_type': br_provider_type,
            'any_bootstrapregion_present': any_bootstrapregion_present
        }
        templates = template_utils.render_template(
            settings.EXPECTED_PODS_TEMPLATES_DIR +
            bv2_template +
            '.yaml', options
        )

        json_body = yaml.load(templates, Loader=yaml.SafeLoader)
        return json_body

    def get_expected_bootstrapv2_objects(self):
        """Returns list of expected pods (and their quantity for the cluster)
           plus all docker objects
        """
        tmpl_name = f"bootstrapv2-{self.clean_kaasrelease_version.replace('.','-')}"
        LOG.info(f"Active kaasrelease is {self.manager.get_active_kaasrelease()}. \n"
                 f"Template for checking: {tmpl_name}.yaml")

        json_body = self._render_bootstrapv2_expected_pods_template(bv2_template=tmpl_name)
        all_pods = {}

        cluster_type = 'bootstrap'

        for ns in json_body[cluster_type]:
            pods = json_body[cluster_type][ns]
            pods = {k: v for k, v in pods.items()}
            if ns not in all_pods:
                all_pods[ns] = pods
            else:
                all_pods[ns].update(pods)

        LOG.info(json.dumps(all_pods, indent=4))
        return all_pods

    # Almost full copy of checker inside CLusterCheck
    def check_actual_expected_pods_in_boostrap(self,
                                               expected_pods=None,
                                               timeout=300,
                                               interval=10,
                                               check_all_nss=False,
                                               exclude_jobs=True):
        """Compare expected list of pods (which is fetched automatically,
           unless explicitly provided) with actual list of pods in this
           cluster. Comparison is conducted for all namespaces by default"""

        LOG.info("Checking that all pods and their replicas are in place")

        def compare(expected_pods_dict, exclude_jobs):
            failed = {}
            namespaces = expected_pods_dict.keys()
            actual_list = [pod for pod in
                           self.k8sclient.pods.list_raw().to_dict()['items']]

            if exclude_jobs:
                before_filtering = \
                    set([x['metadata']['name'] for x in actual_list])
                actual_list = \
                    [x for x in actual_list if not (
                            x['metadata']['owner_references'] and
                            x['metadata'][
                                'owner_references'][0]['kind'] == 'Job')]
                after_filtering = \
                    set([x['metadata']['name'] for x in actual_list])
                LOG.debug(f"These pods are jobs and will be filtered out: "
                          f"{before_filtering - after_filtering}")

            other_ns_pods = [pod for pod in actual_list
                             if pod['metadata']['namespace'] not in namespaces]
            actual_list = [pod for pod in actual_list
                           if pod['metadata']['namespace'] in namespaces]

            not_checked = [pod['metadata']['name'] for pod in actual_list]

            # sort names of expected pods so that
            # long names come first. "/no_owner" tag is excluded
            for ns in namespaces:
                expected_pods_lst = sorted(
                    list(expected_pods_dict[ns].keys()),
                    key=lambda x: len(x.split("/")[0]), reverse=True
                )
                for pod in expected_pods_lst:
                    desired_num = expected_pods_dict[ns][pod]
                    prefix = pod.split("/")[0]
                    compare_list = \
                        [pod for pod in actual_list
                         if pod['metadata']['name'].startswith(prefix) and
                         pod['metadata']['name'] in not_checked]

                    if not compare_list:
                        failed[prefix] = {"actual": 0,
                                          "desired/expected": desired_num}
                        continue

                    if '/no_owner' in pod:
                        # for pods that are marked "no_owner" we do not fetch
                        # owner and use # of pods from the file
                        LOG.debug(f"Number of pods "
                                  f"for {prefix} group will be checked "
                                  f"according to expected pods list")
                    else:
                        # get owner kind and name for the first pod in list
                        first_pod = compare_list[0]
                        owner_references = \
                            first_pod['metadata']['owner_references']
                        kind_name = f"{owner_references[0]['kind']}/" \
                                    f"{owner_references[0]['name']}"

                        if kind_name.split('/')[0] == "Node":
                            # Starting from 1.17 mirror pods have
                            # Node/node-name in owner_references
                            LOG.warning(f"No replica count info for pod "
                                        f"{first_pod['metadata']['name']}. "
                                        f"Using expected number ({desired_num}) "
                                        f"from the list")
                            # Better to handle this type of pods as '/no_owner'
                            LOG.info("Please add '/no_owner' for this group of"
                                     " pods to expected pods list file")
                        else:
                            try:
                                replicas_num = self.__get_desired_replica_number(
                                    kind_name,
                                    first_pod['metadata']['namespace']
                                )
                                LOG.debug(f"First pod is "
                                          f"{first_pod['metadata']['name']}, "
                                          f"owner: {kind_name}, "
                                          f"replica #: {replicas_num}")
                                if int(replicas_num) != int(desired_num):
                                    if kind_name.startswith("DaemonSet"):
                                        LOG.warning(f"Replicas num ({replicas_num}) from {kind_name} "
                                                    f"is not equal to the "
                                                    f"number ({desired_num}) in expected "
                                                    f"pod list. Pod: {prefix} . "
                                                    f"Assuming DaemonSet replicas as expected pods number")
                                        desired_num = int(replicas_num)
                                    else:
                                        LOG.warning(f"Replicas num ({replicas_num}) from {kind_name} "
                                                    f"is not equal to the "
                                                    f"number ({desired_num}) in expected "
                                                    f"pod list. Pod: {prefix}")
                                    # PRODX-30560, DaemonSets may have different replicas
                                    # on different labs because of node taints or labels.
                                    # So do not fail the check until found a better solution.
                                    # failed[prefix] = {"object replicas": replicas_num,
                                    #                   "desired/expected": desired_num}
                            except (Exception, ApiException) as e:
                                LOG.error(e)
                                LOG.error(f"Cannot process {prefix} "
                                          f"group of pods. Skipping")
                                failed[prefix] = {"actual": 0,
                                                  "desired/expected": desired_num}
                                continue
                    if int(desired_num) != len(compare_list):
                        failed[prefix] = {"actual": len(compare_list),
                                          "desired/expected": desired_num}
                    not_checked = [x for x in not_checked
                                   if x not in [pod['metadata']['name']
                                                for pod in compare_list]]

                    actual_ns = set([pod['metadata']['namespace']
                                     for pod in compare_list])
                    if set([ns]) != actual_ns:
                        failed[prefix] = {"actual namespace": actual_ns,
                                          "desired/expected": ns}

            if other_ns_pods:
                for pod in other_ns_pods:
                    LOG.error(f"Extra pod {pod['metadata']['name']} "
                              f"found in {pod['metadata']['namespace']} "
                              f"namespace")
                    if check_all_nss:
                        not_checked.append(pod['metadata']['name'])

            if failed or not_checked:
                result = {"Pods mismatch": failed,
                          "Not checked pods": not_checked}
                LOG.warning(f"Compare pod check failed: {result}")
                return result

        if not expected_pods:
            expected_pods = self.get_expected_bootstrapv2_objects()

        try:
            waiters.wait(lambda: not compare(
                expected_pods_dict=expected_pods, exclude_jobs=exclude_jobs),
                         timeout=timeout, interval=interval)
        except exceptions.TimeoutError:
            result = compare(expected_pods_dict=expected_pods, exclude_jobs=exclude_jobs)
            if result:
                err = f"Timeout waiting for pods. " \
                      f"After {timeout}s there are some fails: " \
                      f"{result}"
                raise TimeoutError(err)
        LOG.info("All pods and their replicas are found")
