#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import os

import pytest
from playwright.sync_api import sync_playwright

from si_tests import logger
from si_tests import settings
from si_tests.managers.kaas_manager import Cluster
from si_tests.utils import utils

LOG = logger.logger


def create_session(p, url, operator_password):
    # Channel can be "chrome", "msedge", "chrome-beta", "msedge-beta" or "msedge-dev".
    browser = p.chromium.launch(headless=True)
    context = browser.new_context(ignore_https_errors=True)
    page = context.new_page()
    page.goto(url)

    page.get_by_role("textbox", name="Username or email").click()
    page.get_by_role("textbox", name="Username or email").fill("operator")
    page.get_by_role("textbox", name="Password").click()
    page.get_by_role("textbox", name="Password").fill(operator_password)
    page.get_by_role("button", name="Sign In").click()
    page.get_by_role("button", name="Close Tour").click()
    return context


@pytest.fixture(scope='module')
def login_browser(kaas_manager):
    """Login into cluster UI for first time

    Scenario:
        1. Login into UI cluster
        2. Save login data into storage
    """
    # Collect user info from cluster
    mgmt_cluster: Cluster = kaas_manager.get_mgmt_cluster()
    d_client = mgmt_cluster.get_mcc_dashboardclient(user="operator")
    assert d_client, "No suitable UI user (operator role) found"
    ui_url = d_client.httpclient.base_url
    operator_password = d_client.password

    if os.path.isfile(os.path.join(settings.UI_LOGIN_STATE_FILE_NAME)):
        os.remove(os.path.join(settings.UI_LOGIN_STATE_FILE_NAME))

    with sync_playwright() as p:
        context = create_session(p, ui_url, operator_password)
        assert context, "Browser context with login information wasn't created, check Credentials and UI URL"
        # Store browser context in file for future tests
        context.storage_state(path=os.path.join(settings.UI_LOGIN_STATE_FILE_NAME))
        context.tracing.start(screenshots=True, snapshots=True, sources=True)

        yield ui_url, context

        context.close()


@pytest.fixture(scope='function')
def traced_page(login_browser, request):
    """Go to Ui URL and start writing trace
    """
    ui_url, context = login_browser
    # Start tracing before creating / navigating a page.
    LOG.info(f"Start recording trace for test: {request.param}")
    context.tracing.start_chunk()
    page = context.new_page()
    page.goto(ui_url)

    yield page

    # Stop tracing and export it into a zip archive.
    context.tracing.stop_chunk(path=os.path.join(
        settings.ARTIFACTS_DIR, f"trace_{request.param}.zip"))


@pytest.fixture(scope='module')
def ui_child_data(kaas_manager):
    """Collect, process and return UI child data

    Scenario:
        1. Collect and process child_data
    """
    # Collect user info from cluster
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    release_name = kaas_manager.si_config.version_generators.get_child_cluster_release_name()
    LOG.info("Release name: {}".format(release_name))
    LOG.info("Available releases: {}".format(kaas_manager.get_clusterrelease_names()))
    assert release_name in kaas_manager.get_clusterrelease_names()
    allowed_distros = kaas_manager.get_allowed_distributions(release_name)
    mgmt_cluster = kaas_manager.get_mgmt_cluster()
    mgmt_version = mgmt_cluster.spec['providerSpec']['value']['kaas']['release']
    region = mgmt_cluster.region_name
    # Get pub_key from job
    with open(settings.KAAS_CHILD_CLUSTER_PUBLIC_KEY_FILE) as pub_key:
        pub_key_content = pub_key.read()
    render_opts = {
        "mgmt_version": mgmt_version,
        "target_region": region,
        "target_cluster": cluster_name,
        "target_namespace": namespace_name,
        "machine_distribution": allowed_distros.get(settings.DISTRIBUTION_RELEVANCE, {}).get('id', ''),
        "pub_key_content": pub_key_content,
    }
    ui_child_data = utils.render_child_data(kaas_manager.si_config, render_opts)
    return ui_child_data
