#    Copyright 2023 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations

import os.path
import time
from abc import abstractmethod
from datetime import datetime, timezone

from cached_property import cached_property

from si_tests import logger
from si_tests import settings
from si_tests.fixtures.workload import store_statistics
from si_tests.managers.openstack_manager import OpenStackManager
from si_tests.managers.openstack_client_manager import OpenStackClientManager
from si_tests.clients.grafana.grafana_client import GrafanaClient

LOG = logger.logger

# ISO 8601
TIME_FORMAT = '%Y-%m-%dT%H:%M:%S.%f'


def calculate_workload_downtime(data: dict, strip=0):
    """
    Calculate potential downtime period based on Prometheus data

    Args:
        data: Prometheus metric timeseries data in format:
         {
            "metric" : {
               "job" : "<job_name>",
               "instance" : "<instance_name>",
               "<label>" : "<value>",
            },
            "values" : [
               [ 1435781430, "1" ],
               [ 1435781445, "1" ],
               [ 1435781460, "0.5" ]
            ]
         },
         strip: number of elements to drop from both ends
    Returns: Dict with downtime data, in format:
        { "duration_total": 5,
          "periods_num": 1,
          "periods": [
               {'start': '2023-05-17 06:01:58',
                'end': '2023-05-17 06:02:03',
                'duration': 5
               },
           ]
        }
    """

    def _add_downtime(res, window):
        start_dt = datetime.fromtimestamp(window['start'], tz=timezone.utc)
        end_dt = datetime.fromtimestamp(window['end'], tz=timezone.utc)
        duration = int((end_dt - start_dt).total_seconds())
        res["duration_total"] += duration
        res["periods_num"] += 1
        res["periods"].append({
            "start": start_dt.strftime(TIME_FORMAT),
            "end": end_dt.strftime(TIME_FORMAT),
            "duration": duration
        })

    res = {"duration_total": 0, "periods_num": 0, "periods": []}
    window = {}
    values = data.get('values', [])
    values = values[strip:len(values)-strip]
    for index, item in enumerate(values):
        ts, value = item
        if value is not None and float(value) < 1:
            if index == 0:
                window["start"] = ts
                window["end"] = ts
            else:
                prev_ts, prev_downtime = values[index - 1]
                window["end"] = ts
                if not window.get("start"):
                    window["start"] = prev_ts
        else:
            if window:
                _add_downtime(res, window)
            window = {}
    # case when downtime is not finished till end of timeseries
    if window:
        _add_downtime(res, window)
    return res


class WorkloadDowntimeReportBase:

    def __init__(self, test_name, release_key):
        self.test_name = test_name
        self.release_key = release_key
        self.cluster_name = 'child-cl'
        self.cluster_namespace = 'child-ns'
        self.cluster_provider_name = ''

    @property
    @abstractmethod
    def kubeconfig_file(self):
        pass

    @property
    @abstractmethod
    def grafanaclient(self):
        pass

    @property
    def os_manager(self):
        return OpenStackManager(kubeconfig=self.kubeconfig_file)

    @property
    def os_client_manager(self):
        return OpenStackClientManager(kubeconfig=self.kubeconfig_file)

    @cached_property
    def cloudprober_enabled(self):
        return self.os_manager.lma_config.get("exporters", {}).get("cloudprober", {}).get("enabled", False)

    @property
    def cloudprober_discovery_needed(self):
        start_ts = self.os_manager.get_cloudprober_start_ts()
        servers = self.os_client_manager.server.list(
            ["--os-compute-api-version=2.26",
             "--tags=openstack.lcm.mirantis.com:prober",
             "-f", "value", "-c", "ID", "-c", "Created At"],
            yaml_output=False
        ).rstrip().split("\n")
        for srv in servers:
            srv_id, ts = srv.split(' ')
            # 2025-01-14T09:50:04Z to timezone aware iso timestamp
            if datetime.fromisoformat(ts[:-1] + "+00:00") > datetime.fromisoformat(start_ts):
                LOG.info(f"Found server {srv_id} created at {ts} after cloudprober start at {start_ts}")
                return True
        LOG.info(f"All servers were created before cloudprober start {start_ts}")
        return False

    @cached_property
    def portprober_enabled(self):
        return self.os_manager.lma_config.get("exporters", {}).get("portprober", {}).get("enabled", False)

    @cached_property
    def enabled(self):
        if not settings.MOSK_WORKLOAD_DOWNTIME_REPORT:
            return False
        if self.cloudprober_enabled or self.portprober_enabled:
            return True
        return False

    def __enter__(self):
        if not self.enabled:
            LOG.info("Skipping worklod downtime report")
            return
        self.set_up()

    def __exit__(self, exc_type, exc_value, exc_traceback):
        # In case pytest.skip() is used inside context manager, there will be Skipped exc_type
        if exc_type is not None:
            LOG.warning(f"Skipping downtime report saving, as encountered exception {exc_type}")
            return
        if not self.enabled:
            LOG.info("Skipping worklod downtime report")
            return
        self.save()

    def set_up(self):
        if self.cloudprober_enabled and self.cloudprober_discovery_needed:
            LOG.info("Deleting cloudprober pods to speedup discovery of workloads")
            self.os_manager.restart_cloudprober()
        self.version_before_test = self.os_manager.os_controller_version()
        self.openstack_version_before_test = self.os_manager.get_osdpl_deployment().read().spec["openstack_version"]
        self.start = time.time()

    def save(self):
        self.end = time.time()
        self.version_after_test = self.os_manager.os_controller_version()
        self.openstack_version_after_test = self.os_manager.get_osdpl_deployment().read().spec["openstack_version"]

        if self.cloudprober_enabled:
            instance_timeseries = self.grafanaclient.get_os_instance_probe_success(
                "openstack-instances-icmp-probe",
                self.start,
                self.end,
            )
            instances = {}
            for ts in instance_timeseries:
                workload_name = ts["metric"]["dst"]
                downtime = calculate_workload_downtime(ts)
                instances[workload_name] = {"downtime": downtime}

            store_statistics(statistic_data=instances,
                             statistic_data_source="cloudprober",
                             statistic_endpoint_name="openstack_instances_icmp",
                             statistic_raw_data=instance_timeseries,
                             test_name=self.test_name,
                             cluster_name=self.cluster_name,
                             cluster_namespace=self.cluster_namespace,
                             cluster_provider_name=self.cluster_provider_name,
                             version_before_test=self.version_before_test,
                             version_after_test=self.version_after_test,
                             openstack_version_before_test=self.openstack_version_before_test,
                             openstack_version_after_test=self.openstack_version_after_test,
                             release_key=self.release_key,
                             start_time_str=datetime.fromtimestamp(self.start, tz=timezone.utc).strftime(TIME_FORMAT),
                             end_time_str=datetime.fromtimestamp(self.end, tz=timezone.utc).strftime(TIME_FORMAT))

        if self.portprober_enabled:
            instance_timeseries = self.grafanaclient.get_os_portprobe_success(
                'arping',
                self.start,
                self.end,
            )
            instances = {}
            for ts in instance_timeseries:
                workload_name = "{}-{}".format(ts["metric"]["ip_address"], ts["metric"]["mac"])
                # NOTE(vsaienko): portprober monitors all ports, including internal. The test may create
                # networks, subnets, routers, VMs. When they starts/removed we may do a false positive
                # check that port is down, while actually VM is starting or was just terminated and
                # removed soon. Drop 4 elements from both ends of samples (which are the frames when
                # resource appear and disappear from monitoring. This will let us to drop false positive
                # alerts.
                downtime = calculate_workload_downtime(ts, strip=4)
                instances[workload_name] = {"downtime": downtime}

            store_statistics(statistic_data=instances,
                             statistic_data_source="portprober",
                             statistic_endpoint_name="openstack_ports_arping",
                             statistic_raw_data=instance_timeseries,
                             test_name=self.test_name,
                             cluster_name=self.cluster_name,
                             cluster_namespace=self.cluster_namespace,
                             cluster_provider_name=self.cluster_provider_name,
                             version_before_test=self.version_before_test,
                             version_after_test=self.version_after_test,
                             openstack_version_before_test=self.openstack_version_before_test,
                             openstack_version_after_test=self.openstack_version_after_test,
                             release_key=self.release_key,
                             start_time_str=datetime.fromtimestamp(self.start, tz=timezone.utc).strftime(TIME_FORMAT),
                             end_time_str=datetime.fromtimestamp(self.end, tz=timezone.utc).strftime(TIME_FORMAT))


class BMWorkloadDowntimeReport(WorkloadDowntimeReportBase):
    def __init__(self, test_name, release_key, child_cluster):
        super().__init__(test_name, release_key)
        self.child_cluster = child_cluster
        self.cluster_name = child_cluster.name
        self.cluster_namespace = child_cluster.namespace
        self.cluster_provider_name = child_cluster.provider.provider_name

    @property
    def kubeconfig_file(self):
        child_kubeconfig_filename = "child_conf"
        child_kubeconfig_name, child_kubeconfig = self.child_cluster.get_kubeconfig_from_secret()
        with open(child_kubeconfig_filename, 'w') as f:
            f.write(child_kubeconfig)
        return os.path.abspath(child_kubeconfig_filename)

    @property
    def grafanaclient(self):
        return self.child_cluster.grafanaclient


class VMWorkloadDowntimeReport(WorkloadDowntimeReportBase):

    @property
    def kubeconfig_file(self):
        return settings.KUBECONFIG_PATH

    @property
    def grafanaclient(self):
        svc = self.os_manager.api.services.get(name='grafana', namespace='stacklight')
        proto = 'http'
        svc_ip = svc.get_external_addr()
        svc_port = next((s.port for s in svc.get_ports() if s.name == 'service'), None)
        client = GrafanaClient(host=svc_ip, port=svc_port, proto=proto)
        return client
