#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import copy
import functools
import os
from io import StringIO
from types import FunctionType

import jinja2
import paramiko
import yaml
import operator as py_operator
from si_tests.utils import packaging_version as ver
from jinja2_ansible_filters import AnsibleCoreFiltersExtension

import exec_helpers

from contextlib import contextmanager

from si_tests import logger
from si_tests import settings
from si_tests.deployments.utils import kubectl_utils
from si_tests.deployments.utils.namespace import NAMESPACE
from si_tests.exceptions import InvalidServiceUserException, UnknownProviderException
from si_tests.utils import waiters, utils, bootstrapv2 as bv2utils

LOG = logger.logger


class YamlEditor(object):
    """Manipulations with local or remote .yaml files.

    Usage:

    with YamlEditor("tasks.yaml") as editor:
        editor.content[key] = "value"

    with YamlEditor("astute.yaml", ip=self.admin_ip) as editor:
        editor.content[key] = "value"
    """

    def __init__(self, file_path, host=None, port=None,
                 username=None, password=None, private_keys=None, remote=None,
                 document_id=0,
                 default_flow_style=False, default_style=None):
        self.__file_path = file_path
        self.remote = remote
        self.host = host
        self.port = port or 22
        self.username = username
        self.__password = password
        self.__private_keys = private_keys or []
        self.__content = None
        self.__documents = [{}, ]
        self.__document_id = document_id
        self.__original_content = None
        self.default_flow_style = default_flow_style
        self.default_style = default_style

    @property
    def file_path(self):
        """Open file path

        :rtype: str
        """
        return self.__file_path

    @property
    def content(self):
        if self.__content is None:
            self.__content = self.get_content()
        return self.__content

    @content.setter
    def content(self, new_content):
        self.__content = new_content

    @contextmanager
    def open(self, mode="r"):
        file = None
        try:
            if self.remote:
                file = self.remote.open(self.__file_path, mode=mode)
            elif self.host:
                keys = map(paramiko.RSAKey.from_private_key,
                           map(StringIO, self.__private_keys))

                remote = exec_helpers.SSHClient(
                    host=self.host,
                    port=self.port,
                    auth=exec_helpers.SSHAuth(
                        username=self.username,
                        password=self.__password,
                        keys=list(keys)
                    )
                )

                file = remote.open(self.__file_path, mode=mode)
            else:
                file = open(self.__file_path, mode=mode)

            yield file
        finally:
            if file:
                file.close()

    def get_content(self):
        """Return a single document from YAML"""

        def multi_constructor(loader, tag_suffix, node):
            """Stores all unknown tags content into a dict

            Original yaml:
            !unknown_tag
            - some content

            Python object:
            {"!unknown_tag": ["some content", ]}
            """
            if type(node.value) is list:
                if type(node.value[0]) is tuple:
                    return {node.tag: loader.construct_mapping(node)}
                else:
                    return {node.tag: loader.construct_sequence(node)}
            else:
                return {node.tag: loader.construct_scalar(node)}

        yaml.add_multi_constructor("!", multi_constructor)
        with self.open('a+') as file_obj:
            file_obj.seek(0)
            self.__documents = [x for x in yaml.load_all(
                file_obj, Loader=yaml.SafeLoader)] or [{}, ]
            return self.__documents[self.__document_id]

    def write_content(self, content=None):
        if content:
            self.content = content
        self.__documents[self.__document_id] = self.content

        def representer(dumper, data):
            """Represents a dict key started with '!' as a YAML tag

            Assumes that there is only one !tag in the dict at the
            current indent.

            Python object:
            {"!unknown_tag": ["some content", ]}

            Resulting yaml:
            !unknown_tag
            - some content
            """
            key = data.keys()[0]
            if key.startswith("!"):
                value = data[key]
                if type(value) is dict:
                    node = dumper.represent_mapping(key, value)
                elif type(value) is list:
                    node = dumper.represent_sequence(key, value)
                else:
                    node = dumper.represent_scalar(key, value)
            else:
                node = dumper.represent_mapping(u'tag:yaml.org,2002:map', data)
            return node

        # FIXME: need to debug, why it not work with latest PyYaml ?
        # yaml.add_representer(dict, representer)
        with self.open('w') as file_obj:
            yaml.dump_all(self.__documents, file_obj,
                          default_flow_style=self.default_flow_style,
                          default_style=self.default_style)

    def __enter__(self):
        self.__content = self.get_content()
        self.__original_content = copy.deepcopy(self.content)
        return self

    def __exit__(self, x, y, z):
        if self.content == self.__original_content:
            return
        self.write_content()


class TemplateFile:
    def __init__(self, templates_dir, path):
        self.templates_dir = templates_dir
        self.path = os.path.normpath(path)
        self.override = None

        override_dir = self.templates_dir.override_path
        if override_dir:
            override_file = os.path.join(override_dir, self.path)
            if os.path.exists(os.path.join(self.templates_dir.path,
                                           override_file)):
                LOG.info("Override file exists ({})".format(override_file))
                self.override = override_file
            else:
                LOG.info("Override file does not exist ({})"
                         .format(override_file))

    def get_template_file(self):
        if self.override:
            LOG.info("Using override file '{}' as '{}'".format(self.override,
                                                               self.path))
            return os.path.join(self.templates_dir.path, self.override)
        else:
            LOG.info("Using file '{}'".format(self.path))
            return os.path.join(self.templates_dir.path, self.path)

    def render(self, **kwargs):
        """
        Render the template with respect to `.overrides` directory.

        :return: Rendered content of a template
        """
        return render_template(self.get_template_file(), **kwargs)


class TemplatesDir(list):
    def __init__(self, path, env_name=None, test_mode=False):
        super(TemplatesDir, self).__init__()
        self.path = path
        self.env_name = env_name
        self._override_path = None

        base_files = list(self.file_list(self.path))

        override_files = []
        if test_mode:
            LOG.info("Skipping overrides dir logic in test mode")
        elif self.env_name:
            path = os.path.join('.override', self.env_name)
            abspath = os.path.join(self.path, path)
            if os.path.exists(abspath):
                LOG.info("Overrides dir exists ({})".format(path))
                self._override_path = path
                override_files = list(self.file_list(abspath))
            else:
                LOG.info("Overrides dir does not exist ({})".format(path))

        LOG.info("Loading templates from '{}', env name '{}'"
                 .format(self.path, self.env_name))
        LOG.info("Base files:\n{}".format('\n'.join(base_files)))
        LOG.info("Override files:\n{}".format('\n'.join(override_files)))

        for file_path in sorted(set(base_files + override_files)):
            self.append(TemplateFile(templates_dir=self, path=file_path))

        LOG.info("{} template(s) found".format(len(self)))

    def file_list(self, path):
        for dirpath, dirs, files in os.walk(path):
            if '.override' in dirs:
                dirs.remove('.override')

            relpath = os.path.relpath(dirpath, path)
            for filename in files:
                if filename == '.defaults':
                    continue

                yield os.path.join(relpath, filename)

    @property
    def override_path(self):
        return self._override_path

    @property
    def filenames(self):
        return sorted([x.path for x in self])


def render_template(file_path, options=None, log_env_vars=True,
                    log_template=True, extra_env_vars=None):
    """Render a Jinja2 template file

    Extra options:
      {{ os_env(SOME_ENV_NAME, "default_value") }} : Get environment variable
      {{ os_env_bool(SOME_ENV_NAME, "1") }} : Get bool environment variable
      {{ feature_flags.enabled("aflag") }}

    Extra filters:
      {{ "aaa/bbb/ccc" | basename }} : get basename of the path
      {{ "aaa/bbb/ccc" | dirname }} : get dirname of the path

    :param log_env_vars:
    :param log_template:
    :param extra_env_vars: dict, extra k-v set of variables, analog global os_env
    :param file_path: str, path to the jinja2 template
    :param options: dict, extra objects to use in Jinja code blocks
    :log_env_vars: bool, log the environment variables used in the
                   template
    """
    required_env_vars = set()
    optional_env_vars = dict()
    if not extra_env_vars:
        extra_env_vars = dict()

    if log_template:
        _LOG = LOG.info
    else:
        _LOG = LOG.debug

    def extra_env(var_name, default=None, env_type=None):
        return os_env(var_name, default=default, env_type=env_type, env_source='extra_env_vars')

    def os_env(var_name: str, default=None, env_type=None, env_source='os'):
        """
        :param env_source: enum: os, env_config
                os -  means os.environ.get()
                env_config - means from dict, extra_env_vars
        :type default: object
        :type var_name: object
        """
        if env_source == 'os':
            requested_var = os.environ.get(var_name) or default
        elif env_source == 'extra_env_vars':
            requested_var = extra_env_vars.get(var_name, default)
        else:
            raise Exception('Wrong invocation')

        if requested_var is None:
            raise Exception("Environment variable '{0}' is undefined!"
                            .format(var_name))

        if default is None:
            required_env_vars.add(var_name)
        else:
            optional_env_vars[var_name] = requested_var

        if env_type == "bool":
            requested_var = settings.get_var_as_bool(var_name, default)

        return requested_var

    def os_env_bool(var_name, default=None):
        return os_env(var_name, default, env_type='bool')

    def basename(path):
        return os.path.basename(path)

    def dirname(path):
        return os.path.dirname(path)

    def version_compare(value, version, operator='eq'):
        """
         Perform a version comparison on a value
         Partial-copy-paste from Ansible 2.10.7
        """
        op_map = {
            '==': 'eq', '=': 'eq', 'eq': 'eq',
            '<': 'lt', 'lt': 'lt',
            '<=': 'le', 'le': 'le',
            '>': 'gt', 'gt': 'gt',
            '>=': 'ge', 'ge': 'ge',
            '!=': 'ne', '<>': 'ne', 'ne': 'ne'
        }
        if operator in op_map:
            operator = op_map[operator]
        else:
            raise Exception('Invalid operator type')

        try:
            method = getattr(py_operator, operator)
            return method(ver.parse(str(value)), ver.parse(str(version)))
        except Exception as e:
            raise Exception('Version comparison: %s' % e)

    render_options = {
        'os_env': os_env,
        'os_env_bool': os_env_bool,
        'extra_env': extra_env,
        'feature_flags': settings.FEATURE_FLAGS,
        'settings': settings,
    }
    if options:
        render_options.update(options)

    _LOG(f"Reading template file '{file_path}'")

    path, filename = os.path.split(file_path)
    environment = jinja2.Environment(
        extensions=[AnsibleCoreFiltersExtension],
        loader=jinja2.FileSystemLoader([path, os.path.dirname(path)],
                                       followlinks=True))
    environment.filters['basename'] = basename
    environment.filters['dirname'] = dirname
    environment.filters['version_compare'] = version_compare
    LOG.debug("Attempt to render template with ops:\n{0}".format(
        yaml.dump(render_options, indent=4)))

    template = environment.get_template(filename).render(render_options)

    if required_env_vars and log_env_vars:
        LOG.info("Required environment variables:")
        for var in required_env_vars:
            LOG.info("    {0}".format(var))
    if optional_env_vars and log_env_vars:
        LOG.info("Optional environment variables:")
        for var, default in sorted(optional_env_vars.items()):
            LOG.info("    {0} , value = {1}".format(var, default))
    return template


class NoAliasDumper(yaml.SafeDumper):
    def ignore_aliases(self, data):
        return True


class Bootstrapv2Applier():
    def __init__(self,
                 kubectl: kubectl_utils.RemoteKubectl,
                 templates_dir: str,
                 provider: str, region: str, namespace: str = None):
        self._kubectl = kubectl
        self._templates_dir = templates_dir
        self._namespace = namespace or NAMESPACE.default
        self._provider = provider
        self._region = region
        self._apply_creds = provider != settings.BAREMETAL_PROVIDER_NAME

    def _show_step(self, step_msg):
        LOG.info(f"\n"
                 f"\n{'=' * len(step_msg)}"
                 f"\n{step_msg}"
                 f"\n{'=' * len(step_msg)}"
                 f"\n")

    def _get_object_status(self,
                           resource_type: str, name: str,
                           readiness_path: str,
                           readiness_value: str = "true") -> bool:
        actual_value = self._kubectl.get(
            resource_type=resource_type,
            command=f"{name} -o=jsonpath='{{{readiness_path}}}'",
            namespace=f"{self._namespace}",
            string=True).lower()
        LOG.info(f"Checking readiness for {resource_type} {self._namespace}/{name}."
                 f" Readiness path: {readiness_path}, expected value: '{readiness_value.lower()}' ,"
                 f" actual value: {actual_value}")
        return actual_value == readiness_value.lower()

    def _get_objects_status(self,
                            resource_type: str, names: list,
                            readiness_path: str,
                            pre_callback: FunctionType = None,
                            readiness_value: str = "true") -> bool:
        if pre_callback:
            pre_callback()

        LOG.info(f"Checking {resource_type} resources in {self._namespace} namespace: {names}")
        for name in names:
            if not self._get_object_status(resource_type, name, readiness_path, readiness_value):
                return False

        return True

    def _wait_for_objects_status(self,
                                 resource_type: str, names: list,
                                 readiness_path: str,
                                 timeout: int = 600, interval: int = 30,
                                 pre_callback: FunctionType = None,
                                 readiness_value: str = "true") -> None:
        self._show_step(f"* Wait for '{resource_type}': '{names}' readiness on '{readiness_path}'")
        waiters.wait(lambda: self._get_objects_status(
            resource_type=resource_type,
            names=names,
            readiness_path=readiness_path,
            pre_callback=pre_callback,
            readiness_value=readiness_value),
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=f"Timeout for waiting {resource_type} {names} state after {timeout} sec.")

    def _get_provider_config_path(self) -> str:
        provider_config_name = f"{self._provider}-config.yaml.template" \
            if self._provider != settings.EQUINIXMETALV2_PROVIDER_NAME \
            else "equinix-config.yaml.template"
        return os.path.join(".", self._templates_dir, provider_config_name)

    def _get_provider_credentials_name(self, config_path: str) -> str:
        with self._kubectl.executor.open(config_path, "r") as r_f:
            creds_data = yaml.load(r_f.read(), Loader=yaml.SafeLoader)
        return creds_data['metadata']['name']

    def _check_vbmc_crd_existance(self) -> bool:
        try:
            crds = self._kubectl.get(resource_type="crds", command="-o custom-columns=NAME:.metadata.name", string=True)
            return "vbmcs.metal3.io" in crds
        except Exception:
            return False

    def apply_bootstrap_region(self) -> None:
        bootstrapregion_tmpl = os.path.join(".", self._templates_dir, "bootstrapregion.yaml.template")
        self._show_step(f"* Apply {bootstrapregion_tmpl}")
        self._kubectl.apply(bootstrapregion_tmpl)

    def wait_bootstrap_region(self) -> None:
        wait_timeout = settings.MCC_BV2_BOOTSTRAP_REGION_READINESS_TIMEOUT
        if settings.KAAS_BOOTSTRAP_INFINITE_TIMEOUT:
            wait_timeout = 24 * 60 * 60
        self._wait_for_objects_status(resource_type="bootstrapregions", names=[self._region],
                                      readiness_path=".status.ready", timeout=wait_timeout)

    def apply_bootstrap_region_and_wait(self) -> None:
        self.apply_bootstrap_region()
        self.wait_bootstrap_region()

    def patch_service_user_template(self, password: str) -> None:
        svc_user_tpl_path = os.path.join(".", self._templates_dir, "serviceusers.yaml.template")

        with self._kubectl.executor.open(svc_user_tpl_path, "r") as r_f:
            service_user_template = yaml.load(r_f.read(), Loader=yaml.SafeLoader)

        service_users = service_user_template['items']
        if len(service_users) != 1:
            raise InvalidServiceUserException(f"observed invalid number of service users ({len(service_users)})"
                                              f"in template {svc_user_tpl_path}")

        service_users[0]["spec"] = {"password": {"value": password}}
        service_users[0]["metadata"] = {"name": "serviceuser"}  # name is fixed always
        with self._kubectl.executor.open(svc_user_tpl_path, "w") as w_f:
            w_f.write(yaml.dump(service_user_template))

    def apply_service_user(self) -> None:
        serviceusers_tmpl = os.path.join(".", self._templates_dir, "serviceusers.yaml.template")
        self._show_step(f"* Apply {serviceusers_tmpl}")
        self._kubectl.apply(serviceusers_tmpl)

    def apply_airgap_values(self) -> None:
        cluster_tmpl_path = os.path.join(".", self._templates_dir, "cluster.yaml.template")
        with self._kubectl.executor.open(cluster_tmpl_path, "r") as r_f:
            cluster_tmpl = yaml.load(r_f.read(), Loader=yaml.SafeLoader)

        helm_releases = cluster_tmpl["spec"]["providerSpec"]["value"]["kaas"]["management"]["helmReleases"]
        helm_releases[0]["values"]["releasesBaseUrl"] = f"{settings.MCC_CDN_BINARY}/releases"
        helm_releases[1]["values"]["releasesBaseUrl"] = f"{settings.MCC_CDN_BINARY}/releases"

        with self._kubectl.executor.open(cluster_tmpl_path, "w") as w_f:
            w_f.write(yaml.dump(cluster_tmpl))

        LOG.info('The airgap values have been applied to cluster.yaml.template')

    def apply_provider_credentials_and_wait(self) -> None:
        # NOTE: for bm case this should be reworked probably into two different methods
        if not self._apply_creds:
            return

        provider_config_path = self._get_provider_config_path()
        self._show_step(f"* Apply '{provider_config_path}'")
        self._kubectl.apply(provider_config_path)

        creds_type = utils.get_credential_type_by_provider(self._provider)
        if not creds_type:
            raise UnknownProviderException(f"unknown provider {self._provider} in bootstrapv2")

        self._wait_for_objects_status(resource_type=creds_type,
                                      names=[self._get_provider_credentials_name(provider_config_path)],
                                      readiness_path=".status.valid")

    def apply_rhel_license(self) -> None:
        RHEL_LICENSE_TPL = os.path.join(".", self._templates_dir, "rhellicenses.yaml.template")
        if self._kubectl.executor.isfile(RHEL_LICENSE_TPL):
            self._show_step(f"* Apply '{RHEL_LICENSE_TPL}'")
            self._kubectl.apply(RHEL_LICENSE_TPL)

    def apply_metallb_config(self) -> None:
        METALLB_CONFIG_TPL = os.path.join(".", self._templates_dir, "metallbconfig.yaml.template")
        if self._kubectl.executor.isfile(METALLB_CONFIG_TPL):
            self._show_step(f"* Apply '{METALLB_CONFIG_TPL}'")
            self._kubectl.apply(METALLB_CONFIG_TPL)

    def apply_vmbc(self) -> None:
        VBMC_TPL = os.path.join(".", self._templates_dir, "vbmc.yaml.template")
        if self._kubectl.executor.isfile(VBMC_TPL):
            # check that the required CRD is in place
            waiters.wait(
                lambda: self._check_vbmc_crd_existance(),
                timeout=600, interval=30,
                timeout_msg="Timeout waiting for vbmc crd")
            self._show_step(f"* Apply '{VBMC_TPL}'")
            self._kubectl.apply(VBMC_TPL)

    def apply_bmhp(self) -> None:
        baremetalhostprofiles_tmpl = os.path.join(".", self._templates_dir, "baremetalhostprofiles.yaml.template")
        self._show_step(f"* Apply {baremetalhostprofiles_tmpl}")
        self._kubectl.apply(baremetalhostprofiles_tmpl)

    def apply_bmh(self) -> None:
        baremetalhosts_tmpl = os.path.join(".", self._templates_dir, "baremetalhosts.yaml.template")
        self._show_step(f"* Apply {baremetalhosts_tmpl}")
        self._kubectl.apply(baremetalhosts_tmpl)

    def wait_all_bmhs(self, namespace=NAMESPACE.default, timeout=3600) -> None:
        bmh_names = self._kubectl.get(
            resource_type="bmh",
            command="-o jsonpath='{.items[*].metadata.name}'",
            namespace=namespace,
            string=True).split()
        self.wait_bmhs(bmh_names, timeout=timeout)

    def wait_machine_status(self, namespace=NAMESPACE.default, timeout=3600) -> None:
        self.wait_machines(namespace, timeout=timeout)

    def wait_machines(self, namespace, timeout=3600) -> None:
        names = self._kubectl.get(
            resource_type="machines",
            command="-o jsonpath='{.items[*].metadata.name}'",
            namespace=namespace,
            string=True).split()
        self._show_step(f"* Wait for status on machines:  {names}")

        if settings.DAY1_PROVISIONING_PAUSE_VALUE == 'manual':
            day1_provision_machines = names.copy()
            for machine_name in names:
                _state = self._kubectl.get(
                    resource_type="machines",
                    command=f"{machine_name} -o jsonpath='{{.spec.providerSpec.value.day1Provision}}'",
                    namespace="default",
                    string=True)
                if _state == 'auto':
                    LOG.info(
                        f'Removing machine:{machine_name} from AwaitsProvision wait, '
                        f'since already have state .spec.providerSpec.value.day1Provision==auto')
                    day1_provision_machines.remove(machine_name)
            assert len(day1_provision_machines) > 0, ("At least 1 machine is expected to "
                                                      "have 'manual' Day1Provisioning value")
            self._wait_for_objects_status(resource_type="machines", names=day1_provision_machines,
                                          readiness_path=".status.providerStatus.status",
                                          timeout=timeout, readiness_value="AwaitsProvisioning")
            for machine_name in day1_provision_machines:
                self._show_step(f"* Removing provisioning pause for machine '{machine_name}'")
                self._kubectl.patch(
                    resource_type="machines",
                    object_name=machine_name,
                    patch='{"spec":{"providerSpec":{"value":{"day1Provisioning":"auto"}}}}',
                    namespace="default"
                    )
        else:
            LOG.info("Day1Provisioning is in 'auto' mode, so not going to wait for machines")

    def wait_bmhs(self, names: list, timeout=3600) -> None:
        self._wait_for_objects_status(resource_type="bmh", names=names,
                                      readiness_path=".status.provisioning.state",
                                      timeout=timeout, readiness_value="available")

    def apply_ipam(self) -> None:
        ipam_objects_config_tpl = os.path.join(".", self._templates_dir, "ipam-objects.yaml.template")
        self._show_step(f"* Apply {ipam_objects_config_tpl}")
        self._kubectl.apply(ipam_objects_config_tpl)

    def _apply_vvmt(self) -> None:
        proxy_name = ""
        if settings.KAAS_OFFLINE_DEPLOYMENT:
            LOG.info("Finding a proxy to use in VM Templates")
            proxy_name = next(iter(self._kubectl.get(
                resource_type="proxies",
                command="-l kaas.mirantis.com/bootstrap-proxy=true" +
                        " -o jsonpath='{.items[*].metadata.name}'",
                namespace="default",
                string=True).split() or []), "")
            LOG.info(f"Using {proxy_name} as proxy for all vvmt objects")

        seed_node_files = self._kubectl.executor.execute(f"ls -1 {self._templates_dir}").stdout
        vvmt_files = []
        # for Core CI we don't want to apply basic (vspherevmtemplates.yaml.template) template
        # because it has no configuration, and we have other configurations;
        # for SI we have the only configuration at once
        for f in seed_node_files:
            f = f.strip(b'\n').decode("utf-8")
            if f.startswith("vspherevmtemplate_") and f.endswith(".yaml.template"):
                vvmt_files.append(f)

        # if no templates has been found then try to find the basic template
        # it is the case of SI scenario
        if not vvmt_files:
            for f in seed_node_files:
                f = f.strip(b'\n').decode("utf-8")
                if f == "vspherevmtemplate.yaml.template":
                    vvmt_files.append(f)

        for vvmt in vvmt_files:
            fname = os.path.join(".", self._templates_dir, vvmt)

            if proxy_name:
                with self._kubectl.executor.open(fname, "r") as r_f:
                    vvmt_yaml_data = yaml.load(r_f.read(), Loader=yaml.SafeLoader)
                vvmt_yaml_data['spec']['proxyName'] = proxy_name
                with self._kubectl.executor.open(fname, "w") as w_f:
                    w_f.write(yaml.dump(vvmt_yaml_data))

            self._show_step(f"* Apply vvmt '{fname}' in the KinD cluster")
            self._kubectl.apply(fname)

    def _wait_all_vvmt(self) -> None:
        vvmt_names = self._kubectl.get(
            resource_type="vspherevmtemplates",
            command="-o jsonpath='{.items[*].metadata.name}'",
            string=True).split()

        self._wait_for_objects_status(
            resource_type="vspherevmtemplates",
            names=vvmt_names,
            readiness_path=".status.templateStatus",
            readiness_value="Present",
            timeout=settings.KAAS_VSPHERE_VVMT_CREATION_TIMEOUT)

    def apply_vvmt_and_wait(self) -> None:
        self._apply_vvmt()
        self._wait_all_vvmt()

    def apply_cluster(self) -> None:
        cluster_tmpl = os.path.join(".", self._templates_dir, "cluster.yaml.template")
        self._show_step(f"* Apply {cluster_tmpl} in the KinD cluster")
        self._kubectl.apply(cluster_tmpl)

    def start_cluster_deployment(self,
                                 cluster_name: str,
                                 provider: str,
                                 logs_callback: FunctionType) -> None:
        self._show_step(f"* Label bootstrapregion '{self._region}' with 'kaas.mirantis.com/region-approved=true'")
        self._kubectl.label(
            resource_type="bootstrapregions",
            value="kaas.mirantis.com/region-approved=true",
            namespace=NAMESPACE.default,
            object_name=self._region,
            flags="--overwrite")

        machine_names = self._kubectl.get(
            resource_type="machines",
            command=f"-l cluster.sigs.k8s.io/cluster-name={cluster_name}" +
                    " -o jsonpath='{.items[*].metadata.name}'",
            namespace="default",
            string=True).split()

        machines_wait_timeout = settings.MCC_BV2_MACHINES_DEPLOY_TIMEOUT
        if provider == settings.BAREMETAL_PROVIDER_NAME:
            machines_wait_timeout = settings.MCC_BV2_BM_MACHINES_DEPLOY_TIMEOUT
        cluster_bootstrap_readiness_timeout = settings.MCC_BV2_CLUSTER_BOOTSTRAP_READINESS_TIMEOUT
        cluster_readiness_timeout = settings.MCC_BV2_CLUSTER_READINESS_TIMEOUT

        if settings.KAAS_BOOTSTRAP_INFINITE_TIMEOUT:
            # make almost infinity timeouts, and relay only to job timeout
            machines_wait_timeout = 24 * 60 * 60
            cluster_bootstrap_readiness_timeout = 24 * 60 * 60
            cluster_readiness_timeout = 24 * 60 * 60

        cluster_release_name = self._kubectl.get(
            resource_type="clusters",
            command=f"{cluster_name} -o jsonpath='{{.spec.providerSpec.value.release}}'",
            namespace="default",
            string=True)
        cluster_release_version = self._kubectl.get(
            resource_type="clusterreleases",
            command=f"{cluster_release_name} -o jsonpath='{{.spec.version}}'",
            namespace="default",
            string=True)
        if utils.clusterrelease_version_greater_than_or_equal_to_kaas_2_30_0(cluster_release_version):
            # We need to wait until all machines are in AwaitsDeployment state
            # despite the setting.MACHINE_PAUSE_DURING_CREATION_ENABLED because
            # admission controller will block setting day1Deployment to 'auto'
            # until all machines are provisioned (in AwaitsDeployment state).
            # This requirement is only for management bootstrap.
            self._show_step("* Wait for all machines to be in AwaitsDeployment state")
            # don't wait for machine, if it was already set into 'auto'
            day1_deployment_machines = machine_names.copy()
            for machine_name in machine_names:
                _state = self._kubectl.get(
                    resource_type="machines",
                    command=f"{machine_name} -o jsonpath='{{.spec.providerSpec.value.day1Deployment}}'",
                    namespace="default",
                    string=True)
                if _state == 'auto':
                    LOG.warning(f'Removing machine:{machine_name} from AwaitsDeployment wait, '
                                f'since already have state .spec.providerSpec.value.day1Deployment==auto')
                    day1_deployment_machines.remove(machine_name)
            LOG.info(f"List machines with Day1Deploy status 'manual' {day1_deployment_machines}")
            if len(day1_deployment_machines) >= 1:
                self._wait_for_objects_status(
                    resource_type="machines",
                    names=day1_deployment_machines,
                    readiness_path=".status.providerStatus.status",
                    readiness_value="AwaitsDeployment",
                    timeout=machines_wait_timeout)
                for machine_name in day1_deployment_machines:
                    self._show_step(f"* Removing deployment pause for machine '{machine_name}'")
                    self._kubectl.patch(
                        resource_type="machines",
                        object_name=machine_name,
                        patch='{"spec":{"providerSpec":{"value":{"day1Deployment":"auto"}}}}',
                        namespace="default"
                    )

        # Wait for machines readiness; should use the same approach of checking
        # like it's being done in the boostrap-controller:
        # num_requested_nodes == num_ready_nodes
        # So we're checking the cluster status instead of per machine obj

        logs_callback_nodes = functools.partial(
            logs_callback,
            self._kubectl,
            extra_resource_message=f"\nLCMMachines status:\n{'-' * 19}\n",
            extra_resource_type="lcmmachines",
            extra_command="-o wide",
            extra_namespace="default")
        self._wait_for_objects_status(
            resource_type="clusters",
            names=[cluster_name],
            readiness_path=".status.providerStatus.nodes.ready",
            readiness_value=str(len(machine_names)),
            timeout=machines_wait_timeout,
            pre_callback=logs_callback_nodes)

        # Try to get kubeconfig
        self._show_step(f"* Try to get kubeconfig for the cluster '{cluster_name}'")
        waiters.wait(
            lambda: bv2utils.get_cluster_kubeconfig(
                self._kubectl.executor, cluster_name, self._provider),
            timeout=900, interval=60,
            timeout_msg="Timeout waiting to get cluster kubeconfig")

        logs_callback_bootstrapstatus = functools.partial(
            logs_callback,
            self._kubectl,
            extra_resource_message=f"\nCluster status.providerStatus.bootstrapStatus:\n{'-' * 46}\n",
            extra_resource_type="clusters",
            extra_command="-o jsonpath='{.items[0].status.providerStatus.bootstrapStatus}'",
            extra_namespace="default")
        # Wait for cluster bootstrap readiness
        self._wait_for_objects_status(
            resource_type="clusters",
            names=[cluster_name],
            readiness_path=".status.providerStatus.bootstrapStatus.bootstrapReady",
            timeout=cluster_bootstrap_readiness_timeout,
            pre_callback=logs_callback_bootstrapstatus)

        logs_callback_warnings = functools.partial(
            logs_callback,
            self._kubectl,
            extra_resource_message=f"\nCluster warnings:\n{'-' * 17}\n",
            extra_resource_type="clusters",
            extra_command="-o jsonpath='{.items[0].status.providerStatus.warnings}'",
            extra_namespace="default")
        # Wait for cluster readiness
        self._wait_for_objects_status(
            resource_type="clusters",
            names=[cluster_name],
            readiness_path=".status.providerStatus.ready",
            timeout=cluster_readiness_timeout,
            pre_callback=logs_callback_warnings)

        self._show_step(f"* Cluster '{cluster_name}' deploy completed")

    def apply_machines(self) -> None:
        machines_tmpl = os.path.join(".", self._templates_dir, "machines.yaml.template")
        self._show_step(f"* Apply {machines_tmpl}")
        self._kubectl.apply(machines_tmpl)
