import yaml
import os
import optparse
import sys

sys.path.append(os.getcwd())

try:
    from si_tests import settings
    from si_tests.managers.kaas_manager import Manager
except ImportError:
    print("ImportError: Run the application from the si-tests directory or "
          "set the PYTHONPATH environment variable to directory which contains"
          " ./si_tests")
    sys.exit(1)


def _find_upgrade_paths(from_version, clusters_versions_data, path=None):
    if path is None:
        path = []
    if from_version not in clusters_versions_data:
        return [path]
    cluster = clusters_versions_data[from_version]
    if 'availableUpgrades' not in cluster:
        return [path + [cluster['name']]]
    all_paths = []
    for upgrade in cluster['availableUpgrades']:
        new_version = upgrade['version']
        new_path = path + [cluster['name']]
        all_paths.extend(_find_upgrade_paths(new_version, clusters_versions_data, path=new_path))

    return all_paths


def _add_upgrade_paths(cluster_type, start_release_name, paths, clusters_names_data, upgrade_data):
    target_release_dict = {}
    for path in paths:
        target_release = path[-1]
        if target_release not in target_release_dict:
            target_release_dict[target_release] = []
        upgrade_path = {
            'optional': False,
            'versions': [{'release_name': name, 'numeric_version': clusters_names_data[name]['version']} for name in
                         path]
        }
        target_release_dict[target_release].append(upgrade_path)

    for target_release, path_for_release in target_release_dict.items():
        upgrade_data['upgrade_data'][cluster_type].append({
            'RELEASE': start_release_name,
            'TARGET_RELEASE': target_release,
            'OPTIONAL': False,
            'UPGRADE_PATH': path_for_release
        })


def generate_upgrade_paths_yaml_file(kaas_manager=None,
                                     store_file_path='/tmp/',
                                     kaas_release_file='',
                                     prev_paths_data_file=''):

    """
    :param kaas_manager: KaasManager object. If  kaas_release_file is set, then KaasManager is not used
    :param store_file_path: Path to directory where to store file. Optional.
                            If not set, then result just printed
    :param kaas_release_file: Path to kaas release file. Optional. If not set, then KaasManager is required.
                              This option can be used when you don't have access to KaasManager
    :param prev_paths_data_file: Path to previous yaml file with upgrade paths. This file will be used as a reference
                                 to mark paths as optional in new file if they existed in previous
    """

    if not kaas_release_file:
        release_data = kaas_manager.get_active_kaasrelease().data
    else:
        with open(kaas_release_file, 'r') as file:
            release_data = yaml.safe_load(file)

    supported = release_data['spec']['supportedClusterReleases']
    kaas_release_version = release_data['spec']['version'].replace('-rc', '')
    data = [cl for cl in supported if 'byo' not in cl.get('providers', {}).get('supported', [])]
    cluster_dict = {cluster['version']: cluster for cluster in data}
    cluster_name_dict = {cluster['name']: cluster for cluster in data}

    all_upgrade_paths = {}

    for cluster in data:
        version = cluster['version']
        upgrade_paths = _find_upgrade_paths(version, cluster_dict)
        filtered_paths = [path[1:] for path in upgrade_paths if len(path) > 1]
        if filtered_paths:
            all_upgrade_paths[cluster['name']] = filtered_paths

    upgrade_data = {
        'upgrade_data': {
            'mosk': [],
            'mke': []
        }
    }

    for release_name, upgrade_paths in all_upgrade_paths.items():
        if release_name.startswith('mosk'):
            _add_upgrade_paths('mosk', release_name, upgrade_paths,
                               clusters_names_data=cluster_name_dict, upgrade_data=upgrade_data)
        elif release_name.startswith('mke'):
            _add_upgrade_paths('mke', release_name, upgrade_paths,
                               clusters_names_data=cluster_name_dict, upgrade_data=upgrade_data)

    if prev_paths_data_file:
        with open(prev_paths_data_file, 'r') as prev_data:
            previous_data = yaml.safe_load(prev_data)
        for prefix in ['mosk', 'mke']:
            for element in previous_data['upgrade_data'][prefix]:
                for newelement in upgrade_data['upgrade_data'][prefix]:
                    if element == newelement:
                        newelement['OPTIONAL'] = True
                        for path in newelement['UPGRADE_PATH']:
                            path['optional'] = True

    if store_file_path:
        store_file_path = os.path.normpath(store_file_path)
        if not os.path.exists(store_file_path):
            try:
                os.makedirs(store_file_path)
                print(f"Directory {store_file_path} created.")
            except Exception as e:
                print(f"Error creating directory {store_file_path}: {e}")
                sys.exit(1)
        elif not os.path.isdir(store_file_path):
            print(f"Error: {store_file_path} is not a valid directory.")
            sys.exit(1)
        file_name = kaas_release_version + '.yaml'
        store_file_path = os.path.join(os.path.normpath(store_file_path), file_name)
        with open(store_file_path, 'w') as filepath:
            yaml.dump(upgrade_data, filepath, default_flow_style=False, sort_keys=False)
            print(f"File stored as {store_file_path}")
    print(yaml.dump(upgrade_data, default_flow_style=False, sort_keys=False))


def main():
    parser = optparse.OptionParser(usage="Usage: %prog --kaas-release-filepath <path_to_KaaSRelease_yaml> "
                                         "[--previous-version-paths-file <path_to_previous_upgrade_paths_yaml>]"
                                         "[--store-file <store_directory>]")
    parser.add_option('--kaas-release-filepath',
                      action="store", dest="releasefile",
                      help="Path to kaasrelease yaml file")
    parser.add_option('--previous-version-paths-file',
                      action="store", dest="prevdata",
                      help="Path to previous yaml with paths to mark as optional in new file")
    parser.add_option('--store-file',
                      action="store", dest="store_file_path",
                      help="Path to store the file")
    options, args = parser.parse_args()
    if not options.releasefile:
        if not settings.KUBECONFIG_PATH or not os.path.isfile(
                settings.KUBECONFIG_PATH):
            print("Please set KUBECONFIG environment variable whith"
                  "the path to the kubeconfig file for KaaS management cluster")
            return 11
        else:
            kaas_manager = Manager(kubeconfig=settings.KUBECONFIG_PATH)
            generate_upgrade_paths_yaml_file(
                kaas_manager=kaas_manager,
                store_file_path=options.store_file_path,
                prev_paths_data_file=options.prevdata)
    else:
        generate_upgrade_paths_yaml_file(
            store_file_path=options.store_file_path,
            kaas_release_file=options.releasefile,
            prev_paths_data_file=options.prevdata)


if __name__ == '__main__':
    sys.exit(main())
