#!/usr/bin/env python3

import argparse
import os
import sys

sys.path.append(os.getcwd())
try:
    from si_tests import settings
    from si_tests.managers.kaas_manager import Manager
except ImportError:
    print("ImportError: Run the application from the si-tests directory or "
          "set the PYTHONPATH environment variable to directory which contains"
          " ./si_tests")
    sys.exit(1)


def load_params():
    """
    Parse CLI arguments and environment variables

    Returns: ArgumentParser instance
    """
    parser = argparse.ArgumentParser(description=(
        'Generate upgrade paths for clusters to test cluster upgrades\n\n'
        'Returns multiple lines with comma-separated kaas clusterrelease\n'
        'versions, each line for a single '
        '"deploy->upgrade->upgrade->.." path.\n'
        'There are two upgrade strategies: inside the major release\n'
        'up to the latest minor version (X->Y->Z), and for the latest\n'
        'minor version of one major release to the latest minor version\n'
        'of the next major release:\n\n'
        '  A.X -> A.Y -> A.Z\n'
        '                 |\n'
        '                 v\n'
        '  B.X -> B.Y -> B.Z\n'
        '                 |\n'
        '                 v\n'
        '  C.X -> C.Y -> C.Z\n\n'
        'Valid --method options now are "minor_major_series" '
        'and "minor_major_pairs":\n\n'
        '"minor_major_series":\n'
        '  A-X,A-Y,A-Z,B-Z,C-Z\n'
        '  B-X,B-Y,B-Z,C-Z\n'
        '  C-X,C-Y,C-Z\n\n'
        '"minor_major_pairs":\n'
        '  A-X,A-Y\n'
        '  A-Y,A-Z\n'
        '  A-Z,B-Z\n'
        '  B-X,B-Y\n'
        '  B-Y,B-Z\n'
        '  B-Z,C-Z\n'
        '  C-X,C-Y\n'
        '  C-Y,C-Z\n\n'
        'There are several providers '
        'that can be included into upgrade procedure.\n'
        'By default upgrade path will consist of '
        'all possible upgrade sequences\n'
        'w/o releases with specified providers.\n'
        'But if you want to include some specific provider(s) \n'
        'you should use --providers argument\n'
        'and pass all needed providers separated by space.\n'
        'Valid --providers usage EXAMPLE is "--providers foo bar"\n'
    ), formatter_class=argparse.RawTextHelpFormatter)
    parser.add_argument('--method',
                        help=(f'Use one of the methods to generate paths, '
                              f'supported methods: {generate_methods.keys()}'),
                        default='kaas_defined_path_with_latest',
                        type=str)
    parser.add_argument('--oneline',
                        action='store_const',
                        const=";",
                        help='Show semicolon-separated release sequences '
                             'instead of newline-separated sequences',
                        default="\n")
    parser.add_argument('--providers',
                        help='Use space separated providers '
                             'from predefined list',
                        default="",
                        choices=['byo', 'baremetal', 'openstack', 'aws',
                                 'vsphere', 'equinixmetal', 'equinixmetalv2'],
                        nargs="+",
                        type=str,
                        required=False)
    return parser


generate_methods = {
    'ucp_defined_release_upgrade_path_with_latest': 'get_ucp_defined_release_upgrade_path_str',
    'openstack_defined_release_upgrade_path_with_latest': 'get_openstack_defined_release_upgrade_path_str',
    # not used:
    'kaas_defined_path_with_latest': 'get_kaas_defined_release_upgrade_path_str',
    'minor_major_series': 'get_upgrade_path_minor_major_series',
    'minor_major_pairs': 'get_upgrade_path_minor_major_pairs',
    'all_combinations': 'get_upgrade_path_combinations',
    'single_sequence': 'get_upgrade_path_single_sequence',
    'single_sequence_pairs': 'get_upgrade_path_single_sequence_pairs'
}


def main():
    parser = load_params()
    opts = parser.parse_args()

    if not settings.KUBECONFIG_PATH or not os.path.isfile(
            settings.KUBECONFIG_PATH):
        print("Please set KUBECONFIG environment variable with "
              "the path to the kubeconfig file for KaaS management cluster")
        return 11

    if opts.method not in generate_methods.keys():
        print(f"Unsupported --method value, please use one of: "
              f"{generate_methods.keys()}")
        return 12

    kaas_manager = Manager(kubeconfig=settings.KUBECONFIG_PATH)
    method_name = generate_methods[opts.method]
    method = getattr(kaas_manager.si_config.version_generators, method_name)

    print(opts.oneline.join(method(providers=opts.providers)))


if __name__ == '__main__':
    sys.exit(main())
