from si_tests.managers.kaas_manager import Manager
from si_tests.managers import helm3_manager
from si_tests.deployments.utils import file_utils
from si_tests.utils import utils
from si_tests import logger
from si_tests import settings
import os
LOG = logger.logger


def add_node_labels(kaas_manager, machine, labels=None):
    if labels:
        node = machine.get_k8s_node()
        node_name = machine.get_k8s_node_name()
        existing_labels = node.read().metadata.labels
        LOG.debug("Node {0} current labels: {1}".format(node_name, existing_labels))
        LOG.info("Adding labels: {0} to node {1}".format(labels, node_name))
        utils.merge_dicts(existing_labels, labels)
        body = {"metadata": {
                    "labels": existing_labels
                    }
                }
        kaas_manager.api.nodes.update(name=node_name, body=body)
        LOG.debug("Updated labels for node {0}: {1}".format(node_name, node.read().metadata.labels))


def install_chart(kubeconfig_path, chart_name, chart_path,
                  namespace, timeout="5m", values_path=None):
    helm = helm3_manager.Helm3Manager(namespace=namespace)
    cmd = [
        "install",
        chart_name,
        chart_path,
        "--kubeconfig",
        kubeconfig_path,
        "--namespace",
        namespace,
        "--wait",
        "--timeout",
        timeout
    ]
    if values_path is not None:
        cmd.extend(["-f", values_path])

    LOG.debug("Helm command: {}".format(cmd))
    stdout, stderr = helm.run_cmd(cmd)
    LOG.info(stdout)
    if stderr:
        LOG.error(stderr)
    return stdout, stderr


def patch_stacklight(kaas_manager, endpoint):
    ns = kaas_manager.get_namespace(settings.CLUSTER_NAMESPACE)
    child_cluster = ns.get_cluster(settings.CLUSTER_NAME)

    assert "stacklight" in child_cluster.get_components(),\
        "Stacklight is not deployed. Nothing to check"

    blackbox_values = file_utils.get_yaml_content(os.path.join(
                                                  os.path.dirname(os.path.abspath(__file__)),
                                                  "./yamls/blackbox_values.yaml"))
    prometheus_values = file_utils.get_yaml_content(os.path.join(
                                                    os.path.dirname(os.path.abspath(__file__)),
                                                    "./yamls/prometheus_values.yaml"))
    body = {
        "spec": {
            "providerSpec": {
                "value": {
                    "helmReleases": child_cluster.data.get(
                        "spec").get("providerSpec").get("value").get(
                        "helmReleases")
                }
            }
        }
    }
    sl_body = [x for x in
               body["spec"]["providerSpec"]["value"]["helmReleases"]
               if x["name"] == "stacklight"][0]

    LOG.info("Patching StackLight cluster")
    # workaround for not merging target lists
    if (sl_body["values"]["prometheusServer"].get("customScrapeConfigs") and
            sl_body["values"]["prometheusServer"]["customScrapeConfigs"].get("refapp-blackbox")):
        sl_body["values"]["prometheusServer"]["customScrapeConfigs"]["refapp-blackbox"]["static_configs"][0]["targets"]\
            .append(endpoint)
    else:
        prometheus_values["prometheusServer"]["customScrapeConfigs"]["refapp-blackbox"]["static_configs"][0]["targets"]\
            = [endpoint]
        utils.merge_dicts(sl_body["values"], prometheus_values)

    utils.merge_dicts(sl_body["values"], blackbox_values)
    kaas_manager.api.kaas_clusters.update(
        name=child_cluster.name, namespace=child_cluster.namespace, body=body)

    LOG.info("Check cluster readiness after lma update")
    child_cluster.check.check_cluster_readiness()

    # TODO wait for new target to appear in Prometheus


def test_deploy_refapp(kaas_manager):
    # Setting variables
    child_name = settings.CLUSTER_NAME
    child_namespace = settings.CLUSTER_NAMESPACE
    refapp_namespace = settings.TARGET_NAMESPACE
    artifacts_dir = settings.ARTIFACTS_DIR

    storage_class_name = "default"
    maria_selector_key = settings.REFAPP_NODE_LABEL.split('=')[0]
    maria_selector_value = settings.REFAPP_NODE_LABEL.split('=')[1]
    maria_volume_size = settings.REFAPP_DB_SIZE
    maria_chart = settings.REFAPP_DB_CHART_URL
    refapp_chart = settings.REFAPP_CHART_URL
    refapp_image = settings.REFAPP_IMAGE
    refapp_replicas = settings.REFAPP_API_REPLICAS
    db_password = utils.gen_random_password(8)

    # Acquiring child kubeconfig
    ns = kaas_manager.get_namespace(child_namespace)
    child_cluster = ns.get_cluster(child_name)
    kubeconfig_name, child_kubeconfig = child_cluster.get_kubeconfig()
    kubeconfig_path = os.path.join(artifacts_dir, kubeconfig_name)
    LOG.info("Saving child cluster kubeconfig to %s", kubeconfig_path)
    with open(kubeconfig_path, "w") as f:
        f.write(child_kubeconfig)
    os.chmod(path=kubeconfig_path, mode=640)

    # Overrides for database chart
    values_maria = {
        "labels": {"server": {
                "node_selector_key": maria_selector_key,
                "node_selector_value": maria_selector_value}
        },
        "volume": {
            "enabled": True,
            "class_name": storage_class_name,
            "size": maria_volume_size,
            "backup": {
                "enabled": False}
        },
        "manifests": {
            "configmap_ingress_conf": False,
            "configmap_ingress_etc": False,
            "deployment_ingress": False,
            "deployment_error": False,
            "service_ingress": False,
            "service_error": False
        },
        "endpoints": {"oslo_db": {"auth": {"admin": {
                        "password": db_password}}}}
    }
    maria_values_path = os.path.join(artifacts_dir, "mariadb_overrides.yaml")
    file_utils.save_to_yaml(values_maria, maria_values_path)

    # Overrides for refapp chart
    values_refapp = {
        "refapp": {
            "image": refapp_image,
            "replicaCount": refapp_replicas},
        "database": {
            "password": db_password}
    }
    refapp_values_path = os.path.join(artifacts_dir, "refapp_overrides.yaml")
    file_utils.save_to_yaml(values_refapp, refapp_values_path)

    # Labeling nodes for database instances
    child_manager = Manager(kubeconfig_path)
    LOG.info("Labeling nodes")
    machines = child_cluster.get_machines(machine_type="worker")
    label = {maria_selector_key: maria_selector_value}
    for machine in machines:
        add_node_labels(child_manager, machine, label)
    LOG.info("Nodes have been labeled")

    # Creating namespace
    LOG.info("Creating refapp namespace")
    child_manager.get_or_create_namespace(refapp_namespace)

    # Installing DB chart
    LOG.info("Installing MariaDB Helm chart")
    LOG.debug("Using Maria chart {}".format(maria_chart))
    install_chart(kubeconfig_path, chart_name="mariadb", chart_path=maria_chart,
                  namespace=refapp_namespace, values_path=maria_values_path)

    # Installing RefApp API chart
    LOG.info("Installing RefApp Helm chart")
    LOG.debug("Using RefApp chart {}".format(refapp_chart))
    install_chart(kubeconfig_path, chart_name="refapp", chart_path=refapp_chart,
                  namespace=refapp_namespace, timeout="3m", values_path=refapp_values_path)

    child_manager.api.pods.check_pods_statuses(target_namespaces=refapp_namespace)

    # TODO check if db-filler job completed successfully

    # Check all API replicas ready
    refapp_deployment = child_manager.api.deployments.get(name="refapp", namespace=refapp_namespace).data
    assert refapp_deployment["status"]["ready_replicas"] == refapp_replicas,\
        "Not all RefApp replicas are ready"

    # Get public endpoint
    child_manager.api.services.get(name="refapp", namespace=refapp_namespace).wait_for_external_ip()
    refapp_endpoint = child_manager.api.services.get(name="refapp", namespace=refapp_namespace).get_external_ip()
    LOG.info("RefApp endpoint is {}:80".format(refapp_endpoint))

    # Add endpoint to Stacklight monitoring
    patch_stacklight(kaas_manager, "http://{}/records".format(refapp_endpoint))
