#    Copyright 2020 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import yaml

from si_tests import logger
from si_tests import settings
from jsonpath_ng.ext import parse

LOG = logger.logger


def get_kaas_release(kaas_manager, upgrade_spec):
    LOG.info("Getting KaasRelease object from the cluster ...")

    kaas_release_spec = upgrade_spec.get('kaasrelease', {})
    if 'name' in kaas_release_spec:
        if 'namespace' in kaas_release_spec:
            return kaas_manager.get_kaasrelease(
                name=kaas_release_spec['name'],
                namespace=kaas_release_spec['namespace'])
        else:
            return kaas_manager.get_kaasrelease(
                name=kaas_release_spec['name'])
    else:
        kaas_releases_available = kaas_manager.get_kaasreleases()
        LOG.info('Available KaaS releases are {}'.format(
            [x.name for x in kaas_releases_available]))

        def kaas_releases_enabled_filter(kaas_release):
            name = 'kaas.mirantis.com/active'
            labels = kaas_release.data['metadata'].get('labels')
            if labels:
                value = labels.get(name)
            else:
                return False
            if value is None:
                return False
            if isinstance(value, str):
                return value.lower() == 'true'
            if isinstance(value, bool):
                return value
            raise Exception("Bad label value: '{}' == '{}'"
                            .format(name, value))

        kaas_releases_enabled = list(
            filter(kaas_releases_enabled_filter, kaas_releases_available))

        assert len(kaas_releases_enabled) > 0, \
            "No active 'kaasrelease' objects found"

        assert len(kaas_releases_enabled) == 1, (
            "There are more than one active 'kaasrelease' objects: {}".format(
                [x.name for x in kaas_releases_enabled]))

        return kaas_releases_enabled[0]


def get_cluster_release(kaas_manager, kaas_release):
    LOG.info("Getting ClusterRelease object from the cluster ...")

    return kaas_manager.get_clusterrelease(
        kaas_release.data['spec']['clusterRelease'])


def get_cluster(kaas_manager, upgrade_spec):
    LOG.info("Getting Cluster object from the cluster ...")

    cluster_spec = upgrade_spec.get('cluster', {})
    cluster_name = cluster_spec.get('name', settings.CLUSTER_NAME)
    cluster_namespace = cluster_spec.get('namespace',
                                         settings.CLUSTER_NAMESPACE)

    namespace = kaas_manager.get_namespace(cluster_namespace)
    cluster = namespace.get_cluster(cluster_name)
    return cluster


def patch_kaas_release(kaas_manager, kaas_release, cluster_release):
    LOG.info("Updating KaasRelease object")

    body = kaas_release.data
    # FIXME: Workaround for PRODX-2534
    body['apiVersion'] = body.pop('api_version')

    name = '{}-test'.format(body['metadata']['name'])
    body['metadata']['name'] = name

    cr_name = cluster_release.data['metadata']['name']
    cr_version = cluster_release.data['spec']['version'].split('+')[0]

    body['spec']['clusterRelease'] = cr_name
    body['spec']['supportedClusterReleases'].append({
        'name': cr_name,
        'version': cr_version
    })

    name = body['metadata']['name']
    namespace = body['metadata'].get('namespace')
    LOG.info("Create new KaasRelease object ({})".format(name))
    return kaas_manager.api.kaas_kaasreleases.create(
        name=name, namespace=namespace, body=body)


def patch_cluster_release(kaas_manager, cluster_release, upgrade_spec):
    LOG.info("Updating ClusterRelease object")

    body = cluster_release.data
    # FIXME: Workaround for PRODX-2534
    body['apiVersion'] = body.pop('api_version')

    name = '{}-test'.format(body['metadata']['name'])
    body['metadata']['name'] = name

    labels = body['metadata'].setdefault('labels', {})
    if isinstance(labels, dict):
        labels['kaas.mirantis.com/active'] = 'false'
    else:
        body['metadata']['labels'] = {
            'kaas.mirantis.com/active': 'false'
        }

    if 'kaas_ubuntu_repo' in upgrade_spec:
        expr = parse(
            "$.spec.machineTypes.['control','worker'][?name='setup']")
        for match in expr.find(body):
            match.value['params']['kaas_ubuntu_repo'] = \
                upgrade_spec['kaas_ubuntu_repo']

    if 'upgrade_packages' in upgrade_spec:
        expr = parse(
            "$.spec.machineTypes.['control','worker'][?name='deploy']")
        for match in expr.find(body):
            match.value['params']['upgrade_packages'] = \
                str(upgrade_spec['upgrade_packages']).lower()
            match.value['params']['reboot_allowed'] = "true"

    name = body['metadata']['name']
    namespace = body['metadata'].get('namespace')
    LOG.info("Create new ClusterRelease object ({})".format(name))
    return kaas_manager.api.kaas_clusterreleases.create(
        name=name, namespace=namespace, body=body)


def patch_cluster(cluster, kaas_release, cluster_release):
    LOG.info("Update KAAS BM components version")

    body = cluster.data
    # FIXME: Workaround for PRODX-2534
    body['apiVersion'] = body.pop('api_version')

    body['spec']['providerSpec']['value']['kaas']['release'] \
        = kaas_release.data['metadata']['name']

    body['spec']['providerSpec']['value']['release'] \
        = cluster_release.data['metadata']['name']

    name = body['metadata']['name']
    LOG.info("Apply updated Cluster object ({})".format(name))
    cluster.patch({'spec': body['spec']})

    return cluster


def switch_active_kaas_release(active, *args):
    for kaasrelease in args:
        LOG.info("Disabling KaasRelease ({})".format(kaasrelease.name))
        kaasrelease.patch(
            {'metadata': {'labels': {'kaas.mirantis.com/active': 'false'}}})

    LOG.info("Enabling KaasRelease ({})".format(active.name))
    active.patch(
        {'metadata': {'labels': {'kaas.mirantis.com/active': 'true'}}})


def lcmmachine_belongs_lcmcluster(lcmmachine, lcmcluster):
    lcmcluster_uid = lcmcluster.data['metadata']['uid']
    for owner_ref in lcmmachine.data['metadata'].get('owner_references', []):
        if owner_ref['uid'] == lcmcluster_uid:
            return True
    return False


def get_lcmmachines(kaas_manager, lcmcluster, namespace=None):
    if not namespace:
        namespace = lcmcluster.data['metadata']['namespace']
    for lcmmachine in kaas_manager.get_lcmmachines(namespace=namespace):
        if lcmmachine_belongs_lcmcluster(lcmmachine, lcmcluster):
            yield lcmmachine


def test_upgrade_kaas_bm_host(kaas_manager):
    """ Test KAAS BM host OS packages upgrade

    This performs a series of actions to test KaaS BM host OS
    upgradability. It requires deployed KaaS Manangement Cluster.

    Steps to test the upgrade:
        * get kaasrelease object (get exact object by name / namespace
          if those are given or find the active object by label)
        * get clusterrelease object
        * get cluster object by name / namespace
        * create a copy of ClusterRelease object and update it by adding
          necessary fields to StateItems
        * create a copy of KaasRelease and update it by adding new
          ClusterRelease to the list of supported upgrades and change current
          release version
        * change 'active' flag in kaasrelease objects
        * update Cluster object by replacing KaasRelease / ClusterRelease
          versions
        * wait until each LCMMachine object changes from Ready state to
          non-Ready state and then back to Ready

    KAAS_BM_UPGRADE_SPEC env var is used to specify a path to a file
    that contains KaaS BM upgrade spec. The structure is the following:

        # Name of a KaasRelease object to use as baseline
        kaasrelease:
          name:
          namespace:
          version:
        # Name of a Cluster object to be updated to trigger an upgrade
        cluster:
          name:
          namespace:
        # STRING: Ubuntu repo to switch to
        kaas_ubuntu_repo:
        # BOOLEAN: Enable updating packages (implies reboot)
        upgrade_packages:

    :param openstack_client:
    :return: None
    """

    upgrade_spec = None

    if settings.KAAS_BM_UPGRADE_SPEC_FILE:
        with open(settings.KAAS_BM_UPGRADE_SPEC_FILE) as f:
            upgrade_spec = yaml.load(f.read(), Loader=yaml.SafeLoader)

    if not upgrade_spec:
        LOG.warning("No KaaS BM upgrade spec given via "
                    "KAAS_BM_UPGRADE_SPEC_FILE, skipping test")
        return

    kaas_release = get_kaas_release(kaas_manager, upgrade_spec)
    cluster = get_cluster(kaas_manager, upgrade_spec)
    cluster_release = get_cluster_release(kaas_manager, kaas_release)

    cluster_release_patched = patch_cluster_release(
        kaas_manager, cluster_release, upgrade_spec)

    kaas_release_patched = patch_kaas_release(
        kaas_manager, kaas_release, cluster_release_patched)

    switch_active_kaas_release(kaas_release_patched, kaas_release)

    cluster_patched = patch_cluster(cluster, kaas_release_patched,
                                    cluster_release_patched)

    cluster_patched.check.check_cluster_release(
        cluster_release_patched.data['metadata']['name']
    )
    cluster_patched.check.check_cluster_nodes(
        timeout=300,
        expected_status=settings.get_machine_updating_status(),
    )
    cluster_patched.check.check_cluster_nodes(timeout=3600)
    cluster_patched.check.check_k8s_nodes(timeout=1800)
