import pytest
import json
import time
import yaml
import os

from si_tests import logger
from si_tests.deployments.deploy_refapp import Refapp
from si_tests.deployments.utils import file_utils
from si_tests.utils.workload_report import VMWorkloadDowntimeReport
from si_tests.lcm import openstack_lcm
from si_tests import settings
from si_tests.utils import utils, waiters, templates

LOG = logger.logger


@pytest.mark.usefixtures("mos_workload_downtime_report")
@pytest.mark.usefixtures("mos_per_node_workload_check_after_test")
@pytest.mark.usefixtures("check_pods_during_update")
def test_update_os_controller_simple(os_manager):
    """Update OS Controller.
       Parameters required for test execution:
         - KUBECONFIG
         - OSH_OPERATOR_UPDATE_REPO
         - OSH_OPERATOR_UPDATE_TAG
         - OSH_OPERATOR_CHART_URL
    """

    openstack_lcm.update_os_controller(os_manager)


@pytest.mark.usefixtures("mos_per_node_workload_check_after_test")
@pytest.mark.usefixtures("check_pods_during_update")
def test_update_os_controller_with_refapp(os_manager,
                                          openstack_client_manager,
                                          func_name):
    """Update OS Controller with refapp tests. Test can be run only on vms.
    1) Deploy RefApp
    2) Check RefApp workability before update
    3) Update OpenStack controller
    4) Check RefApp workability after update
    5) Delete RefApp

       Parameters required for test execution:
         - KUBECONFIG
         - OSH_OPERATOR_UPDATE_REPO
         - OSH_OPERATOR_UPDATE_TAG
         - OSH_OPERATOR_CHART_URL
    """

    refapp = Refapp(openstack_client_manager)
    try:
        refapp.deploy()
        refapp.check()
        with VMWorkloadDowntimeReport(func_name, 'mos'):
            openstack_lcm.update_os_controller(os_manager)
        refapp.check()
    except Exception:
        if (settings.OPENSTACK_COLLECT_REFAPP_MODE.lower() == "on_error"):
            refapp.collect_logs()
        raise
    finally:
        if (settings.OPENSTACK_COLLECT_REFAPP_MODE.lower() == "always"):
            refapp.collect_logs()
        refapp.delete()


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.TARGET_CLUSTER)])
def test_update_os_controller_with_pinning_images(os_manager, _):
    """Update OS Controller to the patch version with pinning the image versions
     of the following components:
     - Open vSwitch
     - Kubernetes entrypoint

     1) Create configmap in OpenStack namespace if Rockoon is used
     2) Add the OpenvSwitch and kubernetes-entrypoint images
     to the OpenStack deployment object
     3) Waiting for network section in OpenStack deployment has been updated
     4) Retrieve OVS pod names before updating OpenStack controller
     3) Start updating OpenStack controller
     4) Retrieve OVS pod names after updating OpenStack controller
     5) Ensure that OVS pods don't restart during update to the patch version

           Parameters required for test execution:
             - KUBECONFIG
             - OSH_OPERATOR_UPDATE_REPO
             - OSH_OPERATOR_UPDATE_TAG
             - OSH_OPERATOR_CHART_URL
        """
    os_helmbundle_images = os_manager.get_os_helmbundle_images()[
        "openstack-networking"]["openstack-openvswitch"]
    osdpl_name = settings.OSH_DEPLOYMENT_NAME
    osdpl = os_manager.get_openstackdeployment(osdpl_name)
    fp_before_update = osdpl.read().status["fingerprint"]
    netw_body = osdpl.data

    if os_manager.is_rockoon_used:
        LOG.info("Rockoon is used")
        openstack_deployment_artifacts_config = file_utils.join(os.path.dirname(
            os.path.abspath(__file__)),
            "../../templates/configmaps/openstack_deployment_artifacts_config.yaml")
        configmap_name = ('test-update-os-controller-with-pinning-images-%s' %
                          str(time.time_ns()))
        pod_template = templates.render_template(
            openstack_deployment_artifacts_config,
            {'OPENSTACK_DEPLOYMENT_ARTIFACTS_NAME': configmap_name,
             'KUBE_ENTRYPOINT_IMAGE_URL': os_helmbundle_images["dep_check"],
             'OPENVSWITCH_IMAGE_URL': os_helmbundle_images["openvswitch_vswitchd"]}
        )
        configmap_json_body = json.dumps(
            yaml.load(pod_template, Loader=yaml.SafeLoader))
        LOG.info(f"Creating configmap {configmap_name} in OpenStack namespace")
        os_manager.api.configmaps.create(
            name=configmap_name, namespace="openstack",
            body=json.loads(configmap_json_body))

    LOG.info("Adding the OpenvSwitch and kubernetes-entrypoint images"
             "to the openstackdeployment object")
    netw_body = utils.merge_dicts(netw_body, {"spec": {"services": {
        "networking": {"openvswitch": {"values": {"images": {"tags": {
            "dep_check": os_helmbundle_images["dep_check"],
            "openvswitch_db_server": os_helmbundle_images[
                "openvswitch_db_server"],
            "openvswitch_vswitchd": os_helmbundle_images[
                "openvswitch_vswitchd"]
        }}}}}}}})
    osdpl.patch(body=netw_body)
    LOG.info("Waiting for network section in OpenStack deployment has been updated")
    timeout_msg = "Network section in Openstack deployment hasn't been updated"
    waiters.wait(os_manager.check_osdpl_status_value,
                 predicate_args=('fingerprint',
                                 fp_before_update,
                                 False),
                 timeout=1200,
                 timeout_msg=timeout_msg)
    LOG.info("Retrieving OVS pod names before updating OpenStack controller")
    ovs_pods_before_update = os_manager.api.pods.list(namespace="openstack",
                                                      name_prefix="openvswitch")
    LOG.info("Start updating OpenStack controller")
    openstack_lcm.update_os_controller(os_manager)

    LOG.info("Retrieving OVS pod names after updating OpenStack controller")
    ovs_pods_after_update = os_manager.api.pods.list(namespace="openstack",
                                                     name_prefix="openvswitch")

    LOG.info("Ensuring that OVS pods do not restart during the patch version update")
    utils.verify(ovs_pods_after_update == ovs_pods_before_update,
                 f"Pods before and after update are different. "
                 f"{ovs_pods_before_update} != {ovs_pods_after_update}",
                 f"Pods before and after update are same. "
                 f"{ovs_pods_before_update} == {ovs_pods_after_update}")
