import pytest

from si_tests import logger
from si_tests import settings
from si_tests.utils import update_child_clusterrelease_actions
from si_tests.utils import update_release_names
from si_tests.utils.utils import check_test_result
from si_tests.utils.workload_report import BMWorkloadDowntimeReport

LOG = logger.logger


update_release_names = list(update_release_names.generate_update_release_names())
is_update_test_failed = False


@pytest.fixture(scope='function', params=update_release_names,
                ids=[f"RELEASE={x}" for x in update_release_names])
def update_release_name(request):
    global is_update_test_failed
    # Check if the previous update steps failed
    if is_update_test_failed:
        msg = (f"Skip updating clusterrelease to {request.param} because "
               f"previous update step failed")
        LOG.info(msg)
        pytest.skip(msg)

    yield request.param

    # Check the result of the current step
    test_passed = (hasattr(request.node, 'rep_call') and
                   request.node.rep_call.passed)
    if not test_passed:
        is_update_test_failed = True


@pytest.fixture(scope='function')
def bm_mcc_workload_downtime_report(func_name, kaas_manager, update_release_name, request):
    if check_test_result(request, ['skipped']):
        LOG.warning("Test was skipped, so skip workload downtime report")
        yield
        return
    if not settings.MOSK_WORKLOAD_DOWNTIME_REPORT:
        LOG.info("MOSK_WORKLOAD_DOWNTIME_REPORT is disabled, skipping workload downtime report")
        yield
        return

    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    child_cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    cr_before = child_cluster.clusterrelease_version

    if not child_cluster.k8sclient.openstackdeployment.available:
        LOG.info("Openstack deployment is not present, skipping workload downtime report")
        yield
        return

    if child_cluster.is_patchrelease_upgrade(clusterrelease_version_before=cr_before,
                                             clusterrelease_version_after=update_release_name):
        release_key = 'patch_release'
    else:
        release_key = 'major_release'

    report = BMWorkloadDowntimeReport(func_name, release_key, child_cluster)
    report.set_up()
    yield
    if check_test_result(request, ['skipped', 'failed']):
        LOG.warning("Test wasn't successful, so skip downtime report saving after test")
        return
    report.save()


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.TARGET_CLUSTER)])
@pytest.mark.usefixtures("store_updated_child_cluster_description")
@pytest.mark.usefixtures("introspect_child_target_objects")
@pytest.mark.usefixtures("introspect_PRODX_9696_target_cluster")
@pytest.mark.usefixtures("introspect_ceph_child")
@pytest.mark.usefixtures("bm_mcc_workload_downtime_report")
@pytest.mark.usefixtures('mcc_per_node_workload_check_after_test')
@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
@pytest.mark.usefixtures('collect_machines_timestamps')
@pytest.mark.usefixtures('introspect_child_lcm_operation_stuck')
@pytest.mark.usefixtures('introspect_machines_stages')
@pytest.mark.usefixtures('log_start_end_timestamps')
@pytest.mark.usefixtures("check_ceph_keyrings")
@pytest.mark.usefixtures('mcc_loadtest_prometheus')
@pytest.mark.usefixtures('mcc_loadtest_grafana')
@pytest.mark.usefixtures('mcc_loadtest_alerta')
@pytest.mark.usefixtures("runtime_restart_checker")
@pytest.mark.usefixtures('mcc_loadtest_kibana')
@pytest.mark.usefixtures('mcc_loadtest_alertmanager')
@pytest.mark.usefixtures('mcc_loadtest_keycloak')
@pytest.mark.usefixtures('create_hoc_before_lcm_and_delete_after')
def test_update_child_clusterrelease(kaas_manager, update_release_name, _):
    """Update child cluster release"""
    pytest.skip("Direct child cluster update is disabled by feature flag, use update plan instead")

    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    child_cluster = ns.get_cluster(cluster_name)
    update_actions = update_child_clusterrelease_actions.UpdateChildClusterreleaseActions(child_cluster)
    update_actions.pre_update(update_release_name)

    # TODO: since 2.31 it will be done automatically, required only for 2.30
    # Delete KaaSCephCluster if required
    if child_cluster.workaround.skip_kaascephcluster_usage():
        child_cluster.remove_kaascephcluster()
        child_cluster.check.wait_kaascephcluster_removed()
    ##########################
    #  Start cluster update  #
    ##########################
    child_cluster.update_cluster(update_release_name)

    if child_cluster.is_ceph_deployed:
        if update_actions.update_info.is_patchrelease:
            LOG.warning("SKIP checking ceph ClusterWorkLoadLock for a patchrelease update")
        else:
            # Ceph cwl name is hardcoded and can't be changed
            ceph_cwl_name = settings.CEPH_CWL_NAME
            LOG.info("Check ceph ClusterWorkLoadLock")
            # State inactive means that ceph cluster is started update
            # Now we need to wait for state active which means ceph
            # cluster is finished update or reconcile to check
            # that update is not required.
            child_cluster.check.wait_clusterworkloadlock_state(
                name=ceph_cwl_name, state='inactive')
            LOG.info("Ceph cwl became inactive. Continue")

    child_cluster.check.check_cluster_release(update_release_name)
    child_cluster.check.check_update_finished(timeout=settings.KAAS_CHILD_CLUSTER_UPDATE_TIMEOUT, interval=120)
    #################################
    #  Cluster update is completed  #
    #################################

    update_actions.base_readiness_check()
    update_actions.post_update()
