import copy
import enum
import pytest
import socket
import random

from si_tests import settings, logger
from si_tests.managers import bootstrap_manager
from si_tests.managers import pdns_manager
from si_tests.managers.kaas_manager import Manager
from si_tests.utils import certs
from OpenSSL import crypto
from kubernetes.client.rest import ApiException

LOG = logger.logger
bootstrap = bootstrap_manager.BootstrapManager.get_si_config_bootstrap_manager()


def generate_custom_ca_cert(cn='si-tests.test'):
    ###########
    # CA Cert #
    ###########
    ca_key = crypto.PKey()
    ca_key.generate_key(crypto.TYPE_RSA, 2048)

    ca_cert = crypto.X509()
    ca_cert.set_version(2)
    ca_cert.set_serial_number(random.randint(50000000, 100000000))

    ca_subj = ca_cert.get_subject()
    ca_subj.commonName = cn

    ca_cert.add_extensions([
        crypto.X509Extension(b"subjectKeyIdentifier", False, b"hash", subject=ca_cert),
    ])

    ca_cert.add_extensions([
        crypto.X509Extension(b"authorityKeyIdentifier", False, b"keyid:always", issuer=ca_cert),
    ])

    ca_cert.add_extensions([
        crypto.X509Extension(b"basicConstraints", False, b"CA:TRUE"),
        crypto.X509Extension(b"keyUsage", False, b"keyCertSign, cRLSign"),
    ])

    ca_cert.set_issuer(ca_subj)
    ca_cert.set_pubkey(ca_key)

    ca_cert.gmtime_adj_notBefore(0)
    ca_cert.gmtime_adj_notAfter(10 * 365 * 24 * 60 * 60)

    ca_cert.sign(ca_key, 'sha256')

    ca_pem = str(crypto.dump_certificate(crypto.FILETYPE_PEM, ca_cert), encoding='utf-8')
    ca_key_pem = str(crypto.dump_privatekey(crypto.FILETYPE_PEM, ca_key), encoding='utf-8')

    return ca_pem, ca_key_pem


def generate_custom_app_cert(ca_path=None,
                             ca_key_path=None,
                             dns_names=None,
                             ips=None,
                             not_valid_before=0,
                             not_valid_after=10*365*24*60*60):
    #############################
    # Application cert and key  #
    #############################

    if not (dns_names or ips):
        raise ValueError("Neither hostname nor ip address were provided to generate certificate")

    subj_alt_name = [b'DNS:'+bytes(dns, encoding='utf8') for dns in dns_names] if dns_names \
        else [b'IP:'+bytes(ip, encoding='utf8') for ip in ips]

    with open(ca_path, "r") as f:
        ca = crypto.load_certificate(crypto.FILETYPE_PEM, f.read())

    with open(ca_key_path, "r") as f:
        ca_key = crypto.load_privatekey(crypto.FILETYPE_PEM, f.read())

    client_key = crypto.PKey()
    client_key.generate_key(crypto.TYPE_RSA, 2048)

    client_cert = crypto.X509()
    client_cert.set_version(2)
    client_cert.set_serial_number(random.randint(50000000, 100000000))

    client_subj = client_cert.get_subject()
    client_subj.commonName = "Custom MCC app cert"

    client_cert.add_extensions([
        crypto.X509Extension(b"basicConstraints", False, b"CA:FALSE"),
        crypto.X509Extension(b"subjectKeyIdentifier", False, b"hash", subject=client_cert),
        crypto.X509Extension(b"authorityKeyIdentifier", False, b"keyid:always", issuer=ca),
        crypto.X509Extension(b"extendedKeyUsage", False, b"serverAuth"),
        crypto.X509Extension(b"keyUsage", False, b'digitalSignature,keyEncipherment'),
        crypto.X509Extension(b"subjectAltName", False, b",".join(subj_alt_name)),
    ])

    client_cert.set_issuer(ca.get_issuer())
    client_cert.set_pubkey(client_key)

    client_cert.gmtime_adj_notBefore(not_valid_before)
    client_cert.gmtime_adj_notAfter(not_valid_after)

    client_cert.sign(ca_key, 'sha256')

    cert_pem = str(crypto.dump_certificate(crypto.FILETYPE_PEM, client_cert), encoding='utf-8')
    cert_key_pem = str(crypto.dump_privatekey(crypto.FILETYPE_PEM, client_key), encoding='utf-8')

    return cert_pem, cert_key_pem


class TLSServices(enum.Enum):
    """Prefixes for interfaces that do not require some types of checks
    Items:
        service_namespace: Namespace of the service
        service_name:      Name of the service
        dns_records:       List of DNS records to add into the Cluster DNS zone for this service
        cluster_tls_name:  Name of the TLS key for the Cluster.spec.providerSpec.value.tls dictionary
        description:       Item description for logging
        for_management:    Boolean, 'True' for testing the service on Management cluster, 'False' is to skip
        for_regional:      Boolean, 'True' for testing the service on Region cluster, 'False' is to skip
        for_child:         Boolean, 'True' for testing the service on Child cluster, 'False' is to skip
        for_mosk:          Boolean, 'True' for testing the service on MOSK cluster, 'False' is to skip
    """
    alerta = ("stacklight", "iam-proxy-alerta", "alerta.stacklight", "iamProxyAlerta",
              "Alerta (StackLight)", True, True, True, True)
    alertmanager = ("stacklight", "iam-proxy-alertmanager", "alertmanager.stacklight", "iamProxyAlertManager",
                    "Alert Manager (StackLight)", True, True, True, True)
    grafana = ("stacklight", "iam-proxy-grafana", "grafana.stacklight", "iamProxyGrafana",
               "Grafana (StackLight)", True, True, True, True)
    kibana = ("stacklight", "iam-proxy-kibana", "kibana.stacklight", "iamProxyKibana",
              "Kibana (StackLight)", True, True, True, True)
    prometheus = ("stacklight", "iam-proxy-prometheus", "prometheus.stacklight", "iamProxyPrometheus",
                  "Prometheus (StackLight)", True, True, True, True)
    cache = ("kaas", "mcc-cache", "cache.kaas", "cache",
             "MCC Cache", True, True, False, False)
    ui = ("kaas", "kaas-kaas-ui", "ui.kaas", "ui",
          "MCC UI", True, False, False, False)
    keycloak = ("kaas", "iam-keycloak-http", "keycloak.kaas", "keycloak",
                "MCC Keycloak", True, False, False, False)
    mke = (None, None, "mke", "mke",
           "MKE API", True, True, True, False)

    def __init__(self, service_namespace, service_name, dns_record, cluster_tls_name, description,
                 for_management, for_regional, for_child, for_mosk):
        self.service_namespace = service_namespace
        self.service_name = service_name
        self.dns_record = dns_record
        self.cluster_tls_name = cluster_tls_name
        self.description = description
        self.for_management = for_management
        self.for_regional = for_regional
        self.for_child = for_child
        self.for_mosk = for_mosk


def save_pem_artifact(filename, pem_data):
    with open(f'{settings.ARTIFACTS_DIR}/{filename}', mode='w') as f:
        f.write(pem_data)


def tls_services():
    """Prepare set of parameters to test each service with updated TLSConfig

    1. Generate CA certificate that will be used for all the services in this cluster
    2. Fill 'yield_data' with the services related only to the current cluster type
       (management, region, child or mosk)
    3. For each related service, create DNS record in the PDNS on seed node, using
       the IP address from the kubernetes 'services' object.
       Exception: MKE IP address, which is taken from 'loadBalancerHost' after resolving
       it from a hostname (if there is a hostname instead of IP)
    """

    cluster_name = settings.TARGET_CLUSTER
    cluster_namespace = settings.TARGET_NAMESPACE
    build = f"build{settings.JENKINS_BUILD_NUMBER}"
    zone_name = f"{build}.{cluster_name}.{cluster_namespace}.{settings.SEED_PDNS_LOCAL_ZONE}"

    LOG.info("Generate CA certificate for the specified TARGET_CLUSTER public services")
    ca_pem, ca_key_pem, ca_path = bootstrap._generate_ca_cert(ca_path=zone_name)
    save_pem_artifact('ca.pem', ca_pem)
    save_pem_artifact('ca_key.pem', ca_key_pem)

    LOG.info("Generate CA certificate for the specified TARGET_CLUSTER public services for negative tests")
    ca_pem_neg, ca_key_pem_neg = generate_custom_ca_cert()
    ca_path_neg = f'{settings.ARTIFACTS_DIR}/ca_neg.pem'
    ca_key_path_neg = f'{settings.ARTIFACTS_DIR}/ca_key_neg.pem'
    save_pem_artifact('ca_neg.pem', ca_pem_neg)
    save_pem_artifact('ca_key_neg.pem', ca_key_pem_neg)

    LOG.info("Create DNS records in PDNS Authoritative server for the specified TARGET_CLUSTER public services")
    pdns = pdns_manager.PDNSManager(bootstrap_manager=bootstrap)
    pdns.create_zone(zone_name)

    kaas_manager = Manager(kubeconfig=settings.KUBECONFIG_PATH)
    ns = kaas_manager.get_namespace(cluster_namespace)
    cluster = ns.get_cluster(cluster_name)
    # Save KUBECONFIG to artifacts to work with job artifacts from the pipeline scenario
    certs.update_seed_kubeconfig(bootstrap=bootstrap, cluster=cluster)

    is_management = cluster.is_management
    is_regional = cluster.is_regional
    is_child = cluster.is_child
    is_mosk = cluster.is_mosk
    k8sclient = cluster.k8sclient
    yield_data = []
    for s in TLSServices:
        if not any([is_management and s.for_management,
                    is_regional and s.for_regional,
                    # if MOSK, then skip some services
                    is_mosk and s.for_mosk,
                    (not is_mosk) and is_child and s.for_child]):
            LOG.warning(f"Skip testing service '{s.description}' for the cluster '{cluster_namespace}/{cluster_name}'")
            continue

        if s.name == 'mke':
            # get IP for MKE from Cluster loadBalancerHost
            load_balancer_address = cluster.get_load_balancer_address()
            try:
                ip = socket.gethostbyname(load_balancer_address)
            except socket.gaierror as e:
                raise Exception(f"Invalid Cluster loadbalancer host '{load_balancer_address}': {e}")
        else:
            # get IP from service
            if not k8sclient.services.present(name=s.service_name, namespace=s.service_namespace):
                raise Exception(f"Service {s.service_namespace}/{s.service_name} not found in the Cluster "
                                f"{cluster_namespace}/{cluster_name}")
            service = k8sclient.services.get(name=s.service_name, namespace=s.service_namespace)
            ip = service.get_external_ip()
            if not ip:
                raise Exception(f"Service {s.service_namespace}/{s.service_name} don't contain external IP address "
                                f"in the Cluster {cluster_namespace}/{cluster_name}")

        LOG.info(f"Creating PDNS records for '{s.name}' with ip '{ip}'")
        pdns.add_record(zone_name, s.dns_record, 'A', ip)

        yield_data.append({
            'name': s.name,
            'dns_name': f"{s.dns_record}.{zone_name}",
            'ip': ip,
            'cluster_tls_name': s.cluster_tls_name,
            'description': s.description,
            'ca': (ca_pem, ca_key_pem, ca_path),
            'is_management': is_management,
            'is_regional': is_regional,
            'is_child': is_child,
            'is_mosk': is_mosk,
            'ca_for_negative_test': (ca_pem_neg, ca_key_pem_neg, ca_path_neg, ca_key_path_neg),
        })

    pdns.restart_service()  # PRODX-50239
    pdns.show_zone(zone_name)
    return yield_data


tls_services_params = list(tls_services())


@pytest.mark.usefixtures("introspect_child_target_objects")
@pytest.mark.parametrize("tls_service", tls_services_params,
                         ids=[f"TLS_SERVICE={x['name']}" for x in tls_services_params])
def test_tlsconfig_for_public_services(kaas_manager, tls_service, show_step):
    """Test TLSConfig with custom DNS names and certificates for cluster public services
    Use only with RUN_ON_REMOTE=True, to access the DNS resolver on the seed node

    Scenario:
        1. Generate certificate for public service using the created DNS records
        2. Apply the certificate using TLSConfig object
        3. Check related Clusters readiness
        4. Check that certificate on the Machines in the current region actually contains the expected DNS name
        5. Check that the new CA is applied for the kubeconfig and helm-controller secrets
        6. Enable RUN_ON_REMOTE=True in SI_CONFIG, so all further tests could resolve the DNS names on seed node PDNS
        # 4. Remove custom TLSConfig objects
    """
    cluster_name = settings.TARGET_CLUSTER
    cluster_namespace = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(cluster_namespace)
    cluster = ns.get_cluster(cluster_name)

    show_step(1)
    LOG.info(f"Generate certificate for the service '{tls_service['description']}'")
    ca_pem, ca_key_pem, ca_path = tls_service['ca']

    cert_pem, key_pem = bootstrap._generate_app_cert(
        dns_names=[tls_service['dns_name']],
        ca_path=ca_path,
        cert_dir=tls_service['name'])
    save_pem_artifact(f"cert_{tls_service['name']}.pem", cert_pem)
    save_pem_artifact(f"cert_{tls_service['name']}_key.pem", key_pem)

    # Update KUBECONFIG_ARTIFACT if MKE certificate will be updated for Management or Regional cluster
    if tls_service['name'] == 'mke':
        # if cluster.is_management or cluster.is_regional:
        # Use KUBECONFIG with disabled TLS verify, to allow working with certificate hostnames which
        # are not resolving from the Jenkins agents
        kubeconfig_path = certs.update_seed_kubeconfig(bootstrap=bootstrap, cluster=cluster, skip_tls_verify=True)
        if cluster.is_management:
            # If MKE certificate will be updated for Management, then re-initialize
            # 'kaas_manager' object using the KUBECONFIG with disabled TLS verification
            # IMPORTANT: use this KUBECONFIG for all next tests in the pipeline
            #            in case if these tests are running without RUN_ON_REMOTE flag
            #            and MKE certificate uses the DNS name, which is created on seed node PDNS
            #            and is not resolved from other nameservers in the network.
            kaas_manager.api.update_kubeconfig(kubeconfig_path)

    # Update the certificate
    show_step(2)
    cert_manager = certs.CertManager(kaas_manager)
    cert_manager.apply_cert_for_app(
        app=tls_service['name'],
        cluster=cluster,
        hostname=tls_service['dns_name'],
        cert_pem=cert_pem,
        key_pem=key_pem,
        ca_pem=ca_pem,
        app_tlsspec=tls_service['cluster_tls_name'])

    # 1. If MKE is updated on Management - check all clusters readiness, and agent/helmcontroller
    #      in clusters of the current region, update kubeconfig
    #    If MKE is updated on Region - check region and it's clusters readiness, and agent/helmcontroller
    #      in clusters of the current region, update kubeconfig
    #    If MKE is updated on Child - check child readiness.
    # 2. If Cache - check readiness as in #1 (except childs)
    # 3. If Keycloak - check readiness as in #1 for Management
    show_step(3)

    # Management cluster to check readiness
    check_clusters = []
    if cluster.is_management:
        if tls_service['name'] in ['mke', 'cache', 'keycloak']:
            # Check cluster readiness after MKE, Cache or Keycloak certificate changed
            check_clusters.append(cluster)  # Management cluster
            check_clusters.extend(kaas_manager.get_regional_clusters())  # All Region clusters
            check_clusters.extend(kaas_manager.get_child_clusters())     # All Child clusters
        else:
            # Check only Management cluster for all other services
            check_clusters.append(cluster)  # Management cluster

    # Region cluster to check readiness
    elif cluster.is_regional:
        if tls_service['name'] in ['mke', 'cache']:
            # Check cluster readiness after MKE, Cache or Keycloak certificate changed
            check_clusters.append(cluster)  # Region cluster
            check_clusters.append(kaas_manager.get_mgmt_cluster())  # Management cluster
            # Child clusers from current MCC region
            check_clusters.extend(kaas_manager.get_child_clusters_in_region(cluster.region_name))
        else:
            # Check only Region cluster for all other services
            check_clusters.append(cluster)  # Region cluster

    # Child cluster to check readiness
    else:
        check_clusters.append(cluster)  # Child cluster

    # Check clusters readiness
    for check_cluster in check_clusters:
        check_cluster.check.check_cluster_readiness(timeout=5400)

    show_step(4)
    # Check that certificate on the host actually contains the expected DNS name
    cluster.check.check_cert_conversation(tls_service['dns_name'])

    show_step(5)
    if cluster.is_management or cluster.is_regional:
        # If Region or Management TLS certificates were changed, then it may affect all the clusters in the region
        if tls_service['name'] == 'mke':
            target_clusters = [cluster]
            target_clusters.extend(kaas_manager.get_child_clusters_in_region(cluster.region_name))
            for target_cluster in target_clusters:
                LOG.info(f"Checking cluster {target_cluster.name} certificate")
                target_cluster.check.validate_agent_kubeconfig(ca=ca_pem)
                target_cluster.check.validate_helm_controller_secret(ca=ca_pem)
        else:
            LOG.info("MKE certificates was not updated, skipping check for updated CA in the secrets")
    else:
        LOG.info("Skip Management/Region CA checks for a Child cluster")

    show_step(6)
    kaas_manager.si_config.update_run_on_remote(run_on_remote=True)


@pytest.mark.usefixtures("introspect_child_target_objects")
@pytest.mark.parametrize(
    "tls_service",
    tls_services_params,
    ids=[f"TLS_SERVICE={x['name']}" for x in tls_services_params],
)
def test_remove_tlsconfig_for_public_services(
    kaas_manager: Manager, tls_service, show_step
):
    """Test remove TLSConfig with custom DNS names and certificates for cluster public services
    Use only with RUN_ON_REMOTE=True, to access the DNS resolver on the seed node

    Scenario:
        1. Remove the certificate using TLSConfig object
        2. Update kubeconfig
        3. Check related Clusters readiness
        4. Check machines are available by ip addresses
        5. Enable RUN_ON_REMOTE=True in SI_CONFIG, so all further tests could resolve the DNS names on seed node PDNS

    """

    cluster_name = settings.TARGET_CLUSTER
    cluster_namespace = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(cluster_namespace)
    cluster = ns.get_cluster(cluster_name)

    old_x509_cert = certs.get_certificate_from_host(hostname=tls_service['dns_name'])

    show_step(1)
    cert_manager = certs.CertManager(kaas_manager)
    if 'mke' in tls_service["name"]:
        try:
            cert_manager.remove_cert_for_app(
                app=tls_service['name'],
                cluster=cluster,
                app_tlsspec=tls_service['cluster_tls_name'],
                old_cert=old_x509_cert,
            )
        except ApiException as e:
            if e.status == 400:
                assert ('unsetting MKE tls spec is not allowed' in e.body)
    else:
        cert_manager.remove_cert_for_app(
            app=tls_service['name'],
            cluster=cluster,
            app_tlsspec=tls_service['cluster_tls_name'],
            old_cert=old_x509_cert,
        )

    show_step(2)
    # Update KUBECONFIG_ARTIFACT if MKE certificate will be updated for Management or Regional cluster
    if tls_service["name"] == "mke":
        # if cluster.is_management or cluster.is_regional:
        # Use KUBECONFIG with disabled TLS verify, to allow working with certificate hostnames which
        # are not resolving from the Jenkins agents
        kubeconfig_path = certs.update_seed_kubeconfig(
            bootstrap=bootstrap, cluster=cluster, skip_tls_verify=True
        )
        if cluster.is_management:
            # If MKE certificate will be updated for Management, then re-initialize
            # 'kaas_manager' object using the KUBECONFIG with disabled TLS verification
            # IMPORTANT: use this KUBECONFIG for all next tests in the pipeline
            #            in case if these tests are running without RUN_ON_REMOTE flag
            #            and MKE certificate uses the DNS name, which is created on seed node PDNS
            #            and is not resolved from other nameservers in the network.
            kaas_manager.api.update_kubeconfig(kubeconfig_path)

    show_step(3)
    # Management cluster to check readiness
    check_clusters = []
    if cluster.is_management:
        if tls_service["name"] in ["mke", "cache", "keycloak"]:
            # Check cluster readiness after MKE, Cache or Keycloak certificate changed
            check_clusters.append(cluster)  # Management cluster
            check_clusters.extend(
                kaas_manager.get_regional_clusters()
            )  # All Region clusters
            check_clusters.extend(
                kaas_manager.get_child_clusters()
            )  # All Child clusters
        else:
            # Check only Management cluster for all other services
            check_clusters.append(cluster)  # Management cluster

    # Region cluster to check readiness
    elif cluster.is_regional:
        if tls_service["name"] in ["mke", "cache"]:
            # Check cluster readiness after MKE, Cache or Keycloak certificate changed
            check_clusters.append(cluster)  # Region cluster
            check_clusters.append(kaas_manager.get_mgmt_cluster())  # Management cluster
            # Child clusers from current MCC region
            check_clusters.extend(
                kaas_manager.get_child_clusters_in_region(cluster.region_name)
            )
        else:
            # Check only Region cluster for all other services
            check_clusters.append(cluster)  # Region cluster

    # Child cluster to check readiness
    else:
        check_clusters.append(cluster)  # Child cluster

    # Check clusters readiness
    for check_cluster in check_clusters:
        check_cluster.check.check_cluster_readiness(timeout=5400)

    show_step(4)
    # Check that certificate on the host actually contains the expected DNS name
    if tls_service["name"] != 'mke':
        cluster.check.check_cert_conversation(tls_service['ip'])


negative_scenarios = ['cert_expired', 'cert_not_valid_yet', 'wrong_ca', 'mismatch_dns_name']


@pytest.mark.usefixtures("introspect_child_target_objects")
@pytest.mark.parametrize("tls_service", tls_services_params,
                         ids=[f"TLS_SERVICE={x['name']}" for x in tls_services_params])
@pytest.mark.parametrize("negative_scenario", negative_scenarios,
                         ids=[f"NEGATIVE_TEST={s}" for s in negative_scenarios])
def test_tlsconfig_for_public_services_negative(kaas_manager, tls_service, negative_scenario, show_step):
    """Negative test TLSConfig with custom DNS names and invalid certificates for cluster public services

    Scenario:
        1. Generate not valid certificate for public service
        2. Try to apply the certificate using TLSConfig object
        3. Check that certificate has not been applied
    """
    cluster_name = settings.TARGET_CLUSTER
    cluster_namespace = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(cluster_namespace)
    cluster = ns.get_cluster(cluster_name)

    show_step(1)

    ca_pem, ca_key_pem, ca_path, ca_key_path = tls_service['ca_for_negative_test']

    if negative_scenario == 'cert_expired':
        LOG.info(f"Generate certificate that has expired for the service '{tls_service['description']}'")
        cert_pem, key_pem = generate_custom_app_cert(
            ca_path=ca_path,
            ca_key_path=ca_key_path,
            dns_names=[tls_service['dns_name']],
            not_valid_before=-10*24*60*60,
            not_valid_after=-1*24*60*60,
        )
    elif negative_scenario == 'cert_not_valid_yet':
        LOG.info(f"Generate certificate that is not valid yet for the service '{tls_service['description']}'")
        cert_pem, key_pem = generate_custom_app_cert(
            ca_path=ca_path,
            ca_key_path=ca_key_path,
            dns_names=[tls_service['dns_name']],
            not_valid_before=1*24*60*60,
            not_valid_after=10*24*60*60,
        )
    elif negative_scenario == 'mismatch_dns_name':
        LOG.info(f"Generate certificate whose DNS name does not match service hostname"
                 f"for the service '{tls_service['description']}'")
        cert_pem, key_pem = generate_custom_app_cert(
            ca_path=ca_path,
            ca_key_path=ca_key_path,
            dns_names=[f"{tls_service['dns_name']}"],
        )
        # Override DNS name for TLSConfig creation
        tls_service = copy.deepcopy(tls_service)
        tls_service['dns_name'] = "mismatch.name.negativetest"
    elif negative_scenario == 'wrong_ca':
        LOG.info(f"Generate certificate signed by another CA for the service '{tls_service['description']}'")
        cert_pem, key_pem = generate_custom_app_cert(
            ca_path=ca_path,
            ca_key_path=ca_key_path,
            dns_names=[f"{tls_service['dns_name']}"],
        )
        ca_pem, ca_key_pem = generate_custom_ca_cert(cn="Any fake CN")
    else:
        pytest.skip(f'Unexpected test case "{negative_scenario}"')

    save_pem_artifact(f"cert_{tls_service['name']}_{negative_scenario}.pem", cert_pem)
    save_pem_artifact(f"cert_{tls_service['name']}_{negative_scenario}_key.pem", key_pem)

    # Update the certificate
    show_step(2)
    cert_manager = certs.CertManager(kaas_manager)

    # Try to apply wrong certificate and check
    error_status = ""
    try:
        cert_manager.apply_cert_for_app(
            app=tls_service['name'],
            cluster=cluster,
            hostname=tls_service['dns_name'],
            cert_pem=cert_pem,
            key_pem=key_pem,
            ca_pem=ca_pem,
            app_tlsspec=tls_service['cluster_tls_name'])
    except ApiException as e:
        if e.status == 400:
            error_status = f"TLSConfig was not applied, that expected for {negative_scenario} test\n{e}"
            LOG.info(error_status)

    show_step(3)
    if error_status == "":
        raise Exception(f"TLSConfig should not be applied in case of negative test {negative_scenario}"
                        f" for the service '{tls_service['description']}'")
