# Copyright 2025 Mirantis, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

import random

import pytest

from si_tests import logger
from si_tests.managers.clustercheck_mos_manager import ClusterCheckMosManager
from si_tests.utils import packaging_version as version

LOG = logger.logger


class TestTFCassandraRestoration(object):
    # Workload details
    lb_url = None

    @staticmethod
    def _set_cls_attr(request, stack_outputs, attr):
        for output in stack_outputs:
            if output['output_key'] == attr:
                setattr(request.cls, attr, output['output_value'])

    @pytest.fixture(scope='class', autouse=True)
    def setup_cls(self, request, openstack_client_manager):
        LOG.info("Create heat stack with loadbalancer before Cassandra cluster tests")
        stack = ClusterCheckMosManager.created_stack_tf_lb(request, openstack_client_manager)
        self._set_cls_attr(request, stack['outputs'], 'lb_url')
        assert ClusterCheckMosManager.is_lb_functional(openstack_client_manager, 2, self.lb_url)

    @pytest.mark.usefixtures('mos_tf_api_loadtest')
    def test_delete_cassandra_db_pod(self, show_step, tf_manager, os_manager, openstack_client_manager):
        """ Check Cassandra database restoration after pod deletion.

        Scenario:
            1. Delete cassandra pod
            2. Wait cassandra cluster become ready
            3. Check cassandra db status
            4. Check tf-api readiness
            5. Check OpenStack workload (TF LB) functionality of created heat stack (Setup class)
        """

        show_step(1)
        pods = tf_manager.get_cassandra_pods()
        pod = random.choice(pods)
        LOG.info(f"Delete pod {pod.data['metadata']['name']}")
        pod.delete()

        show_step(2)
        tf_manager.wait_casasndracluster_status(status="Pending", timeout=180)
        tf_manager.wait_casasndracluster_status(status="Running")

        show_step(3)
        ClusterCheckMosManager.check_cassandra_nodes_config(os_manager=os_manager)

        show_step(4)
        ClusterCheckMosManager.check_contrail_api_readiness(openstack_client_manager)

        show_step(5)
        assert ClusterCheckMosManager.is_lb_functional(openstack_client_manager, 2, self.lb_url)

    @pytest.mark.usefixtures('mos_tf_api_loadtest')
    def test_drop_data_cassandra_db_pod(self, show_step, tf_manager, os_manager, openstack_client_manager):
        """ Check Cassandra database restoration after deletion of persistent volume (PV) data.

        Scenario:
            1. Delete pvc one of cassandra pod and restart it
            2. Wait cassandra cluster become ready
            3. Check cassandra db status
            4. Check tf-api readiness
            5. Check OpenStack workload (TF LB) functionality of created heat stack (Setup class)
        """

        show_step(1)
        pods = tf_manager.get_cassandra_pods()
        pod = random.choice(pods)
        vol = next(filter(lambda vol: vol['name'] == "data", pod.data['spec']['volumes']))
        pvc_name = vol['persistent_volume_claim']['claim_name']
        pvc = tf_manager.api.pvolumeclaims.get(namespace=tf_manager.tf_namespace, name=pvc_name)
        LOG.info(f"Delete pvc {pvc.data['metadata']['name']} and restart {pod.data['metadata']['name']} pod")
        pvc.delete()
        pod.delete()

        show_step(2)
        tf_manager.wait_casasndracluster_status(status="Pending", timeout=180)
        tf_manager.wait_casasndracluster_status(status="Running")

        show_step(3)
        ClusterCheckMosManager.check_cassandra_nodes_config(os_manager=os_manager, actualize_nodes_config=True)

        show_step(4)
        ClusterCheckMosManager.check_contrail_api_readiness(openstack_client_manager)

        show_step(5)
        assert ClusterCheckMosManager.is_lb_functional(openstack_client_manager, 2, self.lb_url)


@pytest.mark.usefixtures('tf_svc_cassandra_cql')
class TestTFDBManageScript(object):
    # Workload details
    lb_url = None
    network_name = None
    # svc for cql
    svc_cql = None

    @staticmethod
    def _set_cls_attr(request, stack_outputs, attr):
        for output in stack_outputs:
            if output['output_key'] == attr:
                setattr(request.cls, attr, output['output_value'])

    @pytest.fixture(scope='class')
    def check_requirements(self, tf_manager):
        tfoperator_version = tf_manager.tfoperator_version()
        if version.parse(tfoperator_version) < version.parse("0.18.1"):
            pytest.skip(f"TFOperator version {tfoperator_version} is too low")

    @pytest.fixture(scope='class', autouse=True)
    def setup_cls(self, request, check_requirements, openstack_client_manager, tf_svc_cassandra_cql):
        setattr(request.cls, 'svc_cql', tf_svc_cassandra_cql)
        LOG.info("Create heat stack with loadbalancer before Cassandra cluster tests")
        stack = ClusterCheckMosManager.created_stack_tf_lb(request, openstack_client_manager)
        self._set_cls_attr(request, stack['outputs'], 'lb_url')
        self._set_cls_attr(request, stack['outputs'], 'network_name')
        assert ClusterCheckMosManager.is_lb_functional(openstack_client_manager, 2, self.lb_url)

    @pytest.fixture(scope='function', autouse=True)
    def setup_function(self):
        # Valid for TF 24.1
        self.python_version = 'python3'
        self.script = '/usr/lib/python3.6/site-packages/vnc_cfg_api_server/db_manage.py'

    def test_heal_deleted_data(self, show_step, tf_manager, openstack_client_manager):
        """ Check heal option of db_manage.py script.

        Scenario:
            1. Get test objects in Cassandra DB
            2. Find and delete data in obj_fq_name_table
            3. Run db_manage.py heal script to find an issue
            4. Run db_manage.py heal script to resolve the issue (--execute flag)
            5. Verify deleted data was restored
            6. Check related TF objects is functional
        """
        show_step(1)
        network = openstack_client_manager.exec_os_cli(f"network show {self.network_name}", deserialize=True)
        assert network

        show_step(2)
        session = tf_manager.cql_session(self.svc_cql.data['spec']['cluster_ip'])

        fq_name = ':'.join(network['fq_name'])
        value = f"{fq_name}:{network['id']}"
        stmt_vn = session.prepare(
            f"SELECT blobAsText(key), blobAsText(column1) FROM config_db_uuid.obj_fq_name_table "
            f"WHERE key = textAsBlob('virtual_network') AND column1 = textAsBlob('{value}')"
        )
        row = session.execute(stmt_vn)
        LOG.info(row.one())
        assert len(row[:]) == 1, f"Record with {value} wasn't found in DB"

        session.execute(
            f"DELETE FROM config_db_uuid.obj_fq_name_table "
            f"WHERE key = textAsBlob('virtual_network') AND column1 = textAsBlob('{value}')"
        )
        row = session.execute(stmt_vn)
        LOG.info(row.one())
        assert len(row[:]) == 0, f"Record with {value} still present in DB"

        show_step(3)
        pods = tf_manager.get_tf_config_pods()
        pod = random.choice(pods)

        cmd = [self.python_version, self.script, 'heal']
        output = pod.exec(cmd)
        LOG.info(output)
        assert fq_name in output, "Deleted entry wasn't detected by script"

        row = session.execute(stmt_vn)
        LOG.info(row.one())
        assert len(row[:]) == 0, f"Record with {value} was restored in DB without --executed flag"

        show_step(4)
        cmd.insert(-1, "--execute")
        output = pod.exec(cmd)
        LOG.info(output)

        show_step(5)
        row = session.execute(stmt_vn)
        LOG.info(row.one())
        assert len(row[:]) == 1, f"Record with {value} wasn't restored in DB"

        show_step(6)
        assert ClusterCheckMosManager.is_network_present(openstack_client_manager, self.network_name)
        assert ClusterCheckMosManager.is_lb_functional(openstack_client_manager, 2, self.lb_url)

    def test_clean_stale_fq_names(self, show_step, tf_manager, openstack_client_manager):
        """ Check heal option of db_manage.py script.

        Scenario:
            1. Insert data in obj_fq_name_table
            2. Run db_manage.py clean_stale_fq_names script to find an issue
            3. Run db_manage.py clean_stale_fq_names script to resolve the issue (--execute flag)
            4. Verify added data was deleted
        """
        show_step(1)
        session = tf_manager.cql_session(self.svc_cql.data['spec']['cluster_ip'])

        value = 'default-domain:default-project:non-existent-net:01234567-890a-bcde-0000-00000000000'
        stmt_insert = session.prepare(
            f"INSERT INTO config_db_uuid.obj_fq_name_table (key, column1, value) VALUES (textAsBlob('virtual_network')"
            f", textAsBlob('{value}'), 'null')"
        )
        session.execute(stmt_insert)

        stmt_check = session.prepare(
            f"SELECT blobAsText(key), blobAsText(column1) FROM config_db_uuid.obj_fq_name_table "
            f"WHERE key = textAsBlob('virtual_network') AND column1 = textAsBlob('{value}')"
        )
        row = session.execute(stmt_check)
        LOG.info(row.one())
        assert len(row[:]) == 1, f"Record with {value} wasn't found in DB"

        show_step(2)
        pods = tf_manager.get_tf_config_pods()
        pod = random.choice(pods)

        cmd = [self.python_version, self.script, 'clean_stale_fq_names']
        output = pod.exec(cmd)
        LOG.info(output)
        assert value in output, "Added entry wasn't detected by script"

        show_step(3)

        cmd.insert(-1, "--execute")
        output = pod.exec(cmd)
        LOG.info(output)

        show_step(4)

        stmt_check = session.prepare(
            f"SELECT blobAsText(key), blobAsText(column1) FROM config_db_uuid.obj_fq_name_table "
            f"WHERE key = textAsBlob('virtual_network') AND column1 = textAsBlob('{value}')"
        )
        row = session.execute(stmt_check)
        LOG.info(row.one())
        assert len(row[:]) == 0, f"Record with {value} still present in DB"
