import yaml
from yaml import scanner as yaml_scanner
from si_tests.utils import utils, waiters
from si_tests import logger, settings
from si_tests.managers.tungstenfafric_manager import TFManager

LOG = logger.logger


def test_tf_backup_resotre(kaas_manager, show_step):
    """ Automatically back up and restore TF data

    Scenario:
        1. Get openstack client pod
        2. Remove test network and tfdbrestore before test if needed
        3. Create network before backup for comparing in database between backup and restore
        4. Enable backup job via TF operator CR
        5. Wait for backup job done
        6. Disable backup job via TF operator CR
        7. Attach network tag after backup for comparing in database between backup and restore
        8. Enable restore via TF operator CR and wait for restore process to be completed
        9. Wait for contrail api readiness
        10. Check for network exist
        11. Check for tags is not exist
    """

    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    operator_name = 'openstack-tf'
    operator_ns = 'tf'
    ns = kaas_manager.get_namespace(namespace_name)
    child_cluster = ns.get_cluster(cluster_name)

    # Write child kubeconfig
    child_kubeconfig_name, child_kubeconfig = child_cluster.get_kubeconfig_from_secret()
    with open('child_conf', 'w') as f:
        f.write(child_kubeconfig)
    tf_manager = TFManager(kubeconfig='child_conf')

    client_pods = child_cluster.k8sclient.pods.list(namespace="openstack", name_prefix='keystone-client')
    show_step(1)
    assert len(client_pods) > 0, "No pods found with prefix keystone-client in namespace openstack"
    client_pod = client_pods[0]
    network_name = 'tf-database-test-net'
    tag_name = 'tf-database-test-tag'
    cmd_net_list = ['/bin/sh', '-c', 'PYTHONWARNINGS=ignore::UserWarning '
                                     'openstack network list -f yaml']
    cmd_remove_network = ['/bin/sh', '-c', f'PYTHONWARNINGS=ignore::UserWarning '
                                           f'openstack network delete {network_name}']

    show_step(2)
    # Check that we don't have tfdbrestore objects before test,
    # https://docs.mirantis.com/mosk/latest/single/#restore-tf-data
    tfdbrestore = [c for c in child_cluster.k8sclient.tfdbrestores_v2.list(namespace=operator_ns)]
    if tfdbrestore:
        tfdbrestore[0].delete()
        LOG.info(f'Removing tfdbrestore object before test: {tfdbrestore[0].name}')
    # Check that we don't have networks with same name before test
    existed_networks = yaml.safe_load(client_pod.exec(cmd_net_list))
    networks = [n for n in existed_networks if n.get('Name') == network_name]
    # If we have network with same name then delete it
    if networks:
        LOG.info(f'Network with same name: {network_name} exists -> delete it')
        client_pod.exec(cmd_remove_network)

    # Create network before backup
    cmd_create_net = ['/bin/sh', '-c', f'PYTHONWARNINGS=ignore::UserWarning '
                                       f'openstack network create {network_name}']
    show_step(3)
    client_pod.exec(cmd_create_net)
    tfoperator = child_cluster.k8sclient.tfoperator_v2.get(name=operator_name, namespace=operator_ns)
    LOG.info(f"tf operator name is {tfoperator.name}")
    show_step(4)
    tf_manager.db_backup(state=True)
    job = tf_manager.tfdbbackup_cronjob(read=True)
    job_template = job.spec.job_template
    backup_jobs_prefix = "tf-dbbackup-job-"
    job_template.metadata.name = backup_jobs_prefix + utils.gen_random_string(3)
    child_cluster.k8sclient.jobs.create(body=job_template, namespace=operator_ns)
    show_step(5)

    waiters.wait(lambda: len([j for j in child_cluster.k8sclient.jobs.list(name_prefix=backup_jobs_prefix,
                                                                           namespace=operator_ns)]) > 0,
                 timeout=120, interval=10, timeout_msg="Timeout for waiting job")

    child_cluster.k8sclient.jobs.check_jobs_completed(target_namespaces=operator_ns, jobs_prefix=backup_jobs_prefix)
    show_step(6)
    tf_manager.db_backup(state=False)

    # Attach tag for network after the backup
    cmd_add_tag_for_net = ['/bin/sh', '-c', f'PYTHONWARNINGS=ignore::UserWarning '
                                            f'openstack network set {network_name} --tag {tag_name}']
    show_step(7)
    client_pod.exec(cmd_add_tag_for_net)

    show_step(8)
    tf_manager.db_restore()

    # Wait for contrail api readiness
    show_step(9)

    def check_contrail_api_readiness():
        try:
            yaml.safe_load(client_pod.exec(cmd_net_list))
            return True
        except yaml_scanner.ScannerError as e:
            LOG.warning(f"Got error {e} during parsing command status'")
            return False
    waiters.wait(check_contrail_api_readiness,
                 timeout=300, interval=15, timeout_msg="Timeout: Contrail api is not Ready")

    # Check for network exist
    show_step(10)
    network_list = yaml.safe_load(client_pod.exec(cmd_net_list))
    network_counter = sum(n.get('Name') == network_name for n in network_list)
    assert network_counter == 1, f'Created network before db backups is not equal to 1, ' \
                                 f'we got next number of networks: {network_counter}'

    # Check for tags is not exist
    cmd_tag_in_net = ['/bin/sh', '-c', f'PYTHONWARNINGS=ignore::UserWarning '
                                       f'openstack network show {network_name} -c tags -f yaml']
    show_step(11)
    tags_list = yaml.safe_load(client_pod.exec(cmd_tag_in_net))
    tag_counter = sum(t == tag_name for t in tags_list['tags'])
    assert tag_counter == 0, "Tags should not be exist after restore"
