import pytest
from si_tests import logger
from si_tests import settings

from si_tests.deployments.utils import kubectl_utils
from si_tests.utils import waiters
from si_tests.managers.openstack_manager import OpenStackManager

LOG = logger.logger


def get_reboot_warning(m_list, namespace_name):
    """Get reboot warning
    Function for get nodes warning from cli
    we need check that kubectl display warning for nodes (reboot for example)
    """
    kubectl = kubectl_utils.Kubectl()
    nodes_warning = {}
    out = kubectl.get('machines', '-o yaml', namespace_name).result_yaml
    for i in out['items']:
        name = i['metadata']['name']
        warning = i['status']['providerStatus'].get('warnings', [])
        if name in m_list and not nodes_warning.get(name):
            nodes_warning[name] = ("Scheduled for a reboot" in warning or "Reboot is in progress" in warning)
    LOG.info("machines without reboot warning:\n" + str([k for k, v in nodes_warning.items() if not v]))
    return all(nodes_warning.values())


def get_node_os_and_current_expected_kernel(cluster):
    expected_versions_map_current = cluster.get_expected_kernel_version()
    nodes_os_and_kernel = cluster.get_nodes_kernel_and_tarfs_versions()
    latest_kernels = {
        i["os_version"]: i["kernel"]
        for i in expected_versions_map_current["allowedDistributions"]
        if not i.get("notgreenfield")
    }
    for node, items in nodes_os_and_kernel.items():
        os_version = items['os_version']
        items.pop('dib_datetime', None)
        items['expected_kernel'] = latest_kernels.get(os_version, None)

    return nodes_os_and_kernel


@pytest.mark.usefixtures("introspect_distribution_not_changed")
@pytest.mark.usefixtures("check_heat_stacks_after_test")
@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
@pytest.mark.usefixtures('mcc_loadtest_prometheus')
@pytest.mark.usefixtures('mcc_loadtest_grafana')
@pytest.mark.usefixtures('mcc_loadtest_alerta')
@pytest.mark.usefixtures('mcc_loadtest_kibana')
@pytest.mark.usefixtures('mcc_loadtest_alertmanager')
@pytest.mark.usefixtures('mcc_loadtest_keycloak')
@pytest.mark.usefixtures('create_hoc_before_lcm_and_delete_after')
@pytest.mark.usefixtures('introspect_no_PRODX_51933_after_lcm')
def test_graceful_reboot(kaas_manager, _, show_step):
    """ Graceful reboot of all machines of the cluster
    Scenario:
        1. Check init state cluster
        2. Check current and available kernel version
        3. Create Graceful Reboot Request
        4. Wait for nodes warning: 'Scheduled for a reboot' or 'Reboot is in progress'
        5. Check all machines are rebooted
        6. Check return cluster to init state
        7. Check expected kernel version
    """
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    show_step(1)
    LOG.info(f"Check init state on the {cluster._cluster_type} cluster {cluster.namespace}/{cluster.name}")
    cluster.check.check_machines_status()
    cluster.check.check_cluster_readiness()
    cluster.check.check_k8s_nodes()
    cluster.check.wait_graceful_reboot_request(expected_status=False)

    show_step(2)
    if settings.ROLLING_REBOOT_MACHINES_KERNEL_CHECK:
        nodes_os_and_kernels_map = get_node_os_and_current_expected_kernel(cluster)
        if any(
                details['kernel'] != details['expected_kernel']
                for details in nodes_os_and_kernels_map.values()):
            LOG.info("New kernel available and must be installed during gracefull reboot")
            LOG.info(nodes_os_and_kernels_map)
            kernel_update = True
        else:
            LOG.info("New kernel not available")
            kernel_update = False
    else:
        LOG.info(f"New kernel check is skipped due to "
                 f"ROLLING_REBOOT_MACHINES_KERNEL_CHECK is {settings.ROLLING_REBOOT_MACHINES_KERNEL_CHECK}")

    show_step(3)
    LOG.info(f"Creating Graceful Reboot Request for all machines in cluster {cluster.namespace}/{cluster.name}")

    m_list = []
    lcm_list = []
    if settings.ROLLING_REBOOT_MACHINES_LABELS:
        for machine in cluster.get_machines():
            labels = machine.data.get('metadata', {}).get('labels', {})
            if set(settings.ROLLING_REBOOT_MACHINES_LABELS).issubset(set(labels.keys())):
                m_list.append(machine.name)
                lcm_list.append(machine.lcmmachine)
        if not m_list or not lcm_list:
            msg = (f"There are no machines with {settings.ROLLING_REBOOT_MACHINES_LABELS} labels to proceed."
                   " Stopping test")
            pytest.skip(msg)
    else:
        m_list = cluster.get_machines_names()
    ns.create_gracefulrebootrequest_object(cluster.name, namespace_name, m_list)
    show_step(4)
    waiters.wait(
        lambda: get_reboot_warning(m_list, namespace_name),
        timeout=3600, interval=10,
        timeout_msg="Wait for 'reboot' warning for all machines")

    cluster.check.wait_graceful_reboot_request(expected_status=True)

    show_step(5)
    LOG.info('Waiting for all machines to reboot')
    if settings.ROLLING_REBOOT_MACHINES_LABELS:
        boot_time_dict = cluster.get_boot_time_dict(exclude_bastion=True, lcm_machines=lcm_list)
    else:
        boot_time_dict = cluster.get_boot_time_dict(exclude_bastion=True)
    machines_number = len(boot_time_dict.keys())
    # Rebooting BM machines takes about 10-15 minutes, but sometimes may take 25+ minutes
    machines_reboot_timeout = 1800 * machines_number
    cluster.check.wait_machines_reboot(boot_time_dict, timeout=machines_reboot_timeout)

    show_step(6)
    LOG.info(f"Check cluster {cluster.namespace}/{cluster.name} for init state")
    cluster.check.wait_graceful_reboot_request(expected_status=False, timeout=600)
    cluster.check.check_machines_status()
    cluster.check.check_cluster_readiness()
    cluster.check.check_diagnostic_cluster_status()

    if cluster.is_mosk:
        LOG.info("Check OpenStack resources")
        child_kubeconfig_name, child_kubeconfig = cluster.get_kubeconfig_from_secret()
        with open('child_conf', 'w') as f:
            f.write(child_kubeconfig)
        os_manager = OpenStackManager(kubeconfig='child_conf')
        if cluster.tf_enabled():
            cluster.mos_check.check_cassandra_nodes_config(
                os_manager=os_manager,
                actualize_nodes_config=settings.TF_CASSANDRA_NODES_CLENAUP)

    if settings.ROLLING_REBOOT_MACHINES_KERNEL_CHECK:
        show_step(7)
        if kernel_update:
            nodes_os_and_kernels_new_map = get_node_os_and_current_expected_kernel(cluster)
            if all(
                    details['kernel'] == details['expected_kernel']
                    for details in nodes_os_and_kernels_new_map.values()):
                LOG.info("Kernel was updated during reboot")
                LOG.info(nodes_os_and_kernels_new_map)
            else:
                LOG.error(f"Some nodes contain old kernel after reboot: {nodes_os_and_kernels_new_map}")
                raise Exception("Failed to update kernel during rolling reboot")

    # Check runtime after reboot
    if settings.DESIRED_RUNTIME:
        machines = cluster.get_machines()
        cluster.check.compare_machines_runtime_with_desired(machines)
