import pytest

from si_tests import logger, settings
from si_tests.managers import openstack_manager, tungstenfafric_manager

LOG = logger.logger

cluster_name = settings.TARGET_CLUSTER
namespace_name = settings.TARGET_NAMESPACE
tfoperator_name = settings.TF_OPERATOR_CR_NAME


@pytest.mark.usefixtures('log_method_time')
@pytest.mark.usefixtures("collect_downtime_statistics")  # Should be used if ALLOW_WORKLOAD == True
def test_replace_failed_tf_control(kaas_manager, show_step):
    """Replace TF control plane node after the outage.

    Scenario:
        1. Get info about replace node
        2. Collect tf services for the cleanup after the replacement
        3. Simulate power outage (Hard power off)
        4. Wait for NotReady node state
        5. Replace the powered off TF control Machine and it's BMH/BMHI
        6. Check Machines and Nodes
        7. Cleanup old node TF data (pvc, pv, configdb, analyticsdb, nodes config)
        8. Force delete all pods and PVC from replaced node
        9. Check child cluster conditions
        10. Check openstack resources
    """

    # Get namespace
    LOG.info(f"Namespace name: {namespace_name}")
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    # Check or set customHostnamesEnabled flag in Cluster object to match settings.CUSTOM_HOSTNAMES
    cluster.set_custom_hostnames_enabled(flag=settings.CUSTOM_HOSTNAMES)

    # Write kubeconfig
    child_kubeconfig_name, child_kubeconfig = cluster.get_kubeconfig_from_secret()
    with open('child_conf', 'w') as f:
        f.write(child_kubeconfig)
    tf_manager = tungstenfafric_manager.TFManager(kubeconfig='child_conf')
    os_manager = openstack_manager.OpenStackManager(kubeconfig='child_conf')

    show_step(1)

    # get cluster machine (outage candidate)
    machines = cluster.get_machines(machine_status='Ready')
    tf_control = []

    for w in machines:
        if ("si-role/replacetfcontrol" in w.machinelabels and
                [n for n in w.nodeLabels if 'tfcontrol' in n.values()]):
            tf_control.append(w)

    assert len(tf_control) == 1, (
        f"BM Machine not found on the cluster, "
        f"or Machine not in Ready status. Found {len(tf_control)} Machines: {tf_control}")
    target_node = tf_control[0]

    k8s_node = target_node.get_k8s_node()
    LOG.info(f"K8s node: {k8s_node.name}")
    old_node_k8s_name = k8s_node.name

    show_step(2)
    tf_data = cluster.day2operations.collect_tf_data_before_replace(target_node)

    # Get bmh and power off
    # Wait form k8s node status NotReady
    # Remove all labels from k8s node
    show_step(3)
    target_node.set_baremetalhost_power(online=False)

    show_step(4)
    LOG.info("Delete, and start waiting k8s node status `NotReady`")
    cluster.check.wait_k8s_node_status(old_node_k8s_name, expected_status='NotReady')

    show_step(5)
    new_machine = cluster.day2operations.replace_baremetal_machine_and_bmh(
        target_node, machine_deletion_policy="unsafe")

    show_step(6)
    # Waiting for cluster,pods,hb are Ready after machines were added
    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()
    cluster.check.check_k8s_nodes()
    cluster.check.check_helmbundles()

    # Check machine runtime
    if settings.DESIRED_RUNTIME:
        machines = cluster.get_machines()
        replaced_machine_for_runtime_check = [machine for machine in machines
                                              if "si-role/replacetfcontrol" in machine.machinelabels][0]
        cluster.check.compare_machines_runtime_with_desired([replaced_machine_for_runtime_check])

    # Check that Machine hostname is created with respect to Cluster flag 'customHostnamesEnabled'
    cluster.check.check_custom_hostnames_on_machines(machines=[cluster.get_machine(name=(new_machine.name))])

    new_node_k8s_name = new_machine.get_k8s_node().name
    LOG.info(f"New node k8s name: {new_node_k8s_name}")

    show_step(7)
    cluster.day2operations.tf_data_cleanup(tf_data, old_node_k8s_name)

    show_step(8)  # Force delete all pv/pvc from old node
    cluster.day2operations.cleanup_pods_and_pvc_from_k8s_node(old_node_k8s_name)

    show_step(9)
    cluster.check.check_cluster_readiness()
    cluster.check.check_k8s_pods()
    cluster.check.check_deploy_stage_success()
    cluster.check.check_diagnostic_cluster_status()

    LOG.info("Check TFOperator health status")
    tf_manager.wait_tfoperator_healthy(timeout=900)
    tf_manager.wait_tf_controllers_healthy()

    show_step(10)
    LOG.info("Check Openstack resources statuses")
    os_manager.wait_all_osdpl_children_status()
    os_manager.wait_openstackdeployment_health_status()
    os_manager.wait_all_os_pods()

    cluster.check.check_bmh_inventory_presense()
