# Copyright 2025 Mirantis, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

import pytest
from si_tests import logger, settings
from si_tests.utils import waiters
from si_tests.managers.netchecker_manager import NetcheckerManager

LOG = logger.logger

icm_name = "test-create-infraconnectivitymonitor-test-toggle"


@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
def test_create_infraconnectivitymonitor_bm(kaas_manager, _, show_step):
    """This test should JUST enable(create InfraConnectivityMonitor) Netchecker feature

    Scenario:
        1. Create a InfraConnectivityMonitor object
        2. Check that corresponding CheckerInventoryConfig has been created
        3. Check that corresponding NetCheckerTargetsConfig object has been created
        4. Check the status of InfraConnectivityMonitor object
    """
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)
    netchecker = NetcheckerManager(cluster)

    if not cluster.netchecker_enabled:
        skip_msg = (f"Netchecker is disabled in Cluster object {cluster_name}."
                    f"Check 'spec.providerSpec.value.disableInfraConnectivityMonitor'")
        pytest.skip(skip_msg)

    icm_data = {
        "apiVersion": "kaas.mirantis.com/v1alpha1",
        "kind": "InfraConnectivityMonitor",
        "metadata": {
            "name": icm_name,
            "namespace": namespace_name,
        },
        "spec": {
            "targetCluster": cluster_name,
        }
    }
    show_step(1)
    LOG.info(f"Create InfraConnectivityMonitor object {icm_data}")
    icm = ns.create_infraconnectivitymonitor_raw(icm_data)
    #
    show_step(2)
    cic_namespace = netchecker.netchecker_namespace
    cic_name = netchecker.inventory_config_name
    cic_creation_timeout = 60 * 3
    cic_creation_interval = 5
    cic_creation_timeout_msg = (f"CheckerInventoryConfig object {cic_name} in "
                                f"namespace {cic_namespace} has not been created")
    waiters.wait(lambda: bool(cluster.k8sclient.cnnc_checkerinventoryconfigs.present(name=cic_name,
                                                                                     namespace=cic_namespace)),
                 timeout=cic_creation_timeout,
                 interval=cic_creation_interval,
                 timeout_msg=cic_creation_timeout_msg)
    #
    show_step(3)
    nctc_namespace = netchecker.netchecker_namespace
    nctc_name = netchecker.targets_config_name
    nctc_creation_timeout = 120
    nctc_creation_interval = 5
    nctc_creation_timeout_msg = (f"NetCheckerTargetsConfig object {nctc_name} in "
                                 f"namespace {nctc_namespace} has not been created")
    waiters.wait(lambda: bool(cluster.k8sclient.cnnc_netcheckertargetsconfigs.present(name=nctc_name,
                                                                                      namespace=nctc_namespace)),
                 timeout=nctc_creation_timeout,
                 interval=nctc_creation_interval,
                 timeout_msg=nctc_creation_timeout_msg)

    show_step(4)
    netchecker.wait_compare_targets(netchecker_obj_name=icm_name)
    netchecker.wait_infraconnectivitymonitor_status(icm)


@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
def test_delete_infraconnectivitymonitor_bm(kaas_manager, _, show_step):
    """This test should JUST disable(delete InfraConnectivityMonitor) Netchecker feature

    Scenario:
        1. Delete a InfraConnectivityMonitor object
    """
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    if not cluster.netchecker_enabled:
        skip_msg = (f"Netchecker is disabled in Cluster object {cluster_name}."
                    f"Check 'spec.providerSpec.value.disableInfraConnectivityMonitor'")
        pytest.skip(skip_msg)

    LOG.info(f"Get InfraConnectivityMonitor by name '{icm_name}'")
    icm = ns.get_infraconnectivitymonitor(name=icm_name)
    show_step(1)
    LOG.info(f"Deleting InfraConnectivityMonitor '{icm.name}'")
    icm.delete(async_req=True)
    icm_deletion_timeout = 60
    icm_deletion_interval = 5
    icm_deletion_timeout_msg = f"Deletion failed, icm with name '{icm_name}' still exists"
    waiters.wait(lambda: len([icm for icm in ns.get_infraconnectivitymonitors() if icm_name in icm.name]) == 0,
                 timeout=icm_deletion_timeout,
                 interval=icm_deletion_interval,
                 timeout_msg=icm_deletion_timeout_msg)
