import pytest
import json
import yaml
from si_tests import logger
from si_tests import settings
from si_tests.utils import waiters
from si_tests.utils import templates
from si_tests.utils import utils
LOG = logger.logger


def _check_waiting_reason(pod):
    pod_status = pod.data.get('status') or {}
    pod_cont_statuses = pod_status.get(
        'container_statuses') or [{}]
    cont_statuses_state = pod_cont_statuses[0].get(
        'state') or {}
    cont_waiting = cont_statuses_state.get('waiting') or {}
    reason = cont_waiting.get('reason') or ''
    return reason


def check_pod_create(cluster, base_image: str, negative_flag=False):
    nodes = cluster.get_machines()
    ns = f'test-ns-{utils.gen_random_string(6)}'
    pod_name = 'test-pod'
    LOG.info(f'Create "{ns}" namespace')
    namespace = cluster.k8sclient.namespaces.create(
        name=ns,
        body={'metadata': {'name': ns}})

    try:
        for n in nodes:
            node_labels = n.get_k8s_node().read().metadata.labels
            node_name = node_labels['kubernetes.io/hostname']
            pod_template = templates.render_template(
                settings.CONTAINERREGISTRY_CHECK_POD_YAML,
                {'POD_NAME': pod_name,
                 'NODE_LABEL': node_name,
                 'CONTAINERREGISTRY_IMAGE': base_image})
            pod_json_body = json.dumps(
                yaml.load(pod_template, Loader=yaml.SafeLoader))
            LOG.info(f'Create pod on {node_name} node ({n.name} machine)')
            pod = cluster.k8sclient.pods.create(name=pod_name,
                                                namespace=ns,
                                                body=json.loads(
                                                     pod_json_body))
            if negative_flag:
                pod.wait_phase(phases='Pending', timeout=180, interval=5)

                waiters.wait(
                    lambda p=pod: (_check_waiting_reason(p) in ['ErrImagePull', 'ImagePullBackOff']),
                    timeout=120,
                    interval=10,
                    timeout_msg="The custom image was installed without "
                                "СontainerRegistry on the cluster"
                )

            else:
                pod.wait_phase(phases='Running', timeout=360, interval=10)
                assert pod.exists(), (
                    "The custom image was not installed with СontainerRegistry on the cluster"
                )

            LOG.info('Delete pod')
            pod.delete()
            assert not pod.exists()

    finally:
        LOG.info('Delete namespace')
        namespace.delete()


def get_container_registry_setting(machines, expected_container_name):
    result = {}
    for machine in machines:
        logs = machine.exec_pod_cmd(
            "ls /etc/docker/certs.d")['logs']
        result[machine.name] = expected_container_name in logs
    return result


@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('log_method_time')
@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
def test_mcc_conteiner_registry(kaas_manager, _, show_step):
    """Custom ContainerRegistries test

    Scenario:
            1. Check that all needed data were passed into test
            2. Check if  cluster has a ContainerRegistries
            3. If it has, delete ContainerRegistries, verify deletion
            4. Negative check funcional test on pods
            5. Add new ContainerRegistries
            6. Update cluster with new ContainerRegistries
            7. Funcional test on pods
            8. Delete ContainerRegistries, verify deletion
            9. Check cluster
    """
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    target_cluster = ns.get_cluster(cluster_name)
    machines = target_cluster.get_machines()
    datetime_cmd = "date -u +'%Y-%m-%d %H:%M:%S'"

    show_step(1)
    assert settings.KAAS_REGISTRYOBJECT_DOMAIN, (
        "value KAAS_REGISTRYOBJECT_DOMAIN is not defined")
    assert settings.KAAS_REGISTRYOBJECT_CERT, (
        "value KAAS_REGISTRY_OBJECT_CERT is not defined")

    show_step(2)
    # Check if cluster has a ContainerRegistries
    pr_spec = target_cluster.spec.get('providerSpec').get('value')
    LOG.info(f"Check if cluster has ContainerRegistries {pr_spec}")
    time_before_update = machines[0].exec_pod_cmd(datetime_cmd)['logs']
    if pr_spec.get('containerRegistries', {}):
        show_step(3)
        LOG.info("Update cluster spec with No ContainerRegistries")
        target_cluster.update_cluster_container_registry()
        waiters.wait(
            lambda: all(target_cluster.check.check_expected_phases_on_machines(
                machines, time_before_update, 'reconfigure')) is True,
            timeout=600,
            interval=20,
            timeout_msg="Reconfigure phase started not on all machines",
        )
        target_cluster.check.check_machines_status("Ready")
        LOG.info(f"Cluster has been updated {target_cluster.name}")

    show_step(4)
    # Negative check
    container_registry_setting = get_container_registry_setting(
        machines, settings.KAAS_REGISTRYOBJECT_DOMAIN)

    assert not any([v for _, v in container_registry_setting.items()]), (
        f"Some machines still have proxy"
        f"'{settings.KAAS_REGISTRYOBJECT_DOMAIN}' "
        f"in ls /etc/docker/certs.d after container registry  deletion: {container_registry_setting}"
    )

    check_pod_create(cluster=target_cluster,
                     base_image=settings.CONTAINERREGISTRY_BASE_IMAGE,
                     negative_flag=True)

    show_step(5)
    container_registry_name = f"{cluster_name}-{settings.KAAS_REGISTRYOBJECT_NAME}"
    existed_registry_objects = kaas_manager.get_containerregistry(namespace=namespace_name)
    create_new_registry = True
    for registry in existed_registry_objects:
        if registry.namespace == namespace_name and registry.registry_domain == settings.KAAS_REGISTRYOBJECT_DOMAIN:
            LOG.info(f"Registry with domain {registry.registry_domain} in namespace {namespace_name} already "
                     f"exists. Using this registry")
            container_registry_name = registry.name
            create_new_registry = False
            break

    if create_new_registry:
        LOG.info("Create СontainerRegistry object")
        container_registry_object = ns.create_registryobject(
            name=container_registry_name,
            container_registry_doman=settings.KAAS_REGISTRYOBJECT_DOMAIN,
            container_registry_cert=settings.KAAS_REGISTRYOBJECT_CERT
        )
        LOG.info(f"СontainerRegistry object {container_registry_object} created")

    # Add ContainerRegistry
    show_step(6)
    LOG.info("Add СontainerRegistry to cluster")
    time_before_update = machines[0].exec_pod_cmd(datetime_cmd)['logs']
    target_cluster.update_cluster_container_registry([container_registry_name])
    waiters.wait(
        lambda: all(target_cluster.check.check_expected_phases_on_machines(
            machines, time_before_update, 'reconfigure')) is True,
        timeout=1800,
        interval=30,
        timeout_msg="Reconfigure phase started not on all machines",
    )
    target_cluster.check.check_machines_status("Ready")
    LOG.info(f"Cluster update {target_cluster.name}")
    container_registry_setting = get_container_registry_setting(
        machines, settings.KAAS_REGISTRYOBJECT_DOMAIN)
    assert all([v for _, v in container_registry_setting.items()]), (
        f"Some machines don't container registry"
        f"'{settings.KAAS_REGISTRYOBJECT_DOMAIN}' "
        f"in ls /etc/docker/certs.d after adding container registry: {container_registry_setting}")

    show_step(7)
    check_pod_create(cluster=target_cluster,
                     base_image=settings.CONTAINERREGISTRY_BASE_IMAGE)
    show_step(8)
    time_before_update = machines[0].exec_pod_cmd(datetime_cmd)['logs']
    target_cluster.update_cluster_container_registry()
    waiters.wait(
        lambda: all(target_cluster.check.check_expected_phases_on_machines(
            machines, time_before_update, 'reconfigure')) is True,
        timeout=1800,
        interval=30,
        timeout_msg="Reconfigure phase started not on all machines",
    )
    target_cluster.check.check_machines_status("Ready")
    container_registry_setting = get_container_registry_setting(
        machines, settings.KAAS_REGISTRYOBJECT_DOMAIN)

    assert not any([v for _, v in container_registry_setting.items()]), (
        f"Some machines still have proxy"
        f"'{settings.KAAS_REGISTRYOBJECT_DOMAIN}' "
        f"in ls /etc/docker/certs.d after container registry  deletion: {container_registry_setting}"
    )
    LOG.info(f"Cluster has been updated {target_cluster.name}")

    show_step(9)
    target_cluster.check.check_k8s_nodes()
    target_cluster.check.check_machines_status()
    target_cluster.check.check_cluster_readiness()
    target_cluster.check.check_deploy_stage_success()
    LOG.info(f"Cluster update {target_cluster.name}")
