import calendar
from datetime import datetime, timedelta
import pytest
import time
import yaml
from si_tests import logger
from si_tests.managers import bootstrap_manager
from si_tests import settings
from si_tests.utils import waiters

LOG = logger.logger


def autoupdate_mgmt_cluster(kaas_manager, show_step):
    cluster = kaas_manager.get_mgmt_cluster()
    r_clusters = kaas_manager.get_regional_clusters()
    # Get the current mgmt clusterrelease version until update is not started
    unexpected_message = 'Unexpected message for favorite schedule'

    def check_message():
        if cluster.workaround.prodx_47277():
            assert (
                    "New release scheduled for upgrade" in
                    kaas_manager.get_kaas_mcc_upgrade_status_condition(
                        'mcc-upgrade', 'message') and
                    kaas_manager.get_kaas_mcc_upgrade_status_condition(
                        'mcc-upgrade', 'reason') == 'ReleaseScheduled'
            ), unexpected_message
        else:
            assert (
                    "not allowed by the schedule" in
                    kaas_manager.get_kaas_mcc_upgrade_status('mcc-upgrade')[
                        'message']
            ), unexpected_message

    def change_weekday():
        if tomorrow_weekday_flag:
            schedule['weekdays'][str.lower(tomorrow_weekday)] = True
        else:
            schedule['weekdays'][str.lower(weekday_now)] = True

        kaas_manager.update_mccupgrade(name='mcc-upgrade', schedule=schedule)
        LOG.info(
            f"Updated mccupgrades crd: \n{yaml.dump(kaas_manager.get_kaas_mcc_upgrade('mcc-upgrade').data)}")
        check_message()

    show_step(1)
    LOG.info(f"Check init state on the {cluster._cluster_type} cluster {cluster.namespace}/{cluster.name}")
    cluster.check.check_cluster_nodes()
    cluster.check.check_machines_status()
    cluster.check.check_cluster_readiness()
    cluster.check.check_k8s_nodes()

    for r_cluster in r_clusters:
        LOG.info(f"Check init state on the {r_cluster._cluster_type} cluster {r_cluster.namespace}/{r_cluster.name}")
        r_cluster.check.check_cluster_nodes()
        r_cluster.check.check_machines_status()
        r_cluster.check.check_cluster_readiness()
        r_cluster.check.check_k8s_nodes()

    show_step(2)
    cr_before = kaas_manager.get_mgmt_cluster().clusterrelease_version
    LOG.info("Management cluster clusterrelease version is "
             "{}".format(cr_before))

    ucp_tag_in_cr_before = set(
        [x['params']['ucp_tag'] for x in kaas_manager.get_clusterrelease(
            cr_before).data['spec']['machineTypes']['control']
         if 'ucp_tag' in x['params'].keys()])
    LOG.info("Current management cluster ucp version is "
             "{}".format(ucp_tag_in_cr_before))

    # Check schedule changes
    show_step(3)

    LOG.info(
        f"Initial state mccupgrade crd:\n{yaml.dump(kaas_manager.get_kaas_mcc_upgrade('mcc-upgrade').data)}")
    log_message = kaas_manager.get_kaas_mcc_upgrade_status_condition(
        'mcc-upgrade', 'message') \
        if cluster.workaround.prodx_47277() \
        else kaas_manager.get_kaas_mcc_upgrade_status('mcc-upgrade')['message']

    LOG.info(f"Message from crd: {log_message}")
    check_message()

    ex_release_version = kaas_manager.get_kaas_mcc_upgrade_status('mcc-upgrade')['nextRelease']['version']
    LOG.info(f"Expected release to be installed: {ex_release_version}")

    show_step(4)
    date = datetime.now()
    hour_now = date.hour
    weekday_now, yesterday_weekday = (calendar.day_name[date.weekday()]), (calendar.day_name[(date.weekday() - 1) % 7])
    tomorrow_weekday = (calendar.day_name[(date.weekday() + 1) % 7])
    tomorrow_weekday_flag = False

    if hour_now == 23:
        tomorrow_weekday_flag = True
        hour_now = 0

    schedule = {
        "hours": {
            "from": hour_now + 2 if cluster.workaround.prodx_47277() else hour_now,
            "to": hour_now + 4,
        },
        "weekdays": {
            str.lower(yesterday_weekday): True,
            str.lower(weekday_now): False,
            str.lower(tomorrow_weekday): False,
        }
    }

    LOG.info("Add hours and weekdays in schedule mccupgrade crd")
    kaas_manager.update_mccupgrade(name='mcc-upgrade', schedule=schedule)
    check_message()

    if cluster.workaround.prodx_47277():
        LOG.info("Update autodelay value to False")
        kaas_manager.update_mccupgrade(name='mcc-upgrade', autodelay=False)

        LOG.info(
            f"State mccupgrade crd:\n{yaml.dump(kaas_manager.get_kaas_mcc_upgrade('mcc-upgrade').data)}")
        check_message()

    else:
        time_schedule = datetime.now()
        LOG.info(f"Time now {time_schedule}")
        update_block_until = time_schedule.isoformat(timespec='seconds')
        LOG.info(f"Update Block until in mccupgrade crd {update_block_until}")
        kaas_manager.update_mccupgrade(name='mcc-upgrade', blockuntil=update_block_until)

        LOG.info("Wait for the expected update not start with schedule options 'next day':true")
        time.sleep(120)
        check_message()

        show_step(5)
        time_schedule = datetime.now() + timedelta(minutes=5)
        LOG.info(f"Time now + delta {time_schedule}")
        update_block_until = time_schedule.isoformat(timespec='seconds')
        LOG.info(f"Update Block until in mccupgrade crd {update_block_until}")
        kaas_manager.update_mccupgrade(name='mcc-upgrade', blockuntil=update_block_until)

        check_message()
        change_weekday()

    show_step(6)
    if not cluster.workaround.prodx_47277():
        exp_time_row = kaas_manager.get_kaas_mcc_upgrade('mcc-upgrade').data['spec']['blockUntil']
        while datetime.strptime(exp_time_row, '%Y-%m-%dT%H:%M:%SZ') >= datetime.now():
            avail_kaas_releases = kaas_manager.get_kaasrelease_names()
            assert len(avail_kaas_releases) == 1, (
                f"There is unexpected kaas releases, current list: {avail_kaas_releases}")
            check_message()
    else:
        schedule['hours']['from'] = hour_now
        LOG.info("Update start time in schedule mccupgrade crd")
        kaas_manager.update_mccupgrade(name='mcc-upgrade', schedule=schedule)
        check_message()
        change_weekday()

    if cluster.workaround.prodx_47277():
        waiters.wait(
            lambda: (
                    "New release scheduled for upgrade" not in
                    kaas_manager.get_kaas_mcc_upgrade_status_condition(
                        'mcc-upgrade', 'message') and
                    kaas_manager.get_kaas_mcc_upgrade_status_condition(
                        'mcc-upgrade', 'reason') == 'UpgradeStarted'
            ),
            timeout=60,
            interval=5,
            timeout_msg=f'The mcc-upgrade crd does not currently display the status of the ongoing update.'
                        f'{log_message}'
        )
    else:
        waiters.wait(
            lambda: (
                    kaas_manager.get_kaas_mcc_upgrade_status('mcc-upgrade')[
                        'message'] not in "not allowed by the schedule"
            ),
            timeout=60,
            interval=5,
            timeout_msg=f'The mcc-upgrade crd does not currently display the status of the ongoing update.'
                        f'{log_message}'
        )

    LOG.info(f"Message from crd: {log_message}")

    LOG.info(
        f"Final mccupgrade crd:\n{yaml.dump(kaas_manager.get_kaas_mcc_upgrade('mcc-upgrade').data)}")

    show_step(7)
    cluster.check.check_cluster_status("Updating", 1800)
    cluster.check.check_cluster_nodes(timeout=settings.KAAS_UPDATE_MGMT_WAIT_TIMEOUT)
    cluster.check.check_k8s_nodes()
    cluster.check.check_machines_status()
    cluster.check.check_cluster_readiness()
    cluster.check.check_upgrade_stage_success()

    kr_data = kaas_manager.get_kaasrelease("kaas-" + ex_release_version.replace(".", "-")).data
    spec_cr = kr_data['spec']['clusterRelease']
    LOG.info(f"Expected cluster release {spec_cr}")

    # Check after upgrade
    show_step(8)
    cluster.check.check_cluster_release(spec_cr)
    if cr_before != spec_cr:
        cluster.check.check_update_finished(check_managed_clusters=True)

    mgmt_cluster = kaas_manager.get_mgmt_cluster().spec['providerSpec']
    cr_after = mgmt_cluster['value']['release']

    assert spec_cr == cr_after, (
        "Cluster version in KaasRelease and Mgmt "
        "Cluster differs, that's wrong")

    LOG.info("Management cluster update clusterrelease version to "
             "{} ".format(spec_cr))

    ucp_tag_after = set(
        [x['params']['ucp_tag']
         for x in kaas_manager.get_clusterrelease(
            spec_cr).data['spec']['machineTypes']['control']
         if 'ucp_tag' in x['params'].keys()])
    if ucp_tag_in_cr_before != ucp_tag_after:
        LOG.info("Management cluster new ucp version will be "
                 "{}".format(ucp_tag_after))

    cluster.check.check_cluster_readiness(timeout=3600)
    # Check/wait for correct docker service replicas in cluster
    cluster.check.check_actual_expected_docker_services()
    cluster.check.check_actual_expected_pods(timeout=600)

    show_step(9)
    for r_cluster in r_clusters:

        r_cluster.check.check_cluster_release(spec_cr)
        r_cluster.check.check_cluster_nodes()
        r_cluster.check.check_cluster_readiness()
        r_cluster.check.check_helmbundles()
        r_cluster.check.check_k8s_nodes()
        r_cluster.check.check_persistent_volumes_mounts()
        r_cluster.check.check_k8s_pods()
        r_cluster.check.check_upgrade_stage_success()

        region_after = r_cluster.spec['providerSpec']['value']['release']
        LOG.info("Regional cluster clusterrelease version is "
                 "{}".format(region_after))
        assert cr_before != region_after, "Clusterrelease version is the same as " \
                                          "the clusterrelease before the upgrade on regional cluster"

        cr_after_regional = r_cluster.spec['providerSpec']['value']['release']
        LOG.info("Regional cluster update clusterrelease version to "
                 "{} ".format(cr_after_regional))

        # Check/wait for correct docker service replicas in cluster
        r_cluster.check.check_actual_expected_docker_services()
        r_cluster.check.check_actual_expected_pods(timeout=600)

    LOG.info("Check clusterRelease objects are existed for every release "
             "announced as supported in kaasRelease object")
    cluster.check.compare_supported_and_existed_clusterreleases()


@pytest.mark.usefixtures("store_updated_mgmt_cluster_description")
def test_mcc_autoupgrade(kaas_manager, mgmt_k8s_ip: str, show_step):
    """Check auto upgrade mgmt cluster

        Scenario:
                1. Check init state clusters
                2. Check mgmt clusterrelease version until update is not started
                3. Check initial state mccupgrade crd
                4. Add days of the week to the schedule mccupgrade
                5. Update time in mccupgrade crd
                6. Check that the update does not start earlier than the planned schedule
                7. Check the transition to the cluster update state
                8. Check mgmt clusterrelease version after update
                9. Check region clusterrelease version after update
        """
    try:
        autoupdate_mgmt_cluster(kaas_manager, show_step)
    except Exception as e:
        LOG.error("Management upgrade failed: {}".format(e))
        raise e
    finally:
        LOG.info('Updating bootstrap version')
        bootstrap = bootstrap_manager.BootstrapManager(seed_ip=settings.SEED_STANDALONE_EXTERNAL_IP)
        bootstrap.update_bootstrap_version(mgmt_k8s_ip)
