import pytest

from si_tests import logger
from si_tests import settings
from si_tests.utils import waiters, utils
from si_tests.managers.kaas_manager import Machine, Manager
from kubernetes.client.rest import ApiException

LOG = logger.logger


class MachineManager:
    MIN_WORKERS_COUNT = 2

    def __init__(self):
        cluster_name = settings.TARGET_CLUSTER
        namespace_name = settings.TARGET_NAMESPACE

        self.kaas_manager = Manager(settings.KUBECONFIG_PATH)
        ns = self.kaas_manager.get_namespace(namespace_name)
        self.cluster = ns.get_cluster(cluster_name)

        machines = self.cluster.get_machines('worker')
        assert len(machines) > 0, "no workers for cluster found"

        self.machine_data = self.cluster.get_existing_machine_spec(machines[0])
        if len(machines) == self.MIN_WORKERS_COUNT:
            LOG.info("Create third machine to speedup tests")
            self._create_machine()

    def _create_machine(self):
        return Machine(
            self.kaas_manager,
            self.cluster,
            self.cluster.create_machine(
                node_type="node",
                region=self.cluster.region_name,
                **self.machine_data))

    def get_machine_with_status(self, status, timeout=1800, interval=30):
        if status is None:
            return self._create_machine()

        self._create_machine()

        if status == 'Ready':
            LOG.info(f"Wait {timeout} seconds until {self.MIN_WORKERS_COUNT + 1} worker machines "
                     f"will be in the status <{status}>")
            waiters.wait(lambda: len(self.cluster.get_machines('worker', 'Ready')) > self.MIN_WORKERS_COUNT,
                         timeout=timeout, interval=interval)

        LOG.info(f"Wait {timeout} seconds until any machine "
                 f"will be in the status <{status}>")

        def _get_machine_with_status():
            machines = self.cluster.get_machines('worker')
            LOG.info("Worker machines status: {}".format({m.name: m.machine_status for m in machines}))

            def _machines_with_status(s):
                return list(filter(lambda m: m.machine_status == s, machines))

            m = _machines_with_status(status)
            if m:
                return m[0]

            no_pending_machines = len(_machines_with_status(None)) == 0
            no_provision_machines = len(_machines_with_status('Provision')) == 0

            # return deploying machine if provisioned to fast
            if status == 'Provision' and no_pending_machines:
                m = _machines_with_status('Deploy')
                if m:
                    return m[0]

            # return preparing machine if deployed to fast
            if status == 'Deploy' and no_pending_machines and no_provision_machines:
                m = _machines_with_status('Prepare')
                if m:
                    return m[0]

            raise Exception(f"No machines with status <{status}>")

        machine = waiters.wait_pass(_get_machine_with_status,
                                    timeout=timeout, interval=interval,
                                    expected=(Exception, ApiException))

        LOG.info(f"{machine.name} is in expected status <{status}>")
        return machine


def is_machine_deletion_policy_enabled():
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE

    ns = Manager(settings.KUBECONFIG_PATH).get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)
    return cluster.machine_deletion_policy_enabled()


machine_deletion_policy_enabled = is_machine_deletion_policy_enabled()
if machine_deletion_policy_enabled:
    machine_manager = MachineManager()


@pytest.mark.skipif(not machine_deletion_policy_enabled, reason="test machine deletion policy disabled")
@pytest.mark.parametrize(
    "machine_status", [None, 'Provision', 'Deploy', 'Ready']
)
@pytest.mark.parametrize(
    'deletion_policy, new_delete_api',
    [
        ('graceful', True),
        ('unsafe', True),
        ('forced', True),
        ('unsafe', False),
        ('forced', False)
    ]
)
def test_machine_deletion_policy(kaas_manager, show_step, deletion_policy, new_delete_api, machine_status):
    """Delete deploying machine in different states.

    Scenario:
        1. Find machine with required state
        2. Delete machine
        3. Wait until machine deleted
    """

    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE

    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    if deletion_policy == "forced" and \
            cluster.provider in [utils.Provider.vsphere, utils.Provider.equinixmetalv2, utils.Provider.baremetal]:
        pytest.skip("skip forced deletion tests, since some left resources may lead to false negative test results")

    show_step(1)
    machine = machine_manager.get_machine_with_status(machine_status)

    machine_name = machine.name
    node_name = None
    try:
        node_name = machine.get_k8s_node_name()
    except Exception as e:
        LOG.info(f"No k8s node found for machine: {e}")

    show_step(2)
    if deletion_policy == "forced":
        LOG.info("Check forced policy forbidden to set initially")
        with pytest.raises(ApiException) as exc_info:
            machine.set_deletion_policy('forced')
        assert exc_info.value.status == 400
        assert exc_info.value.reason == 'Bad Request'
    else:
        machine.set_deletion_policy(deletion_policy)

    machine.delete(check_deletion_policy=True, new_delete_api=new_delete_api)
    if deletion_policy == "forced":
        machine.set_deletion_policy(deletion_policy)

    show_step(3)
    cluster.wait_machine_deletion(machine_name, retries=60)

    if node_name:
        LOG.info(f"Wait k8s node delete - {node_name}")
        waiters.wait(lambda: node_name not in cluster.get_k8s_node_names(),
                     interval=10, timeout=1800)


@pytest.fixture(scope="session", autouse=True)
def check_cluster_nodes(kaas_manager):
    yield

    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE

    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    if cluster.provider == utils.Provider.vsphere:
        for m in cluster.get_machines('worker', 'Provision'):
            m.delete()

    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()
    cluster.check.check_k8s_nodes()
