import pytest
from kombu import Connection
import ssl
import tempfile

from si_tests import logger

LOG = logger.logger
EXTERNAL_RABBITMQ_SERVICE = "rabbitmq-external"
EXTERNAL_RABBITMQ_SECRETS_NAMESPACE = "openstack-external"


def check_connection(username, password, host, port, vhost, ssl=False):
    rabbitmq_url = f"amqp://{username}:{password}@{host}:{port}/{vhost}"
    connection = Connection(rabbitmq_url, ssl=ssl)
    try:
        LOG.info(f"Connecting to the: {rabbitmq_url}")
        connection.ensure_connection(max_retries=3)
        connection.channel()
    except Exception as e:
        LOG.error(f"Connection error. Error: {e}")
        return False
    finally:
        connection.release()
    return True


def get_external_topics(os_manager):
    return (
        os_manager.get_osdpl_deployment()
        .data["spec"]["features"]
        .get("messaging", {})
        .get("notifications", {})
        .get("external", {})
        .get("topics", [])
    )


@pytest.mark.dependency(name="notifications_settings")
def test_external_rabbitmq_notifications_settings(os_manager):
    """Check RabbitMQ external notifications settings
    - check enabled external notifications
    - check external topics
    """
    LOG.info("Check enabled external notifications")
    enabled_ext_notifications = (
        os_manager.get_osdpl_deployment()
        .data["spec"]["features"]
        .get("messaging", {})
        .get("notifications", {})
        .get("external", {})
        .get("enabled", False)
    )
    ext_topics = get_external_topics(os_manager)

    assert enabled_ext_notifications, "RabbitMQ external notifications aren't enabled"
    assert ext_topics, "RabbitMQ has no external topics"
    LOG.info("Rabbitmq external service is enabled")


@pytest.mark.dependency(depends=["notifications_settings"])
def test_external_rabbitmq_notifications_secret_vs_service(os_manager):
    """Compare parameters from secret and service
    - check RabbitMQ service exposed outside
    - get credentials from the external secret
    - comparing parameters of secret vs service
    Parameters required for test execution:
      - KUBECONFIG
    """
    LOG.info("Get rabbitmq external service")
    rabbitmq_ext = os_manager.get_os_service(EXTERNAL_RABBITMQ_SERVICE)
    LOG.info("Rabbitmq external service enabled")
    ext_ip_rabbitmq = rabbitmq_ext.get_external_ip()
    ext_ports_rabbitmq = rabbitmq_ext.get_ports()

    LOG.info(
        f"Rabbitmq external service has external ip: {ext_ip_rabbitmq}, ports: {[p.port for p in ext_ports_rabbitmq]}"
    )

    for ext_topic in get_external_topics(os_manager):
        LOG.info(f"Get credentials from the external secret for topic: {ext_topic}")

        secret_data = os_manager.get_secret_decoded(
            f"openstack-{ext_topic}-notifications", EXTERNAL_RABBITMQ_SECRETS_NAMESPACE
        )
        LOG.info(f"Got credentials for access to external RabbitMQ queue: {secret_data}")

        assert ext_ip_rabbitmq == secret_data["hosts"], "Host mismatch"
        assert all([p.port == int(secret_data[f"port_{p.name}"]) for p in ext_ports_rabbitmq]), "Ports mismatch"
        LOG.info("Service and secret parameters are identical")


@pytest.mark.dependency(depends=["notifications_settings"])
def test_external_rabbitmq_notifications_connection_ssl(os_manager):
    """Check connection to RabbitMQ vhost (ssl)
    Parameters required for test execution:
      - KUBECONFIG
    """
    for ext_topic in get_external_topics(os_manager):
        secret_data = os_manager.get_secret_decoded(
            f"openstack-{ext_topic}-notifications", EXTERNAL_RABBITMQ_SECRETS_NAMESPACE
        )
        certs = {
            "ca_certs": "ca_cert",
            "keyfile": "client_key",
            "certfile": "client_cert",
        }
        # update secret keys with certificates file names
        for k, v in certs.items():
            f = tempfile.NamedTemporaryFile(delete=False)
            certs[k] = f.name
            f.write(secret_data[v].encode("utf-8"))
            f.close()

        LOG.info(f'Check connect to RabbitMQ vhost {secret_data["vhost"]}')
        check_test_connection = check_connection(
            secret_data["username"],
            secret_data["password"],
            secret_data["hosts"],
            secret_data["port_amqp-tls"],
            secret_data["vhost"],
            {**certs, "cert_reqs": ssl.CERT_REQUIRED},
        )

        assert check_test_connection, "Failed connection to RabbitMQ vhost"
        LOG.info("The connection check to the RabbitMQ virtual host is complete")


@pytest.mark.dependency(depends=["notifications_settings"])
def test_external_rabbitmq_notifications_connection_plain(os_manager):
    """Check connection to RabbitMQ vhost (plain)
    Parameters required for test execution:
      - KUBECONFIG
    """
    for ext_topic in get_external_topics(os_manager):
        secret_data = os_manager.get_secret_decoded(
            f"openstack-{ext_topic}-notifications", EXTERNAL_RABBITMQ_SECRETS_NAMESPACE
        )

        LOG.info(f'Check connect to RabbitMQ vhost {secret_data["vhost"]}')
        check_test_connection = check_connection(
            secret_data["username"],
            secret_data["password"],
            secret_data["hosts"],
            secret_data["port_amqp"],
            secret_data["vhost"],
        )

        assert check_test_connection, "Failed connection to RabbitMQ vhost"
        LOG.info("The connection check to the RabbitMQ virtual host is complete")
