import time
from si_tests import settings
from si_tests import logger
from si_tests.utils import waiters, utils

LOG = logger.logger

cluster_name = settings.TARGET_CLUSTER
namespace_name = settings.TARGET_NAMESPACE


# First delete one of existing worker and then re-add it afterwards
def test_delete_readd_bm_worker(kaas_manager, ipmi_client):
    """Delete one worker node and re-add it afterward

    * Assumes the node already exists.
    * Removes (and readds) only node with label 'si-role/node-for-delete'.
    NIT: works_with_raw_k8s_obj: true
    """

    def _check_cluster_ready():
        # Check that all machines in 'Ready' status
        # before machine removing
        cluster.check.check_machines_status()
        cluster.check.check_cluster_nodes()
        cluster.check.check_cluster_readiness()
        cluster.check.check_helmbundles()
        cluster.check.check_k8s_nodes()

    work_with_raw = False
    # Get namespace
    LOG.info("Namespace name - %s", namespace_name)
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    # Get ssh key
    public_key_name = f"{cluster_name}-key"
    LOG.info(f"Public key name - {public_key_name}")
    ns.get_publickey(public_key_name)

    region = kaas_manager.get_mgmt_cluster().region_name

    mgmt_version = \
        kaas_manager.get_mgmt_cluster().spec['providerSpec']['value']['kaas'][
            'release']
    LOG.info(f"KaaS mgmt version is:{mgmt_version}")
    allowed_distros = kaas_manager.get_allowed_distributions(cluster.clusterrelease_version)
    machine_distribution = allowed_distros.get(settings.DISTRIBUTION_RELEVANCE, {}).get('id', '')
    # Gather nodes information from yaml file
    render_opts = {
        "mgmt_version": mgmt_version,
        "target_region": region,
        "target_cluster": cluster_name,
        "target_namespace": namespace_name,
        "machine_distribution": machine_distribution,
    }
    child_data = utils.render_child_data(kaas_manager.si_config, render_opts)
    ns.create_bm_statics(child_data)
    # might be empty if child_data in raw objects format
    bm_hosts_data = child_data.get('nodes', [])
    if (len(bm_hosts_data)) == 0:
        LOG.warning(f"No <nodes:[]> found in {settings.BM_CHILD_SETTINGS_YAML_PATH}, "
                    f"assume we work with raw k8s objects")
        work_with_raw = True
    _check_cluster_ready()

    # Delete machine(s)
    machines_for_delete = []
    for machine in ns.get_machines():
        machine_data = machine.read().to_dict()
        labels = machine_data.get('metadata', {}).get('labels', {})
        if 'si-role/node-for-delete' in labels:
            machines_for_delete.append(machine)

    assert len(machines_for_delete) >= 1, "Not found alive Machine(s) with label " \
                                          "'si-role/node-for-delete'"
    bmh_for_delete = []
    for bmh in ns.get_baremetalhosts():
        labels = bmh.data.get('metadata', {}).get('labels', {})
        if 'si-role/node-for-delete' in labels:
            bmh_for_delete.append(bmh)

    assert len(bmh_for_delete) >= 1, "Not found alive BMH(s) with label " \
                                     "'si-role/node-for-delete'"

    bmh_names_for_delete = [bmh.name for bmh in bmh_for_delete]

    LOG.info("Machines for delete: %s", ','.join([m.name for m in machines_for_delete]))
    # delete all at once,
    for machine in machines_for_delete:
        LOG.info(f"Deleting Machine: {machine.name}")
        machine.delete()
    #  wait one by one
    for machine in machines_for_delete:
        cluster.wait_machine_deletion(machine.name, retries=settings.KAAS_BM_WAIT_BMH_MACHINE_DELETE_RETRY)

    LOG.info(f"Waiting bmh(s):\n{bmh_names_for_delete}\nto be in "
             f"'ready' state")
    ns.wait_baremetalhosts_statuses(nodes=bmh_names_for_delete,
                                    wait_status='ready',
                                    retries=30,
                                    interval=60, self_check=True)
    # assume, we push delete to api, check cluster health, and then check that bmh deleted.
    # that would speed up test, w\o functional logic loss.
    LOG.info("Delete BMH(s) and check it has been deleted")
    LOG.info("BMH for delete: %s", ','.join(bmh_names_for_delete))
    bmcs_data = []
    for bmh in bmh_for_delete:
        bmcs_data.append(cluster.get_bmc_data(bmh))
        bmh_name = bmh.name
        # NOTE(alexz): delete_baremetalhost also manage bmhi case
        LOG.info(f"Deleting BMH: {bmh_name}")
        ns.delete_baremetalhost(name=bmh_name)

    # Waiting for cluster,pods,and are Ready after machines were deleted
    _check_cluster_ready()

    if work_with_raw:
        ns.wait_all_bmhs_deletion_by_names(bmh_names_for_delete)
    else:
        ns.wait_all_bmhs_deletion(bmhs_for_delete=bmh_for_delete, wait_bmh_cred=True, bm_hosts_data=bm_hosts_data)

    if settings.BM_CHECK_BMC_STATUS_AFTER_DELETE:
        for bmc_data in bmcs_data:
            power_on = ipmi_client.get_power_on(ipmi_host=bmc_data['host'],
                                                ipmi_user=bmc_data['username'],
                                                ipmi_password=bmc_data['password'],
                                                ipmi_port=bmc_data['port'])
            LOG.info(f"Chassis power on status for {bmc_data['host']}:{bmc_data['port']} is: {power_on}")
            assert not power_on, \
                f"Chassis status is not powered off." \
                f"Current power_on status is {power_on}"
    LOG.info(f"Next BMHs were deleted successfully:\n{bmh_for_delete}")

    # Recreate BMH and check it has been inspected
    LOG.info("Recreating Baremetal host(s)/machine(s)")
    new_machine_names = []
    if work_with_raw:
        bmhc_objs_to_recreate = []
        machine_objs_to_recreate = []
        bmhc_objs_names_to_recreate = []
        bmh_objs_to_recreate = [b for b in child_data.get('bmh_raw', []) if
                                b['metadata']['labels'].get('si-role/node-for-delete')]
        bmhc_objs_names_to_recreate += [b['metadata']['annotations']['kaas.mirantis.com/baremetalhost-credentials-name']
                                        for b in child_data.get('bmh_raw', []) if
                                        b['metadata']['labels'].get('si-role/node-for-delete')]
        # bmhi scenario
        bmhi_objs_to_recreate = [b for b in child_data.get('bmhi_raw', []) if
                                 b['metadata']['labels'].get('si-role/node-for-delete')]
        bmhc_objs_names_to_recreate += [b['spec']['bmc']['bmhCredentialsName']
                                        for b in child_data.get('bmhi_raw', []) if
                                        b['metadata']['labels'].get('si-role/node-for-delete')]
        #
        bmhc_objs_to_recreate += [c for c in child_data.get('bmhc_raw', []) if
                                  c['metadata']['name'] in bmhc_objs_names_to_recreate]
        new_machine_names += [b['metadata']['name'] for b in child_data['machines_raw'] if
                              b['metadata']['labels'].get('si-role/node-for-delete')]

        machine_objs_to_recreate += [b for b in child_data['machines_raw'] if
                                     b['metadata']['labels'].get('si-role/node-for-delete')]

        for bmhc in bmhc_objs_to_recreate:
            ns.create_baremetalhostcredential_raw(bmhc)
        for bmhi in bmhi_objs_to_recreate:
            ns.create_baremetalhostinventories_raw(bmhi)
        for bmh in bmh_objs_to_recreate:
            ns.create_baremetalhost_raw(bmh)
        for m in machine_objs_to_recreate:
            cluster.create_baremetal_machine_raw(m)

    else:
        machines_to_recreate = [node for node in bm_hosts_data
                                if 'si-role/node-for-delete' in
                                node.get('bmh_labels', {})]
        for node in machines_to_recreate:
            cred_name = node['name'] + '-cred'

            secret_data = {
                "username": node['ipmi']['username'],
                "password": node['ipmi']['password']
            }
            # Wait for bmhc deletion after node was deleted
            waiters.wait(lambda: not kaas_manager.api.kaas_baremetalhostscredentials.present(
                name=cred_name, namespace=namespace_name),
                         timeout_msg=f"bmhc {cred_name} was not "
                                     f"deleted successfully")
            if node['ipmi'].get('monitoringUsername', False):
                secret_data.update({
                    "monitoringPassword": node['ipmi']['monitoringPassword'],
                    "monitoringUsername": node['ipmi']['monitoringUsername']
                })
            if not kaas_manager.api.kaas_baremetalhostscredentials.present(name=cred_name,
                                                                           namespace=ns.name):
                ns.create_baremetalhostcredential(name=cred_name, data=secret_data, region=region,
                                                  provider="baremetal")
            else:
                LOG.warning(f'bmhc: {cred_name} already exist, skipping')

            # Dirty hack, for not copy-paste bmh_name|cred_name across whole
            # test.
            bmh_name = utils.render_bmh_name(node['name'],
                                             cluster_name,
                                             node.get('bootUEFI', True),
                                             node['bmh_labels'],
                                             si_roles=node.get('si_roles',
                                                               False))
            node.update({'bmh_name': bmh_name,
                         'cred_name': cred_name})
            new_machine_names.append(bmh_name)
            ns.create_baremetalhost(bmh_name=bmh_name,
                                    bmh_secret=cred_name,
                                    bmh_mac=node['networks'][0]['mac'],
                                    bmh_ipmi=node['ipmi'],
                                    hardwareProfile=node.get('hardwareProfile',
                                                             False),
                                    labels=node['bmh_labels'],
                                    bootUEFI=node.get('bootUEFI', True),
                                    annotations=node.get('bmh_annotations', {}),
                                    bmhi_credentials_name=cred_name)
            # dnsmasq bug PRODX-7751; epic with workaround PRODX-8106
            # dnsmasq DHCP provides IPs one by one with ping-check
            # for each IP for about 12sec.
            time.sleep(settings.KAAS_BM_CHILD_SLEEP_PRODX_7751)

        LOG.info(f"Wating bmh(s):\n{new_machine_names}\nto be in 'ready' state")
        ns.wait_baremetalhosts_statuses(nodes=new_machine_names,
                                        wait_status='ready',
                                        retries=60,
                                        interval=30, self_check=True)

        # Recreate worker machines
        for node in machines_to_recreate:
            LOG.info('Create BM machine')
            custom_bmhp = False
            if node.get('bmh_profile'):
                custom_bmhp = {
                    'namespace': namespace_name,
                    'name': node['bmh_profile']
                }
            cluster.create_baremetal_machine(
                genname=node['bmh_name'],
                node_pubkey_name=public_key_name,
                matchlabels={'kaas.mirantis.com/'
                             'baremetalhost-id':
                                 node['bmh_labels']
                                 ['kaas.mirantis.com/baremetalhost-id']},
                baremetalhostprofile=custom_bmhp,
                labels=node['machine_labels'],
                l2TemplateSelector=node.get('l2TemplateSelector', dict()),
                node_labels=node.get('node_labels', {}),
                distribution=node.get('distribution', None), )
            # dnsmasq bug PRODX-7751; epic with workaround PRODX-8106
            # dnsmasq DHCP provides IPs one by one with ping-check
            # for each IP for about 12sec.
            time.sleep(settings.KAAS_BM_CHILD_SLEEP_PRODX_7751)

    unset_machine_pauses = False
    if not settings.MACHINE_PAUSE_DURING_CREATION_ENABLED:
        cluster_release_version = cluster.get_desired_clusterrelease_version()
        if utils.clusterrelease_version_greater_than_or_equal_to_kaas_2_30_0(cluster_release_version):
            unset_machine_pauses = True
    machines = cluster.get_machines_uncached()
    if unset_machine_pauses:
        LOG.info("Removing pause before provisioning from machines")
        cluster.set_day1_provisioning('auto', machines)
        machines = cluster.get_machines_uncached()
        cluster.check.check_day1_modes(machines, provisioning='auto')

    LOG.info(f"Waiting bmh(s):\n{new_machine_names}\nto be in 'provisioned' state")
    ns.wait_baremetalhosts_statuses(nodes=new_machine_names,
                                    wait_status='provisioned',
                                    wait_nodes_msg=f'waiting for {new_machine_names}',
                                    retries=60,
                                    interval=60,
                                    self_check=True)
    LOG.info('Baremetal host(s) is successfully recreated')

    if unset_machine_pauses:
        LOG.info("Removing pause before deployment from machines")
        cluster.set_day1_deployment('auto', machines)
        machines = cluster.get_machines_uncached()
        cluster.check.check_day1_modes(machines, deployment='auto')
    LOG.info('Waiting for machines are Ready and helmbundles are deployed')
    _check_cluster_ready()

    # Disable pods check until PRODX-5144 will be fixed
    # if cluster.lcm_type_is_ucp:
    #    # Check/wait for correct docker service replicas in cluster
    #    cluster.check.check_actual_expected_docker_services()
    # cluster.check.check_k8s_pods()
    # cluster.check.check_actual_expected_pods()

    # Rechecking Ceph cluster Health (Disabled until PRODX-2991 will be fixed)
    # ns.wait_child_ceph_cluster(cluster_name=cluster_name, assert_health=True)
