import pytest
import yaml
from si_tests import settings
from si_tests import logger
from si_tests.utils import waiters
from si_tests.managers.machine_deletion_policy_manager import check_machine_graceful_delete

LOG = logger.logger


@pytest.mark.usefixtures("introspect_distribution_not_changed")
@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
@pytest.mark.usefixtures('log_method_time')
def test_delete_bm_compute(kaas_manager, show_step, ipmi_client, os_manager):
    """Delete bm compute node.

    Scenario:
        1. Get hypervisor list and check status
        2. Get workers with compute label
        3. Get target node for delete
        4. Delete node and wait for deletion
        5. Check hypervisor count and status
        6. Delete bmh
        7. Check power status


    """
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    mgmt_version = \
        kaas_manager.get_mgmt_cluster().spec['providerSpec']['value']['kaas'][
            'release']
    LOG.info(f"KaaS mgmt version is:{mgmt_version}")

    client_pod = cluster.k8sclient.pods.list(
        namespace="openstack",
        name_prefix='keystone-client')
    assert len(client_pod) > 0, ("No pods found with prefix "
                                 "keystone-client in namespace "
                                 "openstack")
    client_pod = client_pod[0]
    cmd_hypervisor_list = ['/bin/sh', '-c',
                           'PYTHONWARNINGS=ignore::UserWarning '
                           'openstack hypervisor list -f yaml']
    show_step(1)
    cmp_count = len(yaml.safe_load(client_pod.exec(
        cmd_hypervisor_list)))

    def check(count=None):
        hyper_list = yaml.safe_load(client_pod.exec(cmd_hypervisor_list))
        hyper_states = sum(h.get('State') == 'up' for h in hyper_list)
        LOG.info(f"Count hv: {hyper_states}")
        if hyper_states == count and len(hyper_list) == count:
            return True
        else:
            return False

    waiters.wait(
        lambda: check(count=cmp_count),
        timeout=300, interval=10,
        timeout_msg="Some hypervisor is not ready")
    show_step(2)

    workers = cluster.get_machines(machine_type='worker',
                                   machine_status='Ready')

    cmp = []
    for w in workers:
        if [n for n in w.nodeLabels if 'openstack-compute-node'
                                       in n.values()]:
            cmp.append(w)
    show_step(3)
    target_node = [n for n in cmp if "nodeforscale" in n.name]
    target_node = target_node[0]
    target_node_k8s_name = target_node.get_k8s_node_name()
    # Remove VM's from compute

    def get_os_server_list(node_uid):
        cmd = ['/bin/sh', '-c', 'PYTHONWARNINGS=ignore::UserWarning '
                                'openstack server list --all-projects '
                                f'--host kaas-node-{node_uid} -f yaml']
        result = yaml.safe_load(client_pod.exec(cmd))
        LOG.info(result)
        return result

    LOG.info(f'Get VMs on {target_node.uid} node:')
    vm_list = get_os_server_list(target_node.uid)
    if vm_list:
        vm_ids = ' '.join([x['ID'] for x in vm_list])
        LOG.info(f'Delete next VMs: {vm_ids}')
        client_pod.exec(['/bin/sh', '-c', f'PYTHONWARNINGS=ignore::UserWarning '
                                          f'openstack server delete {vm_ids}'])
        waiters.wait((lambda expected: expected == len(get_os_server_list(target_node.uid))),
                     predicate_kwargs={'expected': 0},
                     timeout=120,
                     timeout_msg="Not all VMs have been deleted")
    else:
        LOG.info(f'No VMs to delete on {target_node.uid}')

    show_step(4)
    LOG.info(f"Deleting node: {target_node.name} uid: {target_node.uid}")
    machine_deletion_policy_enabled = cluster.machine_deletion_policy_enabled()
    if machine_deletion_policy_enabled:
        check_machine_graceful_delete(cluster, target_node, wait_deletion_timeout=3600)
    else:
        target_node.delete()
        cluster.check.check_deleted_node(target_node.name)
    show_step(5)
    new_compute_count = cmp_count - 1
    waiters.wait(
        lambda: check(count=new_compute_count),
        timeout=300, interval=10,
        timeout_msg="Some hypervisor is not ready")

    show_step(6)
    bmh_for_delete = [bmh for bmh in ns.get_baremetalhosts()
                      if 'nodeforscale' in bmh.name]
    bmh = bmh_for_delete[0]

    bmc_data = cluster.get_bmc_data(bmh)
    bmh_name = bmh.name
    ns.wait_baremetalhosts_statuses(nodes=bmh_name,
                                    wait_status='ready',
                                    retries=30,
                                    interval=60)
    ns.delete_baremetalhost(name=bmh_name)
    ns.wait_baremetalhost_deletion(bmh_name, wait_bmh_cred=True)

    show_step(7)
    # Wait for BareMetalHostCredentail deletion after BMH was deleted
    power_on = ipmi_client.get_power_on(ipmi_host=bmc_data['host'],
                                        ipmi_user=bmc_data['username'],
                                        ipmi_password=bmc_data['password'])
    LOG.info("Chassis power on status: {}".format(power_on))
    assert not power_on, \
        "Chassis power status is not OFF. " \
        "Now power status is {}".format(power_on)
    LOG.info(f"Next BMH(s) were deleted successfully:\n{bmh_for_delete}")

    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()
    cluster.check.check_openstack_metadata_deletion(target_node_k8s_name)
    cluster.check.check_k8s_nodes()
    cluster.check.check_k8s_pods()
    cluster.check.check_cluster_readiness()
    # TODO(tleontovich) Delete after https://mirantis.jira.com/browse/FIELD-6500 fixed
    os_manager.wr_field_6500()
