import base64
import configparser

from si_tests import settings
from si_tests import logger


LOG = logger.logger


def check_config_secret(decoded_data,
                        skip_keys_list):
    """
    Method to check 'password' value in secret data
    presented as config file.

    :param decoded_data: str with data of secret
    :param skip_sections_list: list of sections to skip
    :param skip_keys_list: list of keys to skip
    :return: (bool) false if config contains value 'password'
    """
    parser = configparser.ConfigParser(interpolation=None,
                                       strict=False)
    skip_sections_list = settings.SKIP_SECRET_SECTIONS_LIST
    # Try to read config via configparser.
    # With it we can skip sections.
    parser.read_string(decoded_data)

    for section in parser.sections():
        for k, v in parser.items(section):
            if (v == 'password' and
                    section not in skip_sections_list and
                    k not in skip_keys_list):
                return False
    return True


def check_string_secret(decoded_data,
                        skip_keys_list):
    """
    Method to check 'password' value in secret data
    presented as string.

    :param decoded_data: str with data of secret
    :param skip_keys_list: list of keys to skip
    :return: (bool) false if config contains values as
             '=password', ':password@', ':"password"'
    """
    # Remove extensions for checks
    decoded_data = decoded_data.replace(' ', '')

    for skip_key in skip_keys_list:
        decoded_data.replace(
            '{}=password'.format(skip_key), '')
    # Password may contains as:
    # 1. Option of config file
    # 2. Part of URL in config files
    # 3. In generated passwords
    values_to_check = ['=password',
                       ':password@',
                       ':\"password\"']
    contains_password = any(
        value in decoded_data for value in values_to_check)

    if 'password' == decoded_data or contains_password:
        return False
    return True


def test_secret_not_contain_password(os_manager):
    """Test  Secrets don't contain value 'password'
       Parameters required for test execution:
         - KUBECONFIG
    """

    # Get all Secrets
    secrets = os_manager.get_os_secrets(read=True)
    err_msg = "Secrets contains value `password`. Failed for: "
    success = []

    skip_keys_list = settings.SKIP_SECRET_KEYS_LIST

    for secret in secrets:
        secret_name = secret.metadata.name

        if not secret.data:
            continue
        for key, encoded_data in secret.data.items():
            decoded_data = base64.b64decode(encoded_data).decode('utf-8')

            try:
                check_status = check_config_secret(decoded_data,
                                                   skip_keys_list)
            except configparser.MissingSectionHeaderError:
                check_status = check_string_secret(decoded_data,
                                                   skip_keys_list)
            if not check_status:
                err_msg += "\nSecret {secret}, filename: {filename}, " \
                           "data: {data}".format(secret=secret_name,
                                                 filename=key,
                                                 data=decoded_data)
            success.append(success)

    assert all(success), err_msg
