from si_tests import logger
from si_tests.lcm.openstack_lcm.policies import Policy
from si_tests.utils.utils import verify

LOG = logger.logger


def test_get_pool_list_and_loadbalancer_list_policy(openstack_client_manager, policy_user, os_manager):
    """Test get load balancer pool and load balancer lists for octavia policy
       https://mirantis.testrail.com/index.php?/cases/view/4963656

       Parameters required for test execution:
         - KUBECONFIG

       Setup environment:
       Create user with role member
    """
    user_name, password = policy_user
    http_exception = "HTTP 403"
    os_service = 'octavia'
    os_pod = 'octavia-api'
    policy = Policy(os_service, os_pod)
    osdpl = os_manager.get_osdpl_deployment()
    openstack_version = osdpl.read().spec["openstack_version"]

    try:
        get_policy_file_before_update = ['/bin/sh', '-c',
                                         r' grep -irn "pool:get_all:\|loadbalancer:get_all:"'
                                         ' /etc/octavia/policy.d/02-custom.yaml']
        policy.check_policy_absent_before_update(get_policy_file_before_update)

        LOG.info("Step 1: Checking that policies are absent."
                 " Load balancer list and load balancer pool list are available for admin and user")
        step = 1

        loadbalancer_list = openstack_client_manager.loadbalancer.list([], combined_output=True)
        verify(http_exception not in loadbalancer_list["stderr"],
               f"[Step {step}]: Can't perform loadbalancer list with admin with default policies",
               f"[Step {step}]: Can perform loadbalancer list with admin with default policies")

        pool_list = openstack_client_manager.loadbalancer.pool_list([], combined_output=True)
        verify(http_exception not in pool_list["stderr"],
               f"[Step {step}]: Can't perform loadbalancer pool list with admin with default policies",
               f"[Step {step}]: Can perform loadbalancer pool list with admin with default policies")

        # Note for new releases: by default RBAC rules will switch from Octavia Advanced RBAC to the
        # keystone default roles.
        # This means the load_balancer_* roles will not longer have access to the load balancer API
        # https://docs.openstack.org/releasenotes/octavia/2025.1.html#upgrade-notes
        loadbalancer_list = openstack_client_manager.loadbalancer.list(
            [f"--os-username={user_name}", f"--os-password={password}"], combined_output=True)
        test_expression = http_exception not in loadbalancer_list["stderr"]
        if openstack_version not in ["antelope", "caracal"]:
            verify(test_expression,
                   f"[Step {step}]: Can't perform loadbalancer list with user with default policies",
                   f"[Step {step}]: Can perform loadbalancer list with user with default policies")

        else:
            verify(not test_expression,
                   f"[Step {step}]: Can perform loadbalancer list with user with default policies",
                   f"[Step {step}]: Can't perform loadbalancer list with user with default policies"
                   " of Octavia API policies that require octavia-specific roles, which the policy_user used in "
                   "the test does not have")

        pool_list = openstack_client_manager.loadbalancer.pool_list(
            [f"--os-username={user_name}", f"--os-password={password}"], combined_output=True)
        test_expression = http_exception not in pool_list["stderr"]
        if openstack_version not in ["antelope", "caracal"]:
            verify(test_expression,
                   f"[Step {step}]: Can't perform loadbalancer pool list with user with default policies",
                   f"[Step {step}]: Can perform loadbalancer pool list with user with default policies")
        else:
            verify(not test_expression,
                   f"[Step {step}]: Can perform loadbalancer pool list with user with default policies",
                   f"[Step {step}]: Can't perform loadbalancer pool list with user with default policies"
                   " of Octavia API policies that require octavia-specific roles, which the policy_user used in "
                   "the test does not have")

        LOG.info("Step 2: Adding a new rule to OSDPL file")
        step = 2
        policy_add_role = {'os_load-balancer_api:loadbalancer:get_all': '@', 'os_load-balancer_api:pool:get_all': '@'}
        policy.update_policy_osdpl(policy_add_role, step)

        LOG.info("Step 3: Checking that the policies were added to Octavia API pod")
        step = 3
        check_policy_file = ['/bin/sh', '-c', ' cat /etc/octavia/policy.d/02-custom.yaml']
        policy.wait_policy_updated_on_pod(check_policy_file, policy_add_role, step)

        LOG.info("Step 4: The policies were added. Checking that load balancer"
                 " list and load balancer pool list can be obtained by admin and user")
        step = 4

        loadbalancer_list = openstack_client_manager.loadbalancer.list([], combined_output=True)
        verify(http_exception not in loadbalancer_list["stderr"],
               f"[Step {step}]: Can't perform loadbalancer list with admin and applied policies",
               f"[Step {step}]: Can perform loadbalancer list with admin and applied policies "
               "that allow everyone to get load balancer list. Load balancer list can be obtained by admin")

        pool_list = openstack_client_manager.loadbalancer.pool_list([], combined_output=True)
        verify(http_exception not in pool_list["stderr"],
               f"[Step {step}]: Can't perform loadbalancer pool list with admin and applied policies",
               f"[Step {step}]: Can perform loadbalancer pool list with and applied policies "
               "that allow everyone to get load balancer list. Load balancer list can be obtained by admin")

        loadbalancer_list = openstack_client_manager.loadbalancer.list(
            [f"--os-username={user_name}", f"--os-password={password}"], combined_output=True)
        verify(http_exception not in loadbalancer_list["stderr"],
               f"[Step {step}]: Can't perform loadbalancer list with user and applied policies",
               f"[Step {step}]: Can perform loadbalancer list with user and applied policies "
               "that allow everyone to get load balancer list. Load balancer list can be obtained by user")

        pool_list = openstack_client_manager.loadbalancer.pool_list(
            [f"--os-username={user_name}", f"--os-password={password}"], combined_output=True)
        verify(http_exception not in pool_list["stderr"],
               f"[Step {step}]: Can't perform loadbalancer pool list with user and applied policies",
               f"[Step {step}]: Can perform loadbalancer pool list with user and applied policies "
               "that allow everyone to get load balancer list. Load balancer list can be obtained by user")

        LOG.info("Step 5: Updating policies. Checking that the policies were added to Octavia API pod")
        step = 5

        # NOTE(pas-ha): In Octavia, get_all-global policy has higher precedence
        # than get_all, so need to disable both to block admin too.
        policy_update_role = {
            'os_load-balancer_api:pool:get_all': '!',
            'os_load-balancer_api:pool:get_all-global': '!',
            'os_load-balancer_api:loadbalancer:get_all': '!',
            'os_load-balancer_api:loadbalancer:get_all-global': '!',
        }
        policy.update_policy_osdpl(policy_update_role, step)
        policy.wait_policy_updated_on_pod(check_policy_file, policy_update_role, step)

        LOG.info("Step 6: The policies were updated. Checking that load balancer list"
                 " and load balancer pool list can not be obtained by anyone.")
        step = 6

        loadbalancer_list = openstack_client_manager.loadbalancer.list([], combined_output=True)
        verify(http_exception in loadbalancer_list["stderr"],
               f"[Step {step}]: Can perform loadbalancer list for admin",
               f"[Step {step}]: Can't perform loadbalancer list for admin. "
               "Policies disallow anyone to get load balancer list, load balancer list cannot be obtained by admin")

        pool_list = openstack_client_manager.loadbalancer.pool_list([], combined_output=True)
        verify(http_exception in pool_list["stderr"],
               f"[Step {step}]: Can perform loadbalancer pool list for admin",
               f"[Step {step}]: Can't perform loadbalancer pool list for admin. "
               "Policies disallow anyone to get load balancer pool list, load balancer pool list"
               " cannot be obtained by admin")

        error_output = openstack_client_manager.loadbalancer.list(
            [f"--os-username={user_name}", f"--os-password={password}"], combined_output=True)
        verify(http_exception in error_output["stderr"],
               f"[Step {step}]: Can perform loadbalancer list for user",
               f"[Step {step}]: Can't perform loadbalancer list for user. "
               "Policies disallow anyone to get load balancer list, load balancer list cannot be obtained by admin")

        error_output = openstack_client_manager.loadbalancer.pool_list(
            [f"--os-username={user_name}", f"--os-password={password}"], combined_output=True)
        verify(http_exception in error_output["stderr"],
               f"[Step {step}]: Can perform loadbalancer pool list for user",
               f"[Step {step}]: Can't perform loadbalancer pool list for user. "
               "Policies disallow anyone to get load balancer pool list, load balancer pool list "
               "cannot be obtained by admin")

    finally:
        LOG.info("Step 7: Deleting policies. Checking that the policies were deleted and"
                 " the result is the same as in the first step")
        step = 7

        policy_not_role = None
        policy.update_policy_osdpl(policy_not_role, step)
        policy.wait_policy_updated_on_pod(check_policy_file, {}, step)

        loadbalancer_list = openstack_client_manager.loadbalancer.list([], combined_output=True)
        verify(http_exception not in loadbalancer_list["stderr"],
               f"[Step {step}]: Can't perform loadbalancer list with admin with default policies",
               f"[Step {step}]: Can perform loadbalancer list with admin with default policies")

        pool_list = openstack_client_manager.loadbalancer.pool_list([], combined_output=True)
        verify(http_exception not in pool_list["stderr"],
               f"[Step {step}]: Can't perform loadbalancer pool list with admin with default policies",
               f"[Step {step}]: Can perform loadbalancer pool list with admin with default policies")

        # Note for new releases: by default RBAC rules will switch from Octavia Advanced RBAC to the
        # keystone default roles.
        # This means the load_balancer_* roles will not longer have access to the load balancer API
        # https://docs.openstack.org/releasenotes/octavia/2025.1.html#upgrade-notes
        loadbalancer_list = openstack_client_manager.loadbalancer.list(
            [f"--os-username={user_name}", f"--os-password={password}"], combined_output=True)
        test_expression = http_exception not in loadbalancer_list["stderr"]
        if openstack_version not in ["antelope", "caracal"]:
            verify(test_expression,
                   f"[Step {step}]: Can't perform loadbalancer list with user with default policies",
                   f"[Step {step}]: Can perform loadbalancer list with user with default policies")

        else:
            verify(not test_expression,
                   f"[Step {step}]: Can perform loadbalancer list with user with default policies",
                   f"[Step {step}]: Can't perform loadbalancer list with user with default policies"
                   " of Octavia API policies that require octavia-specific roles, which the policy_user used in "
                   "the test does not have")

        pool_list = openstack_client_manager.loadbalancer.pool_list(
            [f"--os-username={user_name}", f"--os-password={password}"], combined_output=True)
        test_expression = http_exception not in pool_list["stderr"]
        if openstack_version not in ["antelope", "caracal"]:
            verify(test_expression,
                   f"[Step {step}]: Can't perform loadbalancer pool list with user with default policies",
                   f"[Step {step}]: Can perform loadbalancer pool list with user with default policies")
        else:
            verify(not test_expression,
                   f"[Step {step}]: Can perform loadbalancer pool list with user with default policies",
                   f"[Step {step}]: Can't perform loadbalancer pool list with user with default policies"
                   " of Octavia API policies that require octavia-specific roles, which the policy_user used in "
                   "the test does not have")
