import pytest
from si_tests import logger
from si_tests import settings

LOG = logger.logger


@pytest.mark.parametrize("cleanup_hoc_and_machine_labels", ["run_on_setup"], indirect=True)
@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
def test_cpushield_create_hoc(kaas_manager, _, show_step, cleanup_hoc_and_machine_labels):
    """ This is a first part of a cpushield module test scenario (see below)
    which is responsible for hoc object creation.

    Scenario:
        1. Create a cpushield HostOSConfiguration object for testing CPU shielding
        via si_tests/tests/lcm/day2operations/test_create_cpushield_hoc_bm.py.
        2. Reboot selected machines via si_tests/tests/lcm/test_rolling_reboot_machines.py
        Use 'export ROLLING_REBOOT_MACHINES_LABELS=day2-cpushield-test' before run
        3. Execute commands like 'stress' on selected machines and verify
        that they run only on 'isolated' CPU cores ('system_cpus' module parameter) via
        si_tests/tests/lcm/day2operations/test_check_cpushielding_bm.py
    """
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    hostoscfg_config_cpushield = [
        {
            "module": "package",
            "moduleVersion": settings.HOC_PACKAGE_MODULE_VERSION,
            "values": {"packages": [{"name": "stress", "state": "present"}]},
        },
        {
            "module": "cpushield",
            "moduleVersion": settings.HOC_CPUSHIELD_MODULE_VERSION,
            "values": {
                "system_cpus": "0-7",
                "systemd_units_to_pin": [
                    "system.slice",
                    "user.slice",
                    "kubepods.slice",
                ]
            },
        }
    ]

    # hoc sample
    hostoscfg_data = {
        "apiVersion": "kaas.mirantis.com/v1alpha1",
        "kind": "HostOSConfiguration",
        "metadata": {
            "name": settings.HOC_CPUSHIELD_TEST_HOC_NAME,
            "namespace": namespace_name,
        },
        "spec": {
            "configs": hostoscfg_config_cpushield,
            "machineSelector": {
                "matchLabels": settings.HOC_CPUSHIELD_REBOOT_LABEL,
            }
        }
    }

    show_step(1)
    """Label machines for reboot test"""
    machines_to_label = []
    mgmt_machines = cluster.get_machines(machine_type='control')
    assert len(mgmt_machines) > 1, f"We need at least 1 mgmt machine in cluster: {cluster_name}"
    machines_to_label.append(mgmt_machines[0])
    if cluster.is_child:
        worker_machines = cluster.get_machines(machine_type='worker')
        assert len(worker_machines) > 1, f"We need at least 1 worker machine in child cluster: {cluster_name}"
        machines_to_label.append(worker_machines[0])

    # Skip cpushield module testing for Ubuntu 20.04
    machines_with_ubuntu_focal = []
    for m in machines_to_label:
        if 'ubuntu/focal' in m.get_distribution():
            machines_with_ubuntu_focal.append(m.name)

    if machines_with_ubuntu_focal:
        msg = (f"Skipping cpushield module testing because machines {machines_with_ubuntu_focal} "
               "are provisoned with Ubuntu 20.04")
        LOG.info(msg)
        pytest.skip(msg)

    for machine in machines_to_label:
        LOG.info(f"Add label {settings.HOC_CPUSHIELD_REBOOT_LABEL} to machine: {machine.name} "
                 f"in cluster: {cluster_name}")
        machine.add_machine_labels(settings.HOC_CPUSHIELD_REBOOT_LABEL)

    """Create HOC object"""
    # Remember LCMMachines timestamps before creating HostOSConfiguration
    lcmmachines_timestamps_before = cluster.get_cluster_lcmmachines_timestamps()

    hostoscfg = ns.create_hostosconfiguration_raw(hostoscfg_data)

    # Wait for the selected Machines in the hostosconfiguration status
    LOG.info("Check that machines from hostosconfiguration status field have labels used for machineSelector")
    cluster.check.check_hostosconfig_machine_selector(hostoscfg)
    LOG.info("Check that new items added into machineTypes in LCMCluster")
    cluster.check.wait_lcmcluster_day2_machinetypes(hostoscfg)
    LOG.info("Check that new items added into stateItems in LCMMachine")
    cluster.check.wait_lcmmachine_day2_stateitems(hostoscfg, lcmmachines_timestamps_before)
    cluster.check.get_hostosconfig_machines_status(hostoscfg)
