import pytest
from si_tests import logger
from si_tests import settings

LOG = logger.logger


def _convert_cpu_numa_range(value):
    """Takes a string representing list of CPU cores or memory NUMA nodes indices or ranges separated
    by either whitespace or commas (systemd format).
    Ranges are specified by the lower and upper indices separated by a dash.

    Valid formats:
    `1-15,18,19`
    `1 2 3 4 5`
    `1-15 16 17 19`

    Invalid formats:
    `1-10-15`   - more than one dash used, not a range.
    `1,2,3 7 8` - mix of commas and whitespaces. Use only one kind of delimiter in a particular string.

    Returns: set of integers.
    """
    result = []
    if ',' in value and ' ' in value:
        raise ValueError('Either comma or whitespace should be used as separator, not both!')
    separator = ' ' if ' ' in value else ','
    for v in value.split(sep=separator):
        v_list = v.split(sep='-')
        if len(v_list) > 2:
            raise ValueError('Only one dash per range should be used, like 0-10, not 0-10-20')
        elif len(v_list) == 2:
            result += list(range(int(v_list[0]), int(v_list[1]) + 1))
        else:
            result.append(int(v_list[0]))
    return set(result)


@pytest.mark.usefixtures("mcc_per_node_workload_check_after_test")
@pytest.mark.usefixtures("collect_downtime_statistics")
@pytest.mark.parametrize("cleanup_hoc_and_machine_labels", ["run_on_teardown"], indirect=True)
@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
def test_check_cpushielding_on_machines(kaas_manager, _, show_step, cleanup_hoc_and_machine_labels):
    """ This is a third part of a cpushield module test scenario (see below)
    which is responsible for verifying CPU isolation on selected machines.

    Scenario:
        1. Create a cpushield HostOSConfiguration object for testing CPU shielding
        via si_tests/tests/lcm/day2operations/test_create_cpushield_hoc_bm.py
        2. Reboot selected machines via si_tests/tests/lcm/test_rolling_reboot_machines.py
        Use 'export ROLLING_REBOOT_MACHINES_LABELS=day2-cpushield-test' before run
        3. Execute commands like 'stress' on selected machines and verify
        that they run only on 'isolated' CPU cores ('system_cpus' module parameter) via
        si_tests/tests/lcm/day2operations/test_check_cpushielding_bm.py. Verify MOSK, Ceph
        workloads health.
    """
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    # Skip cpushield module testing for Ubuntu 20.04
    m_list = []
    for machine in cluster.get_machines():
        labels = machine.data.get('metadata', {}).get('labels', {})
        if set(settings.HOC_CPUSHIELD_REBOOT_LABEL.keys()).issubset(set(labels.keys())):
            m_list.append(machine.name)
    if not m_list:
        msg = (f"There are no machines with {settings.HOC_CPUSHIELD_REBOOT_LABEL} labels to proceed."
               " HOC object should not be created by test_create_cpushield_hoc_bm.py. Stopping test")
        pytest.skip(msg)
    assert ns.hostosconfiguration_is_present(name=settings.HOC_CPUSHIELD_TEST_HOC_NAME), \
        f"HostOSConfiguration object {settings.HOC_CPUSHIELD_TEST_HOC_NAME} is absent"
    hostoscfg = ns.get_hostosconfiguration(name=settings.HOC_CPUSHIELD_TEST_HOC_NAME)
    cpushield_module = cluster.check._get_hostosconfig_module_spec('cpushield', hostoscfg)
    if not cpushield_module:
        raise Exception("No 'cpushield' module under .spec.configs of passed hostoscfg object")
    cpushield_values = cpushield_module.get('values', {})
    if not cpushield_values:
        raise Exception("No 'values' passed for 'cpushield' module in hostoscfg object")
    system_cpus_set = _convert_cpu_numa_range(cpushield_values.get('system_cpus'))

    show_step(3)
    """Get machine where cpushield hoc is applied"""
    chosen_machine_names = cluster.check.get_hostosconfig_machines_status(hostoscfg)
    expected_hoc_machines_count = 2 if cluster.is_child else 1
    assert len(chosen_machine_names) == expected_hoc_machines_count, \
        (f"Expecting exactly {expected_hoc_machines_count} machines in HostOSConfiguration "
         f"object {hostoscfg.name} status but {len(chosen_machine_names)} are listed: {chosen_machine_names}")
    machines = cluster.get_machines()

    """Execute stress command on machines"""
    isol_check_cmd = ("stress -c $(grep -c processor /proc/cpuinfo) 2>&1 > /dev/null & "
                      "sleep 5; STRESS_PID=$!; cut -f 39 -d' ' /proc/${STRESS_PID}/stat; "
                      "for child in $(cat /proc/${STRESS_PID}/task/${STRESS_PID}/children); "
                      "do cut -f 39 -d' ' /proc/${child}/stat; done; "
                      "kill -9 ${STRESS_PID} 2>&1 > /dev/null")
    for machine in machines:
        machine_full_name = f'{machine.namespace}/{machine.name}'
        if machine_full_name in chosen_machine_names:
            cmd_out = machine.exec_pod_cmd(isol_check_cmd, verbose=False)
            assert cmd_out['exit_code'] == 0, \
                f'Bash script for checking CPU isolation has been failed on machine {machine_full_name}'
            cpu_cores_used = set(cmd_out['logs'].strip().split('\n'))
            cpu_cores_used_int = {int(c) for c in cpu_cores_used}
            assert cpu_cores_used_int.issubset(system_cpus_set)

    """Verify Ceph status after reboot and shielding. Mgmt cluster does not use Ceph"""
    if cluster.is_child and cluster.is_ceph_deployed:
        LOG.info("Wait Ceph HEALTH_OK status in cluster object")
        try:
            health_info = cluster.check.get_ceph_health_detail()
            assert health_info['status'] == "HEALTH_OK", f'Health is not OK. Will not proceed. ' \
                                                         f'Current ceph health status: {health_info}'
        except AssertionError:
            cluster.check.wait_ceph_health_status(timeout=600, interval=30)
        cluster.check.check_cluster_readiness()
        cluster.check.check_ceph_pvc()
        LOG.info("Ceph cluster is healthy.")
