# Copyright 2025 Mirantis, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

from si_tests import logger
from si_tests import settings

import pytest
import requests
import yaml

from si_tests.utils import waiters

LOG = logger.logger


def test_add_dev_hocm(kaas_manager, show_step):
    """ Dummy test-configurator for enabling day2 index-dev modules.
        Scenario:
            1. Enable HOCM object located at settings.HOCM_DAY2_INDEX_DEV_URL
    """

    show_step(1)
    if not settings.HOCM_ENABLE_DAY2_DEV_MODULES:
        skip_msg = "HOCM_ENABLE_DAY2_DEV_MODULES is false, nothing to do here."
        LOG.info(skip_msg)
        pytest.skip(skip_msg)

    remote_hocm_url = settings.HOCM_DAY2_INDEX_DEV_URL
    response = requests.get(remote_hocm_url, allow_redirects=True)
    remote_content = yaml.safe_load(response.content.decode("utf-8"))
    dev_hocm_name = remote_content.get('metadata', {}).get('name', '')
    remote_modules = remote_content.get('spec', {}).get('modules', {})

    if not remote_modules:
        skip_msg = f"remote dev modules are empty, skipping setting up '{dev_hocm_name}'."
        LOG.info(skip_msg)
        pytest.skip(skip_msg)

    if kaas_manager.hostosconfigurationmodule_is_present(dev_hocm_name):
        skip_msg = f"{dev_hocm_name} is already present, skipping..."
        LOG.info(skip_msg)
        pytest.skip(skip_msg)

    local_modules_all = []
    duplicate_list = []
    for hocm in kaas_manager.get_hostosconfigurationmodules():
        modules = hocm.read().spec.get('modules', {})
        module_name_version = [f"{m['name']}:{m['version']}" for m in modules]
        local_modules_all.extend(module_name_version)
    for module in remote_modules:
        if f"{module['name']}:{module['version']}" in local_modules_all:
            LOG.error(f"Module '{module['name']}' with version '{module['version']}' already present locally!")
            duplicate_list.append(module)
    assert duplicate_list == [], "Duplicated modules found in installed modules! Check log above"

    kaas_manager.create_hostosconfigurationmodule_raw(remote_content)

    timeout_msg = (f"HostOSConfigurationModule {dev_hocm_name} "
                   f"was not created in 5s")
    waiters.wait(lambda: kaas_manager.hostosconfigurationmodule_is_present(dev_hocm_name),
                 timeout=60,
                 interval=5,
                 timeout_msg=timeout_msg)
