from si_tests import logger
from si_tests import settings
from si_tests.utils import waiters, exceptions
import si_tests.utils.waiters as helpers
import time
import os
import logging
import json
import yaml

import pytest
import si_tests.utils.templates as utils

LOG = logger.logger

# {"role_name": {label}}
# DO NOT USE "-" in keys (role names)
roles = {'stacklight': {"role": "stacklight"},
         'ceph_osd': {"role": "ceph-osd-node"},
         'compute': {"openstack-compute-node": "enabled"},
         'controller': {"openstack-control-plane": "enabled"},
         'k8s_master': {"node-role.kubernetes.io/master": ""},
         'gateway': {"openstack-gateway": "enabled"}
         }


@pytest.fixture(scope='function', params=roles.values(),
                ids=list(roles.keys()))
def nodes(request, target_kubectl):
    LOG.info("\n\n*******Starting new test case*******\n")
    log_name = request.node.name.split('-')[-1].split(']')[0]
    failed_num = request.session.testsfailed
    LOG.parent.handlers.clear()
    log_file = os.path.join(settings.LOGS_DIR,
                            request.node.name.split('[')[0] + "-" +
                            log_name + ".log")

    fh = logging.FileHandler(log_file)
    fh.setFormatter(LOG.handlers[0].formatter)
    LOG.parent.addHandler(fh)
    LOG.info("Checking status of all nodes in cluster")
    assert target_kubectl.nodes.all_ready(), \
        "Some nodes are not Ready: {}".format(
            target_kubectl.nodes.list_statuses())
    LOG.info("Checking phase of all pods in cluster")
    helpers.wait(lambda: target_kubectl.pods.check_pods_statuses(),
                 timeout=240, interval=10,
                 timeout_msg="Some pods are not in correct status/phase")
    label_key = list(request.param.keys())[0]
    label_value = request.param[label_key]
    LOG.info("Fetching nodes with label {0}={1} from k8s".format(label_key,
                                                                 label_value))
    node_name = [x['metadata']['name']
                 for x in target_kubectl.nodes._list_all().to_dict()['items']
                 if label_key in x['metadata']['labels'].keys() and
                 label_value == x['metadata']['labels'][label_key]]

    nodes = [x for x in target_kubectl.nodes.list_all() if x.name in node_name]

    LOG.debug(nodes)
    yield nodes
    LOG.parent.handlers.clear()
    if failed_num == request.session.testsfailed:
        # remove log file if test passed
        os.remove(log_file)


def rescheduling_check(get_failed_pods, excluded_nodes, initial_num=0):
    """Additional method to track rescheduling process
    During shutdown, we need to make sure that:
    a) all pods on unaffected nodes are ok;
    b) some pods are rescheduled to working nodes.
    We can simply wait some time for rescheduling, but it is better
    to check it dynamically (and interrupt waiting earlier if possible).
    First while-true keeps comparing number of failed pods on previous run
    with current number (to make sure rescheduling process started).
    After number starts to decrease, we wait for all pods to be Ready in
    second while-true.
    """
    prev = initial_num
    LOG.info("Initiate rescheduling_check with value: {}".format(prev))
    while True:
        # here we check that number of
        # not ready pods is increasing
        failed_pods, _ = get_failed_pods(check_jobs=False,
                                         excluded_nodes=[excluded_nodes, None])
        current = len(failed_pods)
        LOG.info("Number of failed pods: previous - {0}; "
                 "current - {1}".format(prev, current))
        if current >= prev:
            LOG.info("Still waiting...")
            prev = current
            yield True
        else:
            LOG.info("Number of not ready pods started to decrease")
            prev = current
            break
    LOG.info("==================")
    while True:
        # here we check that number of
        # not ready pods is decreasing to 0
        current = len(get_failed_pods(check_jobs=False,
                                      excluded_nodes=[excluded_nodes, None]))
        LOG.info("Number of failed pods: {0}".format(current))
        if current != initial_num:
            LOG.info("Still waiting...")
            yield True
        else:
            LOG.info("Done.")
            yield False


def create_test_pod(node_name, target_kubectl):
    """Create test pod to check if node is ok"""

    template = utils.render_template(
        settings.DUMMY_TEST_POD_YAML, {'POD_NAME': 'ha-pod', 'NODE_NAME': node_name})
    json_body = json.dumps(yaml.load(template, Loader=yaml.SafeLoader))
    pod = target_kubectl.pods.create(name='ha-pod',
                                     body=json.loads(json_body))
    pod.wait_phase('Succeeded')
    LOG.info("Get test pod output")
    output = target_kubectl.api_core.read_namespaced_pod_log(
        name='ha-pod', namespace='default')
    assert output == 'Hello\n', \
        "Got unexpected output from pod: {}".format(output)
    LOG.info("Rereading test pod")
    actual_node_name = pod.read().to_dict()['spec']['node_name']
    assert actual_node_name == node_name, \
        "Actual node name {0}, expected {1}".format(actual_node_name,
                                                    node_name)
    LOG.info("Deleting test pod")
    pod.delete()


def test_ha_mosk_reboot(openstack_client, target_kubectl,
                        show_step, nodes, cluster_condition_check):
    """Reboot every CHILD cluster node (one at a time)
    Precondition - all expected pods and their replicas must be presented
    The following scenario is executed for every node
    Scenario:
        1. Get Openstack server object and fip for k8s node
        2. Get number of reboots
        3. Reboot node
        4. Wait till node is accessible
        5. Check number of reboots and compare
           with previous value (must be +1)
        6. Check that all nodes are Ready
        7. Check that all pods are Running and Ready
        8. Create test pod on affected node
        9. Wait 60 sec for cluster to sync all data

    Expected: result - every node is rebooted and all pods on every node are
    Running and Ready after this operation.
    """
    for node in nodes:
        show_step(1)
        openstack_instance_name = node.name
        server = openstack_client.get_server_by_name(
            server_name=openstack_instance_name)
        show_step(2)
        init_reboots = \
            openstack_client.run_cmd(
                server=server, cmd="last reboot | wc -l", username="ubuntu",
                key_file=settings.HA_TEST_PRIVATE_KEY_FILE).stdout_str
        LOG.info("Number of reboots {}".format(init_reboots))
        show_step(3)
        server.reboot()
        show_step(4)
        time.sleep(30)
        waiters.wait_pass(
            lambda: openstack_client.run_cmd(
                server=server, cmd="hostname", username="ubuntu",
                key_file=settings.HA_TEST_PRIVATE_KEY_FILE), timeout=200)
        waiters.wait_pass(
            lambda: target_kubectl.pods.list_all(), timeout=120)
        show_step(5)
        reboots = \
            openstack_client.run_cmd(
                server=server, cmd="last reboot | wc -l", username='ubuntu',
                key_file=settings.HA_TEST_PRIVATE_KEY_FILE).stdout_str
        assert int(reboots) > int(init_reboots), \
            "Node has not been rebooted"
        LOG.info("{} has been rebooted".format(node.name))
        show_step(6)
        helpers.wait(
            lambda: target_kubectl.nodes.all_ready(),
            timeout=600, interval=10)
        show_step(7)
        helpers.wait(lambda: target_kubectl.pods.check_pods_statuses(),
                     timeout=660, interval=30,
                     timeout_msg="Some pods are not in correct status/phase")
        show_step(8)
        create_test_pod(node.name, target_kubectl)
        show_step(9)
        time.sleep(60)


def test_ha_mosk_shutdown(openstack_client, target_kubectl,
                          show_step, nodes, cluster_condition_check):
    """Shutdown one of CHILD cluster node
    Precondition - all expected pods and their replicas must be presented
    The following scenario is executed for every node
    Scenario:
        1. Get Openstack server object for k8s node
        2. Power off Openstack server
        3. Wait for Openstack server to be in shutdown status
        4. Wait for k8s node to be not Ready
        5. Check all pods (except those on affected node) are ok
        6. Start Openstack server
        7. Check that all nodes are Ready
        8. Check that all pods are Running and Ready
        9. Create test pod on affected node
        10. Wait 60 sec for cluster to sync all data

    Expected: result - every node went offline and back and all pods
    on every node are Running and Ready after this operation.
    """
    for node in nodes:
        show_step(1)
        openstack_instance_name = node.name
        server = openstack_client.get_server_by_name(
            server_name=openstack_instance_name)
        show_step(2)
        server.stop()
        show_step(3)
        openstack_client.wait_server_vm_state(server, 'shut down')
        show_step(4)
        waiters.wait_pass(
            lambda: node.check_status(expected_status='Unknown'),
            timeout=150, interval=20)
        show_step(5)
        itr1 = rescheduling_check(
            get_failed_pods=target_kubectl.pods.list_not_in_phases,
            excluded_nodes=node.name)
        try:
            helpers.wait(lambda: not next(itr1),
                         timeout=1380, interval=30,
                         timeout_msg="Some pods are not in "
                                     "correct status/phase")
        except exceptions.TimeoutError as e:
            curr, _ = \
                target_kubectl.pods.list_not_in_phases(
                    check_jobs=False,
                    excluded_nodes=[node.name, None])
            LOG.info("Current not ready pods: {0} ({1})".format(len(curr),
                                                                curr))
            if len(curr) == 0:
                LOG.warning("No rescheduling happened")
            else:
                raise exceptions.TimeoutError(e)
        show_step(6)
        server.start()
        openstack_client.wait_server_vm_state(server, 'running')
        show_step(7)
        waiters.wait_pass(
            lambda: node.check_status(expected_status='True'),
            timeout=150, interval=20)
        helpers.wait(
            lambda: target_kubectl.nodes.all_ready(),
            timeout=60, interval=10)
        show_step(8)
        helpers.wait(lambda: target_kubectl.pods.check_pods_statuses(),
                     timeout=660, interval=30,
                     timeout_msg="Some pods are not in correct status/phase")
        show_step(9)
        create_test_pod(node.name, target_kubectl)
        show_step(10)
        time.sleep(60)
