from si_tests import logger
from si_tests import settings
from si_tests.managers.kaas_manager import Manager
from si_tests.utils import waiters

LOG = logger.logger


def reboot(kaas_manager: Manager, show_step, machine_type='control'):
    """
    Reboot each cluster node based on machine type
    Args:
        kaas_manager: Manager object
        show_step: show step fixture
        machine_type: machine type - control or worker

    Returns: None

    """
    namespace = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = namespace.get_cluster(settings.TARGET_CLUSTER)
    machines = cluster.get_machines(machine_type=machine_type)

    LOG.info("{} nodes will be rebooted:\n{}".format(machine_type, '\n'.join([x.name for x in machines])))

    for machine in machines:
        show_step(1)
        LOG.info(f"Target {machine_type} machine {machine.name}")

        show_step(2)
        init_uptime = machine.get_uptime()

        show_step(3)
        machine.reboot()

        show_step(4)
        cluster.check.check_k8s_nodes()
        cluster.check.check_cluster_nodes()

        cluster.check.check_k8s_pods()
        cluster.check.check_actual_expected_pods()

        show_step(5)
        current_uptime = machine.get_uptime()
        machine.get_reboot_list()
        assert current_uptime > init_uptime, \
            "Node has not been rebooted. Uptime before reboot: {} and after reboot: {}".format(init_uptime,
                                                                                               current_uptime)
        LOG.info("{} has been rebooted".format(machine.name))

    show_step(6)
    cluster.check.check_machines_status()
    cluster.check.check_actual_expected_pods()
    cluster.check.check_cluster_readiness()


def shutdown(kaas_manager: Manager, show_step, machine_type='control'):
    """
    Reboot each cluster node based on machine type
    Args:
        kaas_manager: Manager object
        show_step: show step fixture
        machine_type: machine type - control or worker

    Returns: None

    """
    namespace = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = namespace.get_cluster(settings.TARGET_CLUSTER)
    machines = cluster.get_machines(machine_type=machine_type)

    LOG.info("{} nodes will be rebooted:\n{}".format(machine_type, '\n'.join([x.name for x in machines])))

    for machine in machines:
        show_step(1)
        LOG.info(f"Target {machine_type} machine {machine.name}")

        show_step(2)
        machine.power_off()

        show_step(3)
        LOG.info("Wait until powerState become: poweredOn")
        waiters.wait(lambda: machine.get_power_status(), interval=1, timeout=600)

        show_step(4)
        cluster.check.check_k8s_nodes()
        cluster.check.check_cluster_nodes()

        cluster.check.check_k8s_pods()
        cluster.check.check_actual_expected_pods()

    show_step(5)
    cluster.check.check_machines_status()
    cluster.check.check_actual_expected_pods()
    cluster.check.check_cluster_readiness()


def test_ha_child_reboot_control_nodes(kaas_manager: Manager, show_step):
    """Reboot every control cluster node
    Precondition - all expected pods and their replicas must be presented
    The following scenario is executed for every node
    Scenario:
        1. Get vSphere child node
        2. Collect uptime for machines
        3. Reboot node
        4. Wait till docker services and pods are Running and Ready
        5. Check if machine has been rebooted
        6. Check cluster readiness
    Expected: result - every node is rebooted and all pods on every node are
    Running and Ready after this operation.
    """
    reboot(kaas_manager, show_step, 'control')


def test_ha_child_reboot_worker_nodes(kaas_manager: Manager, show_step):
    """Reboot every worker cluster node
    Precondition - all expected pods and their replicas must be presented
    The following scenario is executed for every node
    Scenario:
        1. Get vSphere child node
        2. Collect uptime for machines
        3. Reboot node
        4. Wait till docker services and pods are Running and Ready
        5. Check if machine has been rebooted
        6. Check cluster readiness
    Expected: result - every node is rebooted and all pods on every node are
    Running and Ready after this operation.
    """
    reboot(kaas_manager, show_step, 'worker')


def test_ha_child_shutdown_control_node(kaas_manager: Manager, show_step):
    """Shutdown every control cluster node
    Precondition - all expected pods and their replicas must be presented
    The following scenario is executed for every node
    Scenario:
        1. Get vSphere child nodes
        2. Power off node
        3. Wait until provider power on node
        4. Check that all pods are Running and Ready
        5. Check cluster readiness
    Expected: result - every node is rebooted and all pods on every node are
    Running and Ready after this operation.
    """
    shutdown(kaas_manager, show_step, 'control')


def test_ha_child_shutdown_worker_node(kaas_manager: Manager, show_step):
    """Shutdown every control cluster node
    Precondition - all expected pods and their replicas must be presented
    The following scenario is executed for every node
    Scenario:
        1. Get vSphere child nodes
        2. Power off node
        3. Wait until provider power on node
        4. Check that all pods are Running and Ready
        5. Check cluster readiness
    Expected: result - every node is rebooted and all pods on every node are
    Running and Ready after this operation.
    """
    shutdown(kaas_manager, show_step, 'worker')
