import pytest

from si_tests import logger
from si_tests import settings
from si_tests.managers.kaas_manager import Manager
from si_tests.utils import utils

from si_tests.utils import packaging_version as version

LOG = logger.logger


class TestUpdateGroup(object):
    def test_create_remove(self, kaas_manager: Manager, show_step):
        """Create and remove update group

         Scenario:
             1. Check init state cluster
             2. Check default group automatically assigned to all workers
             3. Check control group automatically assigned to all controls(only after 2-28)
             4. Create new update group
             5. Assign it to the  worker machines
             6. Assign it to controls
             7. Remove custom group - catch admission error
             8. Remove label with custom  group from machine
             9. Remove custom group
             10. Check default group appears on worker machines
         """
        cluster_name = settings.TARGET_CLUSTER
        namespace = settings.TARGET_NAMESPACE
        ns = kaas_manager.get_namespace(namespace)
        cluster = ns.get_cluster(cluster_name)

        show_step(1)
        LOG.info(
            f"Check init state on the {cluster._cluster_type} cluster {cluster.namespace}/{cluster.name}")
        cluster.check.check_machines_status()
        cluster.check.check_cluster_readiness()
        cluster.check.check_k8s_nodes()

        show_step(2)
        LOG.info("Check default update exists")
        d_group = cluster.get_default_update_group()

        # assert that default group exists
        assert d_group, 'Default update group is missing'
        LOG.info(
            f"Check default update group {d_group.name} labels on workers "
            f"from cluster {cluster.namespace}/{cluster.name}")
        cluster.check.check_machine_update_group_labels()

        mgmt_cluster = cluster._manager.get_mgmt_cluster()
        kaas_version = version.parse(mgmt_cluster.get_kaasrelease_version())
        if kaas_version >= version.parse("kaas-2-28-0-rc"):
            show_step(3)
            # Default "control" update groups will be only in 2-28
            LOG.info("Check control update group exists")
            control_group = cluster.get_control_update_group()
            assert control_group, 'Control update group is missing'
            LOG.info(
                f"Check control update group  labels on controls from cluster {cluster.namespace}/{cluster.name}")
            cluster.check.check_machine_update_group_labels(
                group_name='controlplane', machine_type='control')

        show_step(4)
        rnd_string = utils.gen_random_string(4)
        new_group_name = f"test-create-group-{rnd_string}"
        cluster.create_update_group(name=new_group_name, namespace=namespace, cluster_name=cluster_name)
        created_group = cluster.get_update_group_by_name(name=new_group_name)
        assert created_group, 'Created custom update group is missing'

        show_step(5)
        LOG.info("Associate new update groups to workers machines")
        label = {"kaas.mirantis.com/update-group": created_group.name}
        workers = cluster.get_machines(machine_type='worker')
        for w in workers:
            try:
                w.add_machine_labels(labels=label)
            except Exception as e:
                LOG.info(e)

        show_step(6)
        LOG.info("Associate new update groups to controls machines - should got an error")
        controls = cluster.get_machines(machine_type='control')
        for c in controls:
            try:
                c.add_machine_labels(labels=label)
            except Exception as e:
                LOG.info(e)
                if 'denied the request' not in str(e):
                    pytest.fail("It is possible to assign custom group "
                                "to control, when it should be denided ")

        show_step(7)
        try:
            cluster.delete_update_group(new_group_name)
        except Exception as e:
            LOG.info(e)
            if 'is still in use by machines' not in str(e):
                pytest.fail("It is possible to delete update group when "
                            "it is using by some machine, that is wrong")

        show_step(8)
        for w in workers:
            try:
                w.remove_machine_labels(labels=(list(label.keys())))
            except Exception as e:
                LOG.info(e)

        show_step(9)
        cluster.delete_update_group(new_group_name)

        show_step(10)

        d_group = cluster.get_default_update_group()

        # assert that default group exists
        assert d_group, 'Default update group is missing'
        LOG.info(
            f"Check default update group {d_group.name} labels on workers  after "
            f"deletion of custom group"
            f"from cluster {cluster.namespace}/{cluster.name}")
        cluster.check.check_machine_update_group_labels()
