import os
import uuid
import untangle
import paramiko
import pytest

from si_tests import settings
from si_tests.deployments.utils import (
    commons,
    kubectl_utils,
    file_utils,
    namespace,
)

from si_tests.utils import utils
from si_tests.utils import packaging_version as version


def test_nova_compute_instance_sriov_simple(openstack_client_manager, request):
    kubectl = kubectl_utils.Kubectl()
    ns = settings.OSH_NAMESPACE
    osdpl_name = settings.OSH_DEPLOYMENT_NAME
    command = '{} -o yaml'.format(osdpl_name)
    osdpl = kubectl.get('osdpl', command, ns).result_yaml
    if osdpl['spec']['features']['neutron'].get('backend') == 'tungstenfabric':
        template = file_utils.join(
            os.path.dirname(os.path.abspath(__file__)), "templates/sriov_tf.yaml"
        )
    else:
        template = file_utils.join(
            os.path.dirname(os.path.abspath(__file__)), "templates/sriov.yaml"
        )

    stack_name = f"sriov-{uuid.uuid4()}"

    extra_parameters = ({'server_availability_zone': settings.OPENSTACK_DPDK_AVAILABILITY_ZONE_NAME})

    openstack_client_manager.create_stack(request, stack_name, template, stack_params=extra_parameters, finalizer=False)

    server = openstack_client_manager.server.show([stack_name])
    hypervisor_hostname = server["OS-EXT-SRV-ATTR:hypervisor_hostname"].split(
        '.')[0]
    instance_name = server["OS-EXT-SRV-ATTR:instance_name"]
    domain_obj = untangle.parse(
        openstack_client_manager.virsh_dumpXML(
            hypervisor_hostname, instance_name
        )
    )
    for interface in domain_obj.domain.devices.interface:
        interface_type = interface.get_attribute("type")
        assert interface_type == 'hostdev'

    commons.LOG.info(f"Delete heat stack {stack_name}")
    openstack_client_manager.stack.delete([stack_name, "-y", "--wait"])


def test_nova_compute_instance_sriov_advanced(openstack_client_manager, os_manager, request):
    kubectl = kubectl_utils.Kubectl()
    ns = settings.OSH_NAMESPACE
    osdpl_name = settings.OSH_DEPLOYMENT_NAME
    command = '{} -o yaml'.format(osdpl_name)
    osdpl = kubectl.get('osdpl', command, ns).result_yaml

    keystone_client_name = (
        os_manager.api.pods.list_starts_with(
            "keystone-client", namespace.NAMESPACE.openstack)[0].read().metadata.name
    )

    commons.LOG.info('Check count SRIOV configured compute nodes')
    count_sriov_configured_cmps = 0
    sriov_nodes = [n for n in osdpl['spec']['nodes'].values()
                   if n.get('features', {}).get('neutron', {}).get('sriov', {}).get('enabled', False)]
    if len(sriov_nodes) > 1:
        commons.LOG.info(f"Multiple computes with configured SRIOV found. Count: {len(sriov_nodes)}")
        count_sriov_configured_cmps = len(sriov_nodes)
    else:
        message = "Not enough compute nodes with SRIOV. Skipping test"
        commons.LOG.warning(message)
        pytest.skip(message)

    commons.LOG.info("Check target availability zone")
    count_advanced_cmps = 0
    try:
        advanced_cmp_list = kubectl.exec(
            keystone_client_name,
            command="bash -c 'PYTHONWARNINGS=ignore::UserWarning "
                    "openstack aggregate show advanced-compute-aggregate -f yaml -c hosts'",
            namespace=namespace.NAMESPACE.openstack
        ).result_yaml
        count_advanced_cmps = len(advanced_cmp_list['hosts'])
    except AssertionError:
        commons.LOG.info("Aggregation is not exist, will be used az nova")
        count_advanced_cmps = 0

    if count_sriov_configured_cmps == count_advanced_cmps:
        availability_zone = settings.OPENSTACK_DPDK_AVAILABILITY_ZONE_NAME
    else:
        availability_zone = 'nova'
    commons.LOG.info(f"Availability zone: {availability_zone}")

    template_name = "sriov_multi.yaml"

    if os_manager.is_tf_enabled:
        template_name = "sriov_tf_multi.yaml"
    # Drop after 26.2 is released
    elif os_manager.is_ovn_enabled and version.parse(os_manager.os_controller_version()) < version.parse("1.2.0"):
        template_name = "sriov_ovn_multi.yaml"

    template = file_utils.join(
            os.path.dirname(os.path.abspath(__file__)), "templates", template_name
        )

    stack_name = f"sriov-{uuid.uuid4()}"

    extra_parameters = ({'server_availability_zone': availability_zone})

    openstack_client_manager.create_stack(request, stack_name, template, stack_params=extra_parameters, finalizer=False)

    server_name = f"server_one-{stack_name}"
    # Drop after 26.2 is released
    if os_manager.is_ovn_enabled and version.parse(os_manager.os_controller_version()) < version.parse("1.2.0"):
        server_name = f"server_two-{stack_name}"

    stack_data = openstack_client_manager.stack.show([stack_name])

    commons.LOG.info("Try to check connectivity between VMs")
    server_two_sriov_ip = None
    server_one_floating = None
    for output in stack_data['outputs']:
        if output['output_key'] == 'server_two_sriov_ip':
            server_two_sriov_ip = output['output_value']
        if output['output_key'] == 'server_one_fip':
            server_one_floating = output['output_value']

    assert server_two_sriov_ip and server_one_floating, \
        ("Unable to get IP address from stack output.\n"
         f"Floating ip of VM2: {server_one_floating}\n"
         f"SRIOV ip of VM1: {server_two_sriov_ip}\n")

    ping_cmd = f"""
for i in $(seq 4); do
   ping -c1 {server_two_sriov_ip}
   if [ $? -ne 0 ]; then
      exit 2
   fi
done
"""
    try:
        ping_resutl = utils.basic_ssh_command(ping_cmd, server_one_floating, 'ubuntu', 'qalab')
    except (paramiko.ssh_exception.SSHException, TimeoutError):
        commons.LOG.error(
            "Unable connect to instance by SSH. Floating address is unreachable."
        )
        raise Exception("Unable connect to instance by SSH. Floating address unreachable.")

    assert ping_resutl.exit_code == 0, \
        "Unable to check ping between first and second VMs"

    server = openstack_client_manager.server.show([server_name])
    hypervisor_hostname = server["OS-EXT-SRV-ATTR:hypervisor_hostname"].split('.')[0]
    instance_name = server["OS-EXT-SRV-ATTR:instance_name"]
    domain_obj = untangle.parse(
        openstack_client_manager.virsh_dumpXML(
            hypervisor_hostname, instance_name
        )
    )
    interface_type = []
    for interface in domain_obj.domain.devices.interface:
        interface_type.append(interface.get_attribute("type"))
    assert 'hostdev' in interface_type

    commons.LOG.info(f"Delete heat stack {stack_name}")
    openstack_client_manager.stack.delete([stack_name, "-y", "--wait"])
