import os

import kubernetes
import pytest

from si_tests import settings
from si_tests import logger
from si_tests.clients.k8s.cluster import K8sCluster
from si_tests.fixtures.kubectl import create_secret
from si_tests.tests.deployment.test_sl_test import get_sl_release, get_sl_test_image, run_sl_test

LOG = logger.logger


@pytest.mark.parametrize('_', ['CLUSTER_NAME={0}'
                               .format(settings.TARGET_CLUSTER)])
def test_sl(_):
    cluster_name = settings.TARGET_CLUSTER
    keycloak_address = ''
    writer_password = ''

    cluster = K8sCluster(kubeconfig=settings.KUBECONFIG_PATH)

    # in non-MCC env 'stacklight' bundle stores chart info
    sl_hb = cluster.kaas_helmbundles.get(name='stacklight', namespace='stacklight')
    sl_release = get_sl_release(sl_hb)
    sl_test_image = settings.STACKLIGHT_TEST_IMAGE or \
        get_sl_test_image(sl_release)
    sl_chart_url = sl_release['chartURL'] if sl_release else ""

    with open(os.path.join(settings.ARTIFACTS_DIR,
                           "stacklight_test_image.txt"), "w") as f:
        f.write(sl_test_image)
    with open(os.path.join(settings.ARTIFACTS_DIR,
                           "stacklight_chart_url.txt"), "w") as f:
        f.write(sl_chart_url)

    test_pod_name = settings.STACKLIGHT_TEST_POD_NAME
    test_pod_ns = settings.STACKLIGHT_TEST_NAMESPACE

    test_pod_resources = {
        'requests': {
            'cpu': '1',
            'memory': '1Gi'
        },
        'limits': {
            'cpu': '4',
            'memory': '4Gi'
        }
    }

    if settings.STACKLIGHT_TEST_POD_RESOURCES:
        test_pod_resources = settings.STACKLIGHT_TEST_POD_RESOURCES

    prepare_sl_test(cluster, test_pod_name, test_pod_ns)

    payload = {
        'KEYCLOAK_URL': keycloak_address,
        'KEYCLOAK_USER': 'writer',
        'KEYCLOAK_PASSWORD': writer_password,
        'KUBECONFIG_SECRET_NAME': test_pod_name,
        'STACKLIGHT_TEST_SA': test_pod_name,
        'STACKLIGHT_TEST_POD_NAME': test_pod_name,
        'STACKLIGHT_TEST_IMAGE': sl_test_image,
        'STACKLIGHT_TEST_POD_RESOURCES': test_pod_resources
    }
    run_sl_test(cluster,
                test_pod_ns, test_pod_name,
                payload, f'artifacts/lma_{cluster_name}_report.xml')


def prepare_sl_test(kubeclient, pod_name, namespace):
    # copied from ComponentTestResources.setUp
    # create Namespace
    if not kubeclient.namespaces.present(name=namespace):
        LOG.info("Creating Namespace '{0}'".format(namespace))
        ns_body = kubernetes.client.V1Namespace(api_version='v1',
                                                kind='Namespace',
                                                metadata={'name': namespace})
        kubeclient.api_core.create_namespace(body=ns_body)
    else:
        LOG.info("Namespace '{0}' already exists".format(namespace))

    # create ServiceAccount
    if not kubeclient.serviceaccounts.present(name=pod_name,
                                              namespace=namespace):
        LOG.info("Creating ServiceAccount '{0}'".format(pod_name))
        sa_body = kubernetes.client.V1ServiceAccount(
            api_version='v1',
            kind='ServiceAccount',
            metadata={'name': pod_name}
        )
        SA = kubeclient.serviceaccounts.create(
            namespace=namespace, body=sa_body)
        LOG.debug(SA)
    else:
        LOG.info("ServiceAccount '{0}' already exists"
                 .format(pod_name))
        SA = kubeclient.serviceaccounts.get(
            name=pod_name, namespace=namespace)

    # create ClusterRole
    api_rbac = kubeclient.api_rbac_auth
    cr_present = any([
        cr for cr in api_rbac.list_cluster_role().items
        if cr.metadata.name == pod_name])
    if not cr_present:
        LOG.info("Creating ClusterRole '{0}'".format(pod_name))
        pr = kubernetes.client.V1PolicyRule(api_groups=['*'],
                                            resources=['*'],
                                            verbs=['*'])
        # skipping api_version='rbac.authorization.k8s.io / v1beta1'
        cr_body = kubernetes.client.V1ClusterRole(
            kind='ClusterRole',
            metadata={'name': pod_name, 'namespace': namespace},
            rules=[pr]
        )
        CR = api_rbac.create_cluster_role(body=cr_body)
        LOG.debug(CR)
    else:
        LOG.info("ClusterRole '{0}' already exists".format(pod_name))

    # create ClusterRoleBinding
    crb_present = any([
        crb for crb in api_rbac.list_cluster_role_binding().items
        if crb.metadata.name == pod_name])

    if not crb_present:
        LOG.info("Creating ClusterRoleBinding '{0}'"
                 .format(pod_name))
        rr = kubernetes.client.V1RoleRef(
            kind="ClusterRole",
            api_group="rbac.authorization.k8s.io",
            name=pod_name)
        sb = kubernetes.client.V1Subject(name=pod_name,
                                         kind="ServiceAccount",
                                         namespace=namespace)
        # skipping api_version='rbac.authorization.k8s.io / v1beta1'
        crb_template = kubernetes.client.V1ClusterRoleBinding(
            metadata={'name': pod_name},
            kind='ClusterRoleBinding',
            role_ref=rr,
            subjects=[sb]
        )
        CRB = api_rbac.create_cluster_role_binding(
            body=crb_template
        )
        LOG.debug(CRB)
    else:
        LOG.info("ClusterRoleBinding '{0}' already exists"
                 .format(pod_name))

    if kubeclient.secrets.present(name=pod_name,
                                  namespace=namespace):
        LOG.warning("Secret '{0}' already exists. Deleting.."
                    .format(pod_name))
        kubeclient.api_core.delete_namespaced_secret(
            pod_name, namespace)

    # create kubeconfig secret
    LOG.info("Creating Secret '{0}'".format(pod_name))
    with open(settings.KUBECONFIG_PATH) as kubeconfig:
        kubeconfig_content = kubeconfig.read()
    secret = create_secret(
        kubeclient, kubeconfig_content, namespace, pod_name)
    LOG.debug(secret)
