import functools
import os
import tempfile
import uuid
import untangle

from si_tests.deployments.utils import (
    commons,
    kubectl_utils,
    file_utils,
    namespace,
    wait_utils,
)

from si_tests import settings


def test_nova_compute_instance_dpdk(openstack_client_manager, request):
    stack_name = f"dpdk-{uuid.uuid4()}"
    target_template = f"/tmp/{stack_name}.yaml"
    target_parameters = f"/tmp/{stack_name}-parameters.yaml"
    source = file_utils.join(
        os.path.dirname(os.path.abspath(__file__)), "templates/dpdk.yaml"
    )
    client_name = openstack_client_manager.client.name
    kubectl = kubectl_utils.Kubectl()
    destination = (
        f"{namespace.NAMESPACE.openstack}/{client_name}:{target_template}"
    )

    parameters_dest = (
        f"{namespace.NAMESPACE.openstack}/{client_name}:{target_parameters}"
    )
    with tempfile.NamedTemporaryFile(mode="w") as tmp:
        az_name = settings.OPENSTACK_DPDK_AVAILABILITY_ZONE_NAME
        content = f"""
        parameters:
          server_availability_zone: {az_name}
        """
        tmp.write(content)
        tmp.flush()
        kubectl.cp(tmp.name,  parameters_dest)

    commons.LOG.info("Copy heat template")
    kubectl.cp(source, destination)

    commons.LOG.info("Create heat stack")
    openstack_client_manager.stack.create(
        [stack_name, "-t", target_template, "-e", target_parameters]
    )

    request.addfinalizer(
        functools.partial(
            openstack_client_manager.stack.delete, [stack_name, "-y", "--wait"]
        )
    )

    commons.LOG.info("Wait for heat stack to create")
    wait = wait_utils.Waiter(openstack_client_manager.os_manager, 3600)

    def _stack():
        status = openstack_client_manager.stack.show([stack_name])[
            "stack_status"
        ]
        if status == "CREATE_FAILED":
            failed_reason = {'resource_name': '', 'message': ''}
            resource_list = openstack_client_manager.stack.resource_list(
                ["-n", "10", stack_name]
            )
            failed_resources = openstack_client_manager.stack.resource_list(
                ["--filter", "status=CREATE_FAILED", stack_name],
                yaml_output=True
            )
            for r in failed_resources:
                resource_events = openstack_client_manager.stack.event_list(
                    ["--resource", r.get('resource_name'), stack_name],
                    yaml_output=True
                )
                failed_event = [
                    event for event in resource_events if event.get(
                        'resource_status') == 'CREATE_FAILED']
                for failed in failed_event:
                    failed_reason['resource_name'] = failed['resource_name']
                    failed_reason['message'] = failed['resource_status_reason']
            commons.LOG.error(
                "Resource info: \n%s",
                resource_list,
            )
            raise Exception("Failed to create stack: {}".format(failed_reason))
        return status == "CREATE_COMPLETE"

    wait.wait(_stack)
    commons.LOG.info("Heat stack created successfully")

    server = openstack_client_manager.server.show([stack_name])
    hypervisor_hostname = server["OS-EXT-SRV-ATTR:hypervisor_hostname"].split(
        '.')[0]
    instance_name = server["OS-EXT-SRV-ATTR:instance_name"]
    domain_obj = untangle.parse(
        openstack_client_manager.virsh_dumpXML(
            hypervisor_hostname, instance_name
        )
    )
    # Check if TungstenFabric enabled
    ns = settings.OSH_NAMESPACE
    osdpl_name = settings.OSH_DEPLOYMENT_NAME
    command = '{} -o yaml'.format(osdpl_name)
    osdpl = kubectl.get('osdpl', command, ns).result_yaml
    backend = osdpl['spec']['features']['neutron'].get('backend')
    tf = True if backend == 'tungstenfabric' else False

    for interface in domain_obj.domain.devices.interface:
        target_dev = interface.target.get_attribute("dev")
        if tf:
            cmd = ['vif', '--list']
            interface_list = openstack_client_manager.exec_vrouter_agent(
                hypervisor_hostname, cmd)
            assert 'PMD: {}'.format(target_dev) in interface_list
        else:
            cmd = ['ovs-vsctl', 'get', 'interface', target_dev, 'type']
            interface_type = openstack_client_manager.exec_neutron_ovs_agent(
                hypervisor_hostname, cmd)
            assert 'dpdk' in interface_type
