import pytest

from si_tests import settings
from si_tests import logger
from si_tests.utils import templates
from si_tests.utils import utils


LOG = logger.logger


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.TARGET_CLUSTER)])
@pytest.mark.usefixtures("store_cluster_description")
def test_add_os_cred_failed(kaas_manager, show_step, _):
    """Add wrong credential for os provider."""
    cluster_labels = kaas_manager.get_mgmt_cluster().metadata['labels']
    LOG.info('Cluster labels {}'.format(cluster_labels))
    if 'openstack' not in cluster_labels['kaas.mirantis.com/provider']:
        msg = ('Current provider is not match test requirement. '
               'Test should run on Openstack provider')
        LOG.warning(msg)
        pytest.skip(msg)

    rnd_string = utils.gen_random_string(6)
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    region = kaas_manager.get_mgmt_cluster().region_name

    # Creating openstack credential
    credential_name = "{rnd_string}-cred".format(rnd_string=rnd_string)
    LOG.info("Credential name - {credential_name}".format(
        credential_name=credential_name))
    clouds_yaml_str = templates.render_template(
        settings.CLOUDS_FAIL_YAML_PATH)
    ns.create_openstack_credential(
            name=credential_name,
            clouds_yaml_str=clouds_yaml_str,
            region=region,
            validate=False)
    # step 002 - Get status(should be invalid)
    try:
        ns.wait_provider_cred_is_valid(
                credential_name=credential_name,
                provider=settings.OPENSTACK_PROVIDER_NAME,
                retries=10,
                interval=60)
    except Exception as e:
        LOG.debug("We should not fail here as we "
                  "wait for false in status valid. Current {} ".format(e))
    cred = ns.get_openstackcredential(name=credential_name).data
    LOG.debug(f"Creds '{credential_name}' data:\n{cred}")
    if cred['status']:
        assert not cred['status']['valid'], (
            "Status should be invalid for "
            "cred with wrong data, but current one "
            "is {}".format(cred['status']['valid']))
    else:
        pytest.fail(
            'There is no expected status. '
            'Current cred is {}'.format(cred))


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.TARGET_CLUSTER)])
@pytest.mark.usefixtures("store_cluster_description")
def test_add_os_cred_fake_region(kaas_manager, show_step, _):
    """Add credential for os provider with invalid region."""

    if utils.kaas_greater_than_2_26_0(kaas_manager):
        LOG.info("Test is invalid for clusters after 2.26 MCC release")
        return

    cluster_labels = kaas_manager.get_mgmt_cluster().metadata['labels']
    LOG.info('Cluster labels {}'.format(cluster_labels))
    if 'openstack' not in cluster_labels['kaas.mirantis.com/provider']:
        msg = ('Current provider is not match test requirement. '
               'Test should run on Openstack provider')
        LOG.warning(msg)
        pytest.skip(msg)

    rnd_string = utils.gen_random_string(6)
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    # Creating openstack credential
    credential_name = "{rnd_string}-cred".format(rnd_string=rnd_string)
    LOG.info("Credential name - {credential_name}".format(
        credential_name=credential_name))
    clouds_yaml_str = templates.render_template(settings.CLOUDS_YAML_PATH)
    try:
        ns.create_openstack_credential(
            name=credential_name,
            clouds_yaml_str=clouds_yaml_str,
            region='fake',
            validate=False)
    except Exception as e:
        assert e.status == 400, ('Wrong response code. {}'.format(e))
    else:
        pytest.fail("It is possible to create cred for non exists region")


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.TARGET_CLUSTER)])
@pytest.mark.usefixtures("store_cluster_description")
def test_add_aws_cred_failed(kaas_manager, show_step, _):
    """Add wrong credential for aws provider."""
    cluster_labels = kaas_manager.get_mgmt_cluster().metadata['labels']
    LOG.info('Cluster labels {}'.format(cluster_labels))
    if 'aws' not in cluster_labels['kaas.mirantis.com/provider']:
        msg = ('Current provider is not match test requirement. '
               'Test should run on AWS provider')
        LOG.warning(msg)
        pytest.skip(msg)

    rnd_string = utils.gen_random_string(6)
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    region = kaas_manager.get_mgmt_cluster().region_name
    # Creating aws credential
    credential_name = "{rnd_string}-cred".format(rnd_string=rnd_string)
    LOG.info("Credential name - {credential_name}".format(
        credential_name=credential_name))
    try:
        ns.create_aws_credential(
            name=credential_name,
            aws_key_id='fake_key',
            aws_secret_access_key='fake',
            region=region,
            validate=False)
    except Exception as e:
        assert e.status == 400, ('Wrong response code. {}'.format(e))
    else:
        pytest.fail("It is possible to create cred with fake access key")


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.TARGET_CLUSTER)])
@pytest.mark.usefixtures("store_cluster_description")
def test_add_aws_cred_non_exists_region(kaas_manager, show_step, _):
    """Add wrong credential for aws provider."""

    if utils.kaas_greater_than_2_26_0(kaas_manager):
        LOG.info("Test is invalid for clusters after 2.26 MCC release")
        return

    cluster_labels = kaas_manager.get_mgmt_cluster().metadata['labels']
    LOG.info('Cluster labels {}'.format(cluster_labels))
    if 'aws' not in cluster_labels['kaas.mirantis.com/provider']:
        msg = ('Current provider is not match test requirement. '
               'Test should run on AWS provider')
        LOG.warning(msg)
        pytest.skip(msg)

    rnd_string = utils.gen_random_string(6)
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    # Get aws bootstrap credentials from management cluster
    aws_key_id, aws_secret_access_key = kaas_manager.get_aws_credential(
        "cloud-config", "default")

    # Creating aws credential
    credential_name = "{rnd_string}-cred".format(rnd_string=rnd_string)
    LOG.info("Credential name - {credential_name}".format(
        credential_name=credential_name))
    try:
        ns.create_aws_credential(
            name=credential_name,
            aws_key_id=aws_key_id,
            aws_secret_access_key=aws_secret_access_key,
            region='fake',
            validate=False)
    except Exception as e:
        assert e.status == 400, ('Wrong response code. {}'.format(e))
    else:
        pytest.fail("It is possible to create cred for non exists region")
