#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import copy
import pytest
import yaml
from dataclasses import dataclass
from typing import Dict

from si_tests import logger
from si_tests import settings
from si_tests.utils import templates, waiters
from si_tests.utils import packaging_version as version
from si_tests.utils import exceptions as timeout_exceptions
from si_tests.managers.netchecker_manager import NetcheckerManager
from kubernetes.client.exceptions import ApiException as k8sApiException

LOG = logger.logger


# Configuration constants
PARAM_MAPPING = {
    'ping_interval_msec': 'interval_msec',
    'ping_timeout_msec': 'timeout_msec',
    'ping_packets_interval_msec': 'packets_interval_msec',
    'file_targets_re_eval_sec': 're_eval_sec',
}

WAIT_STATUS_TIMEOUT = 600
CONFIG_UPDATE_TIMEOUT = 300
CONFIG_UPDATE_INTERVAL = 20


@dataclass
class VersionConstraints:
    """Version constraints for netchecker support"""
    child_min_version: str = 'mosk-21-0-0-rc-25-2'
    management_min_version: str = 'mke-20-0-0-rc-3-7'


@dataclass
class TestConfig:
    """Test configuration container"""
    namespace: str
    cluster_name: str
    netchecker_obj_name: str
    inventory_config_name: str
    targets_config_name: str
    netchecker_label: Dict[str, str]
    netchecker_config_path: str


def get_test_config() -> TestConfig:
    """Create test configuration from settings"""
    return TestConfig(
        namespace=settings.TARGET_NAMESPACE,
        cluster_name=settings.TARGET_CLUSTER,
        netchecker_obj_name=settings.NETCHECKER_OBJECT_NAME,
        inventory_config_name=settings.NETCHECKER_INVENTORY_CONFIG_NAME,
        targets_config_name=settings.NETCHECKER_TARGETS_CONFIG_NAME,
        netchecker_label=settings.NETCHECKER_SI_LABEL,
        netchecker_config_path=settings.NETCHECKER_FILE_PATH
    )


def is_netchecker_supported(cluster) -> bool:
    """Check if netchecker is supported for the cluster version"""
    constraints = VersionConstraints()
    cl_version = cluster.clusterrelease_version

    if cluster.is_child:
        return version.parse(cl_version) >= version.parse(constraints.child_min_version)
    elif cluster.is_management:
        return version.parse(cl_version) >= version.parse(constraints.management_min_version)

    return False


def perform_cluster_health_checks(cluster):
    """Perform all cluster health checks"""
    cluster.check.check_cluster_nodes()
    cluster.check.check_k8s_nodes()
    cluster.check.check_cluster_readiness()


def set_cluster_overrides(target_cluster, data):
    """Set helm overrides for cluster"""
    patch_data = {
        'spec': {
            'providerSpec': {
                'value': {
                    'helmReleases': data
                }
            }
        }
    }
    target_cluster.patch(patch_data)


def create_or_get_infraconnectivitymonitor(netchecker_manager, netchecker_object_data, config: TestConfig):
    """Create InfraConnectivityMonitor or get existing one"""
    try:
        icm = netchecker_manager.create_infraconnectivitymonitor(data=netchecker_object_data)
        return icm, config.netchecker_obj_name
    except k8sApiException as e:
        if e.status in (409, 400):
            icm = netchecker_manager.infraconnectivitymonitors.list(namespace=netchecker_manager.cluster.namespace)[0]

            if e.status == 409:
                LOG.info(f"InfraConnectivityMonitor object already exists for cluster {config.cluster_name} "
                         f"with same name: {config.netchecker_obj_name}. Performing checks for existing object")
            elif e.status == 400:
                LOG.info(f"InfraConnectivityMonitor object already exists with different name: "
                         f"{config.netchecker_obj_name}. Performing checks for this object")

            return icm, icm.name
        else:
            LOG.error("An error during InfraConnectivityMonitor object creation")
            raise


def _validate_existing_request(ncr, config: TestConfig):
    """Validate existing Netchecker object"""
    nc_cluster = ncr.data['spec']['targetCluster']
    nc_namespace = ncr.data['metadata']['namespace']

    errors = []
    if nc_cluster != config.cluster_name:
        errors.append(f"Netchecker object is existed for wrong cluster. "
                      f"Cluster for testing: {config.cluster_name}. "
                      f"Cluster in netchecker specs: {nc_cluster}")

    if nc_namespace != config.namespace:
        errors.append(f"Netchecker object is existed for wrong namespace. "
                      f"Namespace for testing: {config.namespace}. "
                      f"Namespace in netchecker metadata: {nc_namespace}")

    if errors:
        raise AssertionError("\n".join(errors))


def create_probe_options():
    """Create default probe options for testing"""
    return {
        'ping_interval_msec': 5000,
        'ping_packets_interval_msec': 150,
        'ping_timeout_msec': 1000,
        'file_targets_re_eval_sec': 10
    }


def test_create_check_netchecker(kaas_manager, show_step):
    """This test creates and checks InfraConnectivityMonitor resources.

    Scenario:
        1. Check cluster readiness
        2. Label nodes with specific label
        3. Create InfraConnectivityMonitor object
        4. Check parameters for existed netchecker object
        5. Compare targets for each nodes
        6. Compare subnets from inventory config and actual used subnets
        7. Compare ipadresses from inventory config and actual ipadresses from host netplan
        8. Check machine selection is working as expected
        9. Check overrides are working as expected
        10. Check configs management works as expected
        11. Set constant overrides for cloudprober for further checks
        12. Check cluster readiness after InfraConnectivityMonitor is created
        13. Save netchecker objects specs to artifacts
    """
    config = get_test_config()
    ns = kaas_manager.get_namespace(config.namespace)
    cluster = ns.get_cluster(config.cluster_name)

    if not is_netchecker_supported(cluster):
        pytest.skip(f"Netchecker is not supported in cluster version {cluster.clusterrelease_version}")

    show_step(1)
    perform_cluster_health_checks(cluster)

    netchecker_manager = NetcheckerManager(cluster)

    # Prepare request data
    render_options = {
        "TARGET_CLUSTER": config.cluster_name,
        "TARGET_NAMESPACE": config.namespace,
        "NETCHECKER_OBJECT_NAME": config.netchecker_obj_name
    }
    netchecker_object_data = yaml.safe_load(templates.render_template(config.netchecker_config_path,
                                                                      options=render_options))
    netchecker_object_data['spec']['machineSelector'] = {'matchLabels': config.netchecker_label}

    show_step(2)
    for machine in cluster.get_machines():
        machine.add_machine_labels(config.netchecker_label)

    show_step(3)
    ncr, request_object_name = create_or_get_infraconnectivitymonitor(netchecker_manager,
                                                                      netchecker_object_data,
                                                                      config)

    LOG.info("Waiting for InfraConnectivityMonitor statuses")
    netchecker_manager.wait_infraconnectivitymonitor_status(ncr, timeout=300, interval=30)
    LOG.info(f"InfraConnectivityMonitor {request_object_name} is ready")

    show_step(4)
    _validate_existing_request(ncr, config)

    show_step(5)
    netchecker_manager.compare_targets(request_object_name=config.netchecker_obj_name)

    show_step(6)
    netchecker_manager.compare_subnets(inventory_config_name=config.inventory_config_name)

    show_step(7)
    netchecker_manager.compare_ips(inventory_config_name=config.inventory_config_name)

    show_step(8)
    _check_netchecker_machine_selector(netchecker_manager=netchecker_manager,
                                       netchecker_obj_name=config.netchecker_obj_name,
                                       netchecker_label=config.netchecker_label)

    show_step(9)
    _check_cloudprober_config_changed(netchecker_manager=netchecker_manager)

    show_step(10)
    _check_skip_configs_updates_work_as_expected(netchecker_manager=netchecker_manager)

    show_step(11)
    probe_options = create_probe_options()
    netchecker_manager.apply_cnnc_agent_chart_overrides(options=probe_options)
    netchecker_manager.wait_cnnc_agent_chart_values_applied(params_mapping=PARAM_MAPPING, expected_values=probe_options)

    show_step(12)
    perform_cluster_health_checks(cluster)

    show_step(13)
    netchecker_manager.save_configs()


def test_check_netchecker(kaas_manager, show_step):
    """This test compares and checks actual and monitored by netchecker resources.

    Scenario:
        1. Check cluster readiness
        2. Check InfraConnectivityMonitor object is ready
        3. Compare targets for each nodes
        4. Compare subnets from inventory config and actual used subnets
        5. Compare ipadresses from inventory config and actual ipadresses from host netplan
        6. Compare current specs with expected
        7. Check machine selection is working as expected
        8. Compare overrides with expected
        9. Check overrides working as expected
        10. Check configs management works as expected
        11. Check cluster readiness after checks
    """
    config = get_test_config()
    ns = kaas_manager.get_namespace(config.namespace)
    cluster = ns.get_cluster(config.cluster_name)

    if not is_netchecker_supported(cluster):
        pytest.skip(f"Netchecker is not supported in cluster version {cluster.clusterrelease_version}")

    show_step(1)
    perform_cluster_health_checks(cluster)

    netchecker_manager = NetcheckerManager(cluster)

    show_step(2)
    netchecker_obj = netchecker_manager.get_infraconnectivitymonitor(name=config.netchecker_obj_name)
    assert netchecker_obj, (f"Netchecker {config.netchecker_obj_name} object not found for cluster "
                            f"{cluster.name}. Can not perform checks")
    netchecker_manager.wait_infraconnectivitymonitor_status(netchecker_obj, timeout=300, interval=30)

    show_step(3)
    netchecker_manager.compare_targets(request_object_name=config.netchecker_obj_name)

    show_step(4)
    netchecker_manager.compare_subnets(inventory_config_name=config.inventory_config_name)

    show_step(5)
    netchecker_manager.compare_ips(inventory_config_name=config.inventory_config_name)

    show_step(6)
    _compare_infraconnectivitymonitor_specs(netchecker_manager=netchecker_manager,
                                            netchecker_obj_name=config.netchecker_obj_name)

    show_step(7)
    _check_netchecker_machine_selector(netchecker_manager=netchecker_manager,
                                       netchecker_obj_name=config.netchecker_obj_name,
                                       netchecker_label=config.netchecker_label)

    if cluster.is_child:
        cloudprober_specs_file = settings.NETCHECKER_CHILD_CLOUDPROBER_OVERRIDES_FILE
    else:
        cloudprober_specs_file = settings.NETCHECKER_MGMT_CLOUDPROBER_OVERRIDES_FILE

    if cloudprober_specs_file:
        show_step(8)
        with open(cloudprober_specs_file, 'r') as expected_overrides:
            expected_data = yaml.safe_load(expected_overrides)
        _check_overrides_as_expected(netchecker_manager=netchecker_manager, expected_overrides=expected_data)

    show_step(9)
    _check_cloudprober_config_changed(netchecker_manager=netchecker_manager)

    show_step(10)
    _check_skip_configs_updates_work_as_expected(netchecker_manager=netchecker_manager)

    show_step(11)
    perform_cluster_health_checks(cluster)


def _compare_infraconnectivitymonitor_specs(netchecker_manager, netchecker_obj_name):
    """
    This check compares netchecker object specs with expected.
    """
    if netchecker_manager.cluster.is_child:
        specs_artifact = settings.NETCHECKER_CHILD_OBJECTS_SPECS_FILE
    else:
        specs_artifact = settings.NETCHECKER_MGMT_OBJECTS_SPECS_FILE

    if not specs_artifact:
        LOG.info("Expected specs artifact was not provided. Skipping check")
        return

    with open(specs_artifact, 'r') as data:
        specs = yaml.safe_load(data)

    request_specs_from_artifact = specs.get('infraconnectivitymonitor_specs', {})
    LOG.info(f"Expected request specs from given artifact {specs_artifact}:\n{yaml.dump(request_specs_from_artifact)}")

    netchecker_obj = netchecker_manager.get_infraconnectivitymonitor(name=netchecker_obj_name)
    current_request_specs = netchecker_obj.data.get('spec', {})

    LOG.info(f"Current request specs:\n{yaml.dump(current_request_specs)}")
    err_msg = (f"Request specs were changed.\nSpecs from given artifact:\n{yaml.dump(request_specs_from_artifact)}.\n"
               f"Current specs:\n{yaml.dump(current_request_specs)}")

    assert request_specs_from_artifact == current_request_specs, err_msg


def _check_netchecker_machine_selector(netchecker_manager, netchecker_obj_name, netchecker_label):
    """
    Check that machineselector in netchecker object works as expected
    :param netchecker_manager: netchecker manager object
    """

    wait_status_timeout = 600
    machines = netchecker_manager.cluster.get_machines()
    netchecker_obj = netchecker_manager.get_infraconnectivitymonitor(name=netchecker_obj_name)
    netchecker_obj.patch({'spec': {'machineSelector': {'matchLabels': netchecker_label}}})
    label_name, = netchecker_label
    if netchecker_manager.cluster.is_child:
        # Will check for 1 controller and 1 worker
        machines_for_test = [[m for m in machines if m.machine_type == 'worker'][0],
                             [m for m in machines if m.machine_type == 'control'][0]]
    else:
        # Will take two nodes for check
        machines_for_test = machines[:2]
    LOG.info(f"Next machines are chosen for test: {[m.name for m in machines_for_test]}")
    # Cleanup netchecker labels from all machines
    LOG.info("Cleanup si-netchecker-label from all machines")
    initial_machines_with_labels = netchecker_manager.cluster.get_machines_by_label(label_name)
    for machine in initial_machines_with_labels:
        machine.add_machine_labels({label_name: None})

    LOG.info("Set labels only for chosen machines")
    for machine in machines_for_test:
        machine.add_machine_labels(netchecker_label)

    def _inventory_nodes_cnt(n_request):
        return len(n_request.data.get('status', {}).get('inventoryConfigStatus', {}).get('nodes', []))

    def _targets_nodes_cnt(n_request):
        return len(n_request.data.get('status', {}).get('targetsConfigStatus', {}).get('nodes', []))

    try:
        LOG.info("Waiting for all objects contain correct nodes count")
        waiters.wait(lambda: _inventory_nodes_cnt(netchecker_obj) == _targets_nodes_cnt(netchecker_obj) ==
                     len(machines_for_test), timeout=wait_status_timeout, interval=10)
    except timeout_exceptions.TimeoutError:
        msg = (f"Timeout waiting after {wait_status_timeout} sec for corrent nodes count. Nodes count in inventory: "
               f"{_inventory_nodes_cnt(netchecker_obj)}, nodes count in targets: "
               f"{_targets_nodes_cnt(netchecker_obj)}, but must be equal {len(machines_for_test)}")
        raise TimeoutError(msg)
    finally:
        # Turn labels back to origin
        LOG.banner("Bring machines labels back to initial state")
        for machine in machines_for_test:
            machine.add_machine_labels({label_name: None})

        if initial_machines_with_labels:
            for machine in initial_machines_with_labels:
                machine.add_machine_labels(netchecker_label)

        # Remove machine selector from request at the end of check to handle all machines for further checks
        netchecker_obj.patch({'spec': {'machineSelector': {'matchLabels': None}}})
        waiters.wait(lambda: _inventory_nodes_cnt(netchecker_obj) == _targets_nodes_cnt(netchecker_obj) ==
                     len(machines), timeout=wait_status_timeout, interval=10)


def _check_cloudprober_config_changed(netchecker_manager):
    """
    Check that cloudprober overrides in cnnc-agent chart works as expected
    """
    cluster = netchecker_manager.cluster
    configmap_param_list = PARAM_MAPPING.values()

    LOG.info("Get existed options from cnnc-agent configmap")
    existed_options = netchecker_manager.get_values_from_cnnc_configmap_raw_data(configmap_param_list)
    LOG.info(f"Current options in cnnc-configmap:\n{yaml.dump(existed_options)}")

    LOG.info("Check if overrides for cnnc-agent already existed in cluster")
    existed_helm_charts_overrides = cluster.data['spec']['providerSpec']['value'].get('helmReleases', [])
    initial_overrides_state = copy.deepcopy(existed_helm_charts_overrides)

    cnnc_agent_chart_values = netchecker_manager.get_netchecker_cloudprober_overrides().get(
        'values', {}).get('probe_options', {})

    if cnnc_agent_chart_values:
        _test_existing_overrides(netchecker_manager, cnnc_agent_chart_values, existed_options,
                                 existed_helm_charts_overrides, initial_overrides_state)
    else:
        _test_new_overrides(netchecker_manager, existed_options, initial_overrides_state)

    LOG.info("All overrides work as expected")


def _test_existing_overrides(netchecker_manager, chart_values, existed_options, helm_overrides, initial_state):
    """Test existing cnnc chart overrides"""
    cluster = netchecker_manager.cluster
    wrong_values = {}

    LOG.info(f"Next overrides for cnnc-agent chart exist in cluster:\n{yaml.dump(chart_values)}")
    LOG.info("Checking configmap for correct values")

    for param, expected_value in chart_values.items():
        configmap_param = PARAM_MAPPING.get(param)
        if configmap_param:
            actual_value = existed_options.get(configmap_param, 0)
            if actual_value != expected_value:
                wrong_values[param] = {
                    'expected_value': expected_value,
                    'actual_value': actual_value
                }

    assert not wrong_values, (f"Values from overrides are not equal to actual values in "
                              f"configmap:\n{yaml.dump(wrong_values)}")

    LOG.info("Will change values and check they are applied")
    new_data = {k: v + 10 for k, v in chart_values.items()}

    for rel in helm_overrides:
        if rel.get('name', '') == 'cnnc-agent':
            rel['values']['probe_options'] = new_data
            break

    set_cluster_overrides(target_cluster=cluster, data=helm_overrides)
    netchecker_manager.wait_cnnc_agent_chart_values_applied(params_mapping=PARAM_MAPPING, expected_values=new_data)

    LOG.info("All parameters are applied")
    LOG.info("Turn overrides back to initial state")
    set_cluster_overrides(target_cluster=cluster, data=initial_state)
    netchecker_manager.wait_cnnc_agent_chart_values_applied(params_mapping=PARAM_MAPPING, expected_values=chart_values)


def _test_new_overrides(netchecker_manager, existed_options, initial_state):
    """Test new cnnc chart overrides"""
    cluster = netchecker_manager.cluster

    LOG.info("No overrides exist in cluster object. Will set overrides for cnnc-agent chart")

    initial_params = {
        'ping_interval_msec': existed_options[PARAM_MAPPING['ping_interval_msec']],
        'ping_timeout_msec': existed_options[PARAM_MAPPING['ping_timeout_msec']],
        'ping_packets_interval_msec': existed_options[PARAM_MAPPING['ping_packets_interval_msec']],
        'file_targets_re_eval_sec': existed_options[PARAM_MAPPING['file_targets_re_eval_sec']]
    }

    values_to_set = {k: v + 10 for k, v in initial_params.items()}

    cnnc_chart_overrides = {
        'name': 'cnnc-agent',
        'values': {
            'probe_options': values_to_set
        }
    }

    helm_charts = cluster.data['spec']['providerSpec']['value'].get('helmReleases', [])
    helm_charts.append(cnnc_chart_overrides)
    set_cluster_overrides(target_cluster=cluster, data=helm_charts)

    LOG.info("Waiting for values applied in configmap")
    netchecker_manager.wait_cnnc_agent_chart_values_applied(params_mapping=PARAM_MAPPING, expected_values=values_to_set)

    LOG.info("Turn overrides back to initial state")
    set_cluster_overrides(target_cluster=cluster, data=initial_state)
    netchecker_manager.wait_cnnc_agent_chart_values_applied(params_mapping=PARAM_MAPPING,
                                                            expected_values=initial_params)


def _check_overrides_as_expected(netchecker_manager, expected_overrides):
    """
    This check compares overrides for cnnc-agent chart with expected values
    """
    current_overrides = netchecker_manager.get_netchecker_cloudprober_overrides()
    assert current_overrides == expected_overrides, (f"Some parameters/values are changed. "
                                                     f"Expected:\n{yaml.dump(expected_overrides)}\n"
                                                     f"Actual:\n{yaml.dump(current_overrides)}")
    LOG.info("Overrides for cnnc-agent chart as expected")


def _check_skip_configs_updates_work_as_expected(netchecker_manager):
    """
    This check validates correct behaviour when annotation kaas.mirantis.com/skip-netchecker-config-updates
    is set in netchecker object. When this annotation is set, then it must be possible to
    change spec in targets and inventory configs and this data should not be overwritten by controller
    """
    config = get_test_config()
    netchecker_obj = netchecker_manager.get_infraconnectivitymonitor(name=config.netchecker_obj_name)
    cluster = netchecker_manager.cluster
    skip_annotation_name = 'kaas.mirantis.com/skip-netchecker-config-updates'

    # Check if already exists
    existed_annotations = netchecker_obj.data.get('metadata', {}).get('annotations', {}) or {}
    skip_netchecker_config_updates = bool(existed_annotations.get(skip_annotation_name, False))
    already_exists = skip_netchecker_config_updates

    if not already_exists:
        LOG.info(f"Add {skip_annotation_name} annotation to infraconnectivitymonitor request")
        netchecker_obj.patch({'metadata': {'annotations': {skip_annotation_name: 'true'}}})
    else:
        LOG.info(f"InfraConnectivityMonitor object has already configured with annotation "
                 f"{skip_annotation_name}")

    expected_message = "InfraConnectivityMonitor configuration updates and status propagation are skipped"

    def check_status_message(message):
        LOG.info("Check for correct message in inventoryConfigStatus and targetsConfigStatus")
        netchecker_obj_data = netchecker_manager.get_infraconnectivitymonitor(name=config.netchecker_obj_name).data

        inventory_config_message = netchecker_obj_data.get(
            'status', {}).get('inventoryConfigStatus', {}).get('statusMsg', '')
        targets_config_message = netchecker_obj_data.get(
            'status', {}).get('targetsConfigStatus', {}).get('statusMsg', '')

        LOG.info(f"Inventory status message:\n{inventory_config_message}.\nTargets status "
                 f"message:\n{targets_config_message}")

        return message in inventory_config_message and message in targets_config_message

    waiters.wait(lambda: check_status_message(message=expected_message), timeout=300, interval=10)

    LOG.info("Patch inventory and targets configs with test data and check it is not overwritten")
    inventory_config = netchecker_manager.get_inventory_config(name=config.inventory_config_name)
    targets_config = netchecker_manager.get_targets_config(name=config.targets_config_name)

    # Store initial states
    initial_inventory_state = copy.deepcopy(inventory_config.data.get('spec', {}).get('nodesConfig', []))
    initial_targets_state = copy.deepcopy(targets_config.data.get('spec', {}).get('nodesConfig', []))

    # Create test data
    inventory_test_data = {
        'expectedSubnetTags': [f'{cluster.namespace}/si-test-subnet'],
        'nodeName': 'si-test-node'
    }
    targets_test_data = {
        'targets': [
            {
                'nodeName': 'si-test-target-node',
                'subnetTags': [f'{cluster.namespace}/si-test-subnet']
            }
        ],
        'nodeName': 'si-test-node'
    }

    # Modify configs
    updated_inventory_config = initial_inventory_state + [inventory_test_data]
    updated_targets_config = initial_targets_state + [targets_test_data]
    inventory_config.patch({'spec': {'nodesConfig': updated_inventory_config}})
    targets_config.patch({'spec': {'nodesConfig': updated_targets_config}})

    LOG.info("Will wait a few minutes to be sure data is not overwritten")

    def check_data_consistent(netchecker, expected_inventory_data, expected_targets_data):
        current_inventory_data = netchecker.get_inventory_config(
            name=config.inventory_config_name).data.get('spec', {}).get('nodesConfig', [])
        current_targets_data = netchecker.get_targets_config(
            name=config.targets_config_name).data.get('spec', {}).get('nodesConfig', [])

        return (current_inventory_data == expected_inventory_data and
                current_targets_data == expected_targets_data)

    try:
        waiters.wait(lambda: not check_data_consistent(
            netchecker=netchecker_manager,
            expected_inventory_data=updated_inventory_config,
            expected_targets_data=updated_targets_config), timeout=180, interval=30)
        raise AssertionError("Data in netchecker configs were changed, however it should not due to "
                             "kaas.mirantis.com/skip-netchecker-config-updates annotation is set in netchecker object")
    except timeout_exceptions.TimeoutError:
        LOG.info("Data was not overwritten as expected")

    LOG.info("Bring objects back to initial state")
    if not already_exists:
        LOG.info("Remove annotation from netchecker object")
        netchecker_obj.patch({'metadata': {'annotations': {skip_annotation_name: None}}})
        LOG.info("waiting for data returns to original")
        waiters.wait(lambda: check_data_consistent(
            netchecker=netchecker_manager,
            expected_inventory_data=initial_inventory_state,
            expected_targets_data=initial_targets_state), timeout=180, interval=30)
    else:
        LOG.info("Remove test data from netchecker inventory and targets")
        inventory_config.patch({'spec': {'nodesConfig': initial_inventory_state}})
        targets_config.patch({'spec': {'nodesConfig': initial_targets_state}})

    LOG.info("Configs management works as expected")
