import pytest


from si_tests import settings
from si_tests import logger
from si_tests.utils import utils
from si_tests.utils.utils import Provider

LOG = logger.logger


@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('log_method_time')
@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.KAAS_CHILD_CLUSTER_NAME)])
@pytest.mark.usefixtures("store_cluster_description")
@pytest.mark.usefixtures('introspect_child_deploy_objects')
def test_kaas_create_regional_equinix_child(kaas_manager, show_step, _):
    """Deploy child cluster on top of region cluster.

    Scenario:
        1. Collecting env data
        2. Creating namespace
        3. Creating public key
        4. Creating equinixmetal credential
        5. Create child cluster and machines with SL label
        6. Check cluster readiness
        7. Pod checks (optional)
        8. Check offline isolation (optional)

    """

    # Collecting env data
    show_step(1)
    cluster_name = settings.KAAS_CHILD_CLUSTER_NAME
    namespace_name = settings.KAAS_NAMESPACE
    release_name = settings.KAAS_CHILD_CLUSTER_RELEASE_NAME
    LOG.info("Available releases: {}".format(
        kaas_manager.get_clusterrelease_names()))
    assert release_name in kaas_manager.get_clusterrelease_names()
    child_dns = settings.KAAS_CHILD_CLUSTER_DNS
    slave_nodes_count = int(settings.KAAS_CHILD_CLUSTER_SLAVE_NODES_COUNT)
    master_nodes_count = int(settings.KAAS_CHILD_CLUSTER_MASTER_NODES_COUNT)
    provider = Provider.equinixmetal
    rnd_string = utils.gen_random_string(4)
    region = kaas_manager.get_mgmt_cluster().region_name

    # Creating namespace
    show_step(2)
    LOG.info("Namespace name - %s", namespace_name)
    ns = kaas_manager.get_or_create_namespace(namespace_name)

    # Creating public key
    show_step(3)
    public_key_name = "{cluster_name}-{rnd_string}-key".format(
        cluster_name=cluster_name, rnd_string=rnd_string)
    LOG.info("Public key name - {public_key_name}".format(
        public_key_name=public_key_name))
    with open(settings.KAAS_CHILD_CLUSTER_PUBLIC_KEY_FILE) as pub_key:
        pub_key_content = pub_key.read()
    ns.create_publickey(public_key_name, pub_key_content)

    # Creating Equinix Metal credentials
    show_step(4)
    credential_name = "{cluster_name}-{rnd_string}-cred".format(
        cluster_name=cluster_name, rnd_string=rnd_string)
    LOG.info("Credential name - {credential_name}".format(
        credential_name=credential_name))

    api_token = settings.KAAS_EQUINIX_API_TOKEN
    project_id = settings.KAAS_EQUINIX_PROJECT_ID
    ns.create_equinix_credential(
        credential_name,
        api_token,
        project_id,
        region=region)

    show_step(5)
    cluster = ns.create_cluster(
        cluster_name,
        release_name,
        credentials_name=credential_name,
        provider=provider.provider_name,
        region=region,
        dns=child_dns,
        lma_enabled=settings.KAAS_CHILD_CLUSTER_DEPLOY_LMA,
        public_key_name=public_key_name,
        secure_overlay=settings.KAAS_CHILD_CLUSTER_SECURE_OVERLAY_ENABLED)

    facility = cluster.spec['providerSpec']['value'].get('facility')
    assert facility, (
        f"Cluster {cluster.namespace}/{cluster.name} don't have 'facility' key in the providerSpec.value")

    for _ in range(master_nodes_count):
        cluster.create_equinix_machine(
            node_type="master",
            os=settings.KAAS_EQUINIX_OS,
            billing_cycle=settings.KAAS_EQUINIX_BILLING_CYCLE,
            machine_type=settings.KAAS_EQUINIX_CHILD_MACHINE_TYPE_MASTER,
            region=region,
            facility=facility)

    labels = \
        [{'key': k, 'value': v} for k, v in
         cluster.get_allowed_node_labels().items()]

    stacklight_nodes_count = \
        3 if slave_nodes_count >= 3 else slave_nodes_count
    for _ in range(stacklight_nodes_count):
        cluster.create_equinix_machine(
            node_type="node",
            os=settings.KAAS_EQUINIX_OS,
            billing_cycle=settings.KAAS_EQUINIX_BILLING_CYCLE,
            machine_type=settings.KAAS_EQUINIX_CHILD_MACHINE_TYPE_STACKLIGHT,
            region=region,
            facility=facility,
            node_labels=labels)

    slave_nodes_count -= stacklight_nodes_count

    for _ in range(slave_nodes_count):
        cluster.create_equinix_machine(
            node_type="node",
            os=settings.KAAS_EQUINIX_OS,
            billing_cycle=settings.KAAS_EQUINIX_BILLING_CYCLE,
            machine_type=settings.KAAS_EQUINIX_CHILD_MACHINE_TYPE_WORKER,
            region=region,
            facility=facility)

    # Waiting for machines are Ready and helmbundles are deployed
    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()

    machines = [
        x for x in cluster.get_machines(machine_type="worker")
        if 'nodeLabels' in x.spec['providerSpec']['value'].keys() and
        labels == x.nodeLabels
    ]
    assert len(machines) == stacklight_nodes_count,\
        "Not all machines have expected nodeLabels"

    for machine in machines:
        machine.check_k8s_nodes_has_labels()

    show_step(6)
    cluster.check.check_cluster_readiness()
    cluster.check.check_deploy_stage_success()
    cluster.check.check_helmbundles()

    # Collecting artifacts
    cluster.store_k8s_artifacts()

    cluster.check.check_k8s_nodes()

    show_step(7)
    if settings.RUN_POD_CHECKS:
        # Check/wait for correct docker service replicas in cluster
        cluster.check.check_actual_expected_docker_services()
        cluster.check.check_k8s_pods()
        cluster.check.check_actual_expected_pods(timeout=3200)

        # Due deployment of ceph takes long time we need wait when all pods
        # are created and running
        try:
            health_info = cluster.check.get_ceph_health_detail()
            assert health_info['status'] == "HEALTH_OK", f'Health is not OK. Will not proceed. ' \
                                                         f'Current ceph health status: {health_info}'
        except AssertionError:
            cluster.check.wait_ceph_health_status('HEALTH_OK')
        cluster.check.check_ceph_pvc()
    cluster.check.check_overlay_encryption_functionality()

    cluster.provider_resources.save_artifact()

    show_step(8)
    if settings.KAAS_OFFLINE_DEPLOYMENT:
        cluster.check.check_offline_isolation(
            settings.KAAS_EXTERNAL_PROXY_ACCESS_STR)
