import pytest


from si_tests import settings
from si_tests import logger
from si_tests.utils import utils

LOG = logger.logger


@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('log_method_time')
@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.KAAS_CHILD_CLUSTER_NAME)])
@pytest.mark.usefixtures("store_cluster_description")
@pytest.mark.usefixtures('introspect_child_deploy_objects')
def test_kaas_create_azure_child(kaas_manager, show_step, _):
    """Deploy child cluster on top of management cluster.

    Scenario:
        1. Collecting env data
        2. Creating namespace
        3. Creating public key
        4. Creating azure credential
        5. Create child cluster and machines with SL label
        6. Check cluster readiness
        7. Pod checks (optional)

    """
    # Collecting env data
    show_step(1)
    cluster_name = settings.KAAS_CHILD_CLUSTER_NAME
    namespace_name = settings.KAAS_NAMESPACE
    release_name = settings.KAAS_CHILD_CLUSTER_RELEASE_NAME
    LOG.info("Available releases: {}".format(
        kaas_manager.get_clusterrelease_names()))
    assert release_name in kaas_manager.get_clusterrelease_names()
    child_dns = settings.KAAS_CHILD_CLUSTER_DNS
    slave_nodes_count = int(settings.KAAS_CHILD_CLUSTER_SLAVE_NODES_COUNT)
    master_nodes_count = int(settings.KAAS_CHILD_CLUSTER_MASTER_NODES_COUNT)

    rnd_string = utils.gen_random_string(4)
    custom_indexes = settings.KAAS_CHILD_CLUSTER_SET_CUSTOM_INDEXES
    indexes = utils.generate_upgrade_indexes()

    region = kaas_manager.get_mgmt_cluster().region_name

    # Creating namespace
    show_step(2)
    LOG.info("Namespace name - %s", namespace_name)
    ns = kaas_manager.create_namespace(namespace_name)

    # Creating public key
    show_step(3)
    public_key_name = "{cluster_name}-{rnd_string}-key".format(
        cluster_name=cluster_name, rnd_string=rnd_string)
    LOG.info("Public key name - {public_key_name}".format(
        public_key_name=public_key_name))
    with open(settings.KAAS_CHILD_CLUSTER_PUBLIC_KEY_FILE) as pub_key:
        pub_key_content = pub_key.read()
    ns.create_publickey(public_key_name, pub_key_content)

    # Creating Azure credentials
    show_step(4)
    credential_name = "{cluster_name}-{rnd_string}-cred".format(
        cluster_name=cluster_name, rnd_string=rnd_string)
    LOG.info("Credential name - {credential_name}".format(
        credential_name=credential_name))

    ns.create_azure_credential(
        name=credential_name,
        subscription_id=settings.KAAS_AZURE_SUBSCRIPTION_ID,
        tenant_id=settings.KAAS_AZURE_TENANT_ID,
        client_id=settings.KAAS_AZURE_CLIENT_ID,
        client_secret=settings.KAAS_AZURE_CLIENT_SECRET,
        region=region)

    show_step(5)
    cluster = ns.create_cluster(
        cluster_name,
        release_name,
        credentials_name=credential_name,
        provider=settings.AZURE_PROVIDER_NAME,
        region=region,
        dns=child_dns,
        lma_enabled=settings.KAAS_CHILD_CLUSTER_DEPLOY_LMA,
        public_key_name=public_key_name,
        secure_overlay=settings.KAAS_CHILD_CLUSTER_SECURE_OVERLAY_ENABLED)

    if custom_indexes:
        LOG.info("Custom indexes is enabled. Random index will be set during machine creation")
    for node in range(master_nodes_count):
        machine_index = indexes.__next__() if custom_indexes else None
        cluster.create_azure_machine(
            node_type="master",
            os=settings.KAAS_AZURE_OS_TYPE,
            vm_size=settings.KAAS_AZURE_VM_SIZE_CHILD,
            vm_disk_size=settings.KAAS_AZURE_VM_DISK_SIZE,
            region=region,
            upgrade_index=machine_index)

    labels = \
        [{'key': k, 'value': v} for k, v in
         cluster.get_allowed_node_labels().items()]

    stacklight_nodes_count = \
        3 if slave_nodes_count >= 3 else slave_nodes_count
    for node in range(stacklight_nodes_count):
        machine_index = indexes.__next__() if custom_indexes else None
        cluster.create_azure_machine(
            node_type="node",
            os=settings.KAAS_AZURE_OS_TYPE,
            vm_size=settings.KAAS_AZURE_VM_SIZE_CHILD,
            vm_disk_size=settings.KAAS_AZURE_VM_DISK_SIZE,
            region=region,
            node_labels=labels,
            upgrade_index=machine_index)

    slave_nodes_count -= stacklight_nodes_count

    for node in range(slave_nodes_count):
        machine_index = indexes.__next__() if custom_indexes else None
        cluster.create_azure_machine(
            node_type="node",
            os=settings.KAAS_AZURE_OS_TYPE,
            vm_size=settings.KAAS_AZURE_VM_SIZE_CHILD,
            vm_disk_size=settings.KAAS_AZURE_VM_DISK_SIZE,
            region=region,
            upgrade_index=machine_index)

    # Waiting for machines are Ready and helmbundles are deployed
    cluster.check.check_machines_status()
    cluster.check.check_cluster_nodes()

    machines = [
        x for x in cluster.get_machines(machine_type="worker")
        if 'nodeLabels' in x.spec['providerSpec']['value'].keys() and
        labels == x.nodeLabels
    ]
    assert len(machines) == stacklight_nodes_count,\
        "Not all machines have expected nodeLabels"

    for machine in machines:
        machine.check_k8s_nodes_has_labels()

    show_step(6)
    cluster.check.check_helmbundles()
    cluster.check.check_cluster_readiness()
    cluster.check.check_deploy_stage_success()
    # Collecting artifacts
    cluster.store_k8s_artifacts()

    cluster.check.check_k8s_nodes()

    show_step(7)
    if settings.RUN_POD_CHECKS:
        # Check/wait for correct docker service replicas in cluster
        cluster.check.check_actual_expected_docker_services()
        cluster.check.check_k8s_pods()
        cluster.check.check_actual_expected_pods(timeout=3200)

    cluster.provider_resources.save_artifact()
