#    Copyright 2022 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import os
import pytest

from si_tests import settings
from si_tests import logger
from si_tests.managers import bootstrap_manager
from si_tests.managers.kaas_manager import Manager
from si_tests.utils import waiters


LOG = logger.logger


class TestProvisionBootstrapv2MgmtCluster():

    @pytest.mark.usefixtures("introspect_standalone_seed_management_deploy_objects")
    def test_bootstrapv2_em2_mgmt(self, show_step):
        """Deploy Bootstrap V2 Management EquinixMetalV2 cluster

        Scenario:
            1. Check if a provider is eligible to start
            2. Generate Keycloak user credentials and store them
               in si-config.yaml before rendering templates
            3. Prepare templates to apply on bootstrap cluster
            4. Check bootstrap templates for management cluster
            5. Start bootstrap V2 management cluster deployment
            6. Create default MCC users, if requested
            7. Check versions
            8. Store machine artifacts
            9. Wait for all pods to be Running and Ready
            10. Check that all expected pods exist
            11. Download bootstrap artifacts
            12. Update SI_CONFIG with SI_SCENARIO and network configuration
            13. Check LDAP integration if enabled and required
            14. Erase created KaaS environment
        """

        # step 001: Check if a provider is eligible to start
        show_step(1)
        if not settings.KAAS_EQUINIXMETALV2_ENABLED:
            pytest.fail("Current scenario supports only EquinixMetalV2 provider")

        seed_ip = settings.SEED_STANDALONE_EXTERNAL_IP
        assert waiters.icmp_ping(settings.SEED_STANDALONE_EXTERNAL_IP)

        # step 002: Generate Keycloak user credentials and store them in si-config.yaml before rendering templates
        show_step(2)
        bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)
        bootstrap.add_credentials_data_to_si_config()

        # step 003: Prepare templates to apply on bootstrap cluster
        show_step(3)
        bootstrap.step_003_prepare_seed_node_templates()
        bootstrap.step_003_patch_bootstrapv2_release_refs()
        management_network_config, region_network_configs, child_network_configs = \
            bootstrap.equinixmetalv2_configure_networking()

        # In bootstrap v2 the proxy object is pre-created in the bootstrap,
        # so we need to set that proxy ref in cluster object
        bootstrap.equinixmetalv2_patch_templates(settings.CLUSTER_NAMESPACE,
                                                 settings.CLUSTER_NAME,
                                                 management_network_config,
                                                 settings.KAAS_MANAGEMENT_CLUSTER_NODES,
                                                 'templates/equinixmetalv2')

        # step 004: Check bootstrap templates for management cluster
        if settings.SKIP_BOOTSTRAP_TEMPLATES_CHECK:
            LOG.warning("Skip bootstrap templates check due "
                        "SKIP_BOOTSTRAP_TEMPLATES_CHECK flag is set")
            if not settings.KAAS_CUSTOM_LMA_LOGGING_ENABLED:
                bootstrap.disable_lma_logging()
            if settings.CORE_KEYCLOAK_LDAP_ENABLED:
                bootstrap.enable_keycloak_ldap()
        else:
            show_step(4)
            bootstrap.check_templates()

        # step 005: Start bootstrap V2 management cluster deployment
        show_step(5)
        bootstrap.step_004_deploy_kaas_cluster(extra_config={"network_config": management_network_config})

        # step 006: Create default MCC users, if requested
        if settings.CREATE_DEFAULT_MCC_USERS_VIA_CLI:
            show_step(6)
            bootstrap.step_004_create_default_mcc_users()

        # step 007: Check versions
        show_step(7)
        bootstrap.step_005_check_versions()

        show_step(8)
        kubeconfig_path = f"{settings.ARTIFACTS_DIR}/management_kubeconfig"
        if not os.path.isfile(kubeconfig_path):
            raise Exception(f"KUBECONFIG {kubeconfig_path} not found in"
                            f" artifacts!")
        kaas_manager = Manager(kubeconfig=kubeconfig_path)
        # store private Ips for equinix mgmt nodes
        mgmt_cluster = kaas_manager.get_mgmt_cluster()
        mgmt_cluster.store_machines_artifacts(public_ip=False)

        # step 009: Wait for all pods to be Running and Ready
        show_step(9)
        bootstrap.step_006_wait_for_pods()

        # step 010: Check that all expected pods exist
        show_step(10)
        bootstrap.step_006_postdeployment_checks()

        # step 010: Download bootstrap artifacts
        show_step(11)
        bootstrap.step_007_download_bootstrap_artifacts()

        show_step(12)
        kaas_manager.si_config.create_si_scenario()
        region = kaas_manager.get_mgmt_cluster().region_name
        kaas_manager.si_config.create_equinixmetalv2_network_config(region, management_network_config,
                                                                    region_network_configs, child_network_configs)

        if settings.KEYCLOAK_LDAP_ENABLED:
            show_step(13)
            bootstrap.check_keystone_ldap_integration()

        show_step(14)
        # step 014: Erase created KaaS environment
        if settings.KEEP_ENV_AFTER:
            LOG.warning("Skip erase due KEEP_ENV_AFTER flag is set")
        else:
            # Erase kaas environment after bootstrap is passed
            bootstrap.step_007_erase_env_after()
